#!/usr/bin/env python

"""
Small library to help generating SVG files
"""

import cgi
import sys

class AllowInherit(type):
    def __init__(cls, name, bases, dict):
        super(AllowInherit, cls).__init__(name, bases, dict)
        for cl in bases:
            if 'allowed' in cl.__dict__:
                cls.allowed.update(cl.allowed)
            

class SVG(object):
    __metaclass__ = AllowInherit
    allowed = {'class':1, 'style':1}
    prepend = ''
    
    def __init__(self, *content, **attributes):
        self.content = list(content)
        self.attributes = attributes
    def name(self):
        return self.__class__.__name__
    def tag(self):
        s = self.__class__.prepend + '<' + self.name()
        for name, value in self.attributes.iteritems():
            name = name.replace('__',':').replace('_','-')
            if name in self.__class__.allowed:
                s += ' %s="%s"\n'%( name, str(value).replace('"','\\"'))
            else:
                sys.stderr.write("%s:%s\n"% (self.name(), name))
        content = self.content_svg()
        if content:
            s += '>' + content + '</' + self.name()
        else:
            s += '/'
        return s + '>\n'
    def content_svg(self):
        s = []
        for content in self.content:
            if issubclass(content.__class__, SVG):
                s.append(content.tag())
            if isinstance(content, str):
                return cgi.escape(content)
        return '\n'.join(s)

class svg(SVG):
    allowed = {'xmlns':1,
               'xmlns:cc': 1,
               'xmlns:xlink': 1,
               'xmlns:dc': 1,
               'xmlns:rdf': 1,
               'xmlns:svg': 1,
               }  
    prepend = '''<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 20010904//EN" "http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd">
'''
class text(SVG):
    allowed = {'x':1, 'y':1}
class g(SVG):
    allowed = {'transform':1}
    def append(self, t):
        self.content.append(t)
class rect(SVG):
    allowed = {'x':1, 'y':1, 'width':1, 'height':1}
class line(SVG):
    allowed = {'x1':1, 'y1':1, 'x2':1, 'y2':1}
class path(SVG):
    allowed = {'d':1}
    def moveto(self, p):
        self.attributes['d'] = "M%g %g" % p
    def lineto(self, p):
        self.attributes['d'] += " L%g %g" % p
    def end(self):
        self.attributes['d'] += " Z"


def path3d(points, **attributes):
    p = path(**attributes)
    p.moveto(points[0].projection())
    for point in points:
        p.lineto(point.projection())
    p.lineto(points[0].projection())
    p.end()
    return p


import math

class Point:
    def __init__(self, position=(0,0,0), goal=None):
        self.position = self.initial_position = position
        self.goal = goal
        self.forces = None
    def projection(self):
        p = self.position
        return (p[0]+p[1]/2, p[2]+p[1]/2)
    def distance(self, p):
        p = p.position
        return math.sqrt(float(p[0]**2 + p[1]**2 + p[2]**2))
    def init_forces(self):
        self.forces = (0,0,0)
    def update_forces(self, p, length):
        d = self.distance(p)
        dd = d - length
        f = [ dd * ( pp - position ) / d
              for pp, position in zip(p.position, self.position) ]
        self.forces = [ force + df for force,df in zip(self.forces, f) ]
        p.forces = [ force - df for force,df in zip(p.forces, f) ]
    def move(self):
        self.position = [ p + f/8 for p,f in zip(self.position, self.forces) ]
        self.forces = None

class Python:
    def __init__(self, length=20, width=40, height=60):
        self.width = width
        self.height = height
        self.diagonal = math.sqrt(width**2 + height**2)
        self.points = [ Point(position=(200 + width * (i % 2),
                                        0,
                                        -height*int(i/2) + 600))
                        for i in xrange(length) ]
        self.points[10].goal = (0,100,0)

    def svg(self, style=''):
        a = g()
        for i in range(0,len(self.points)-2,2):
            a.append(path3d((self.points[i],
                             self.points[i+1],
                             self.points[i+3],
                             self.points[i+2],
                             ),
                             style=style))
        for p in self.points:
            if p.forces == None:
                continue
            pp = p.projection()
            vp = Point(p.forces).projection()
            a.append( line( x1=pp[0], y1=pp[1],
			    x2=pp[0]+vp[0], y2=pp[1]+vp[1] ) )
        return a

    def animate(self):
        a = g(style="stroke:black;fill:#8F8;fill-opacity:0.25")
        a.append( self.svg() )
        for i in range(10):
            self.simulate()
            self.move_points()
            a.append( self.svg() )
        return a
    
    def simulate(self):
        # Compute forces
        for p in self.points:
            p.init_forces()
        for i in range(0,len(self.points)-2,2):
            self.points[i].update_forces(self.points[i+1], self.width)
            self.points[i].update_forces(self.points[i+2], self.height)
            self.points[i].update_forces(self.points[i+3], self.diagonal)
            self.points[i+1].update_forces(self.points[i+3], self.height)
            self.points[i+1].update_forces(self.points[i+2], self.diagonal)
        self.points[i+2].update_forces(self.points[i+3], self.width)

    def move_points(self):
        for p in self.points:
            p.move()
    




if __name__ == '__main__':
    t1 = text('toto', style='fill:rgb(255,0,0)')
    t2 = text('titi', style="font-family:times", y="10")
    r = rect(x="20", y="20", width="10", height="20",
             style="border-width:10; stroke: rgb(0,255,0)")
    g1 = g(t1, t2, r, transform='translate(200,100) scale(10,10)')
    
    print svg(g1, Python().animate(),
        xmlns__dc="http://purl.org/dc/elements/1.1/",
        xmlns__cc="http://web.resource.org/cc/",
        xmlns__rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#",
        xmlns__svg="http://www.w3.org/2000/svg",
        xmlns="http://www.w3.org/2000/svg",
        xmlns__xlink="http://www.w3.org/1999/xlink").tag()
    



