#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2006 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr

import questions
import colorsys
import utilities
import os
import student
import time
import configuration
import cgi
import types

_stats = None

def select(value, mean, coef):
    i = -2
    for c in coef:
        if value < c*mean :
            break
        i += 1
    return i

class Stats:
    def __init__(self):
        self.last_time = 0
        self.ttl = 1
        
    def update(self):
        """Compute statistics about questions and students"""

        t = time.time()
        if t - self.last_time < self.ttl:
            return
        self.last_time = t
        
        self.all_students = student.all_students()

        for question in questions.questions.values():
            question.student_time_searching = 0
            question.student_time_after = 0
            question.student_given = 0
            question.student_view = 0
            question.student_bad = 0
            question.student_good = 0
            question.student_indice = 0
            question.student_indices = [0] * len(question.indices)
            question.student_nr_comment = 0

        self.nr_good_answers = 0
        self.nr_bad_answers = 0
        self.nr_bad_answers_normed = 0
        self.nr_given_indices = 0
        self.nr_given_indices_normed = 0
        self.nr_given_comment = 0
        self.time_after = 0
        self.time_after_normed = 0

        normed_count = 0
        for s in self.all_students:
            s.the_number_of_good_answers = s.number_of_good_answers()
            s.the_number_of_given_questions = s.number_of_given_questions()
            s.the_number_of_bad_answers = s.number_of_bad_answers()
            s.the_number_of_given_indices = s.number_of_given_indices()
            s.the_number_of_comment = s.number_of_comment()
            s.the_time_searching = s.time_searching()
            s.the_time_after = s.time_after()
            s.the_time_first = s.time_first()
            s.the_time_last = s.time_last()
            s.the_time_variance = s.time_variance()

            if s.the_number_of_good_answers:
                normed_count += 1
                s.the_number_of_bad_answers_normed = s.the_number_of_bad_answers / float(s.the_number_of_good_answers)
                s.the_number_of_given_indices_normed = s.the_number_of_given_indices / float(s.the_number_of_good_answers)
                s.the_time_after_normed = s.the_time_after / float(s.the_number_of_good_answers)
            else:
                s.the_number_of_bad_answers_normed = None
                s.the_number_of_given_indices_normed = None
                s.the_time_after_normed = None

            self.nr_good_answers += s.the_number_of_good_answers
            self.nr_bad_answers += s.the_number_of_bad_answers
            self.nr_given_indices += s.the_number_of_given_indices
            self.nr_given_comment += s.the_number_of_comment
            self.time_after += s.the_time_after

            if s.the_number_of_good_answers:
                self.nr_bad_answers_normed += s.the_number_of_bad_answers_normed
                self.nr_given_indices_normed += s.the_number_of_given_indices_normed
                self.time_after_normed += s.the_time_after_normed
                
            
            for answer in s.answers.values():
                try:
                    q = questions.questions[answer.question]
                except KeyError:
                    continue
                if answer.nr_asked == 0:
                    continue
                q.student_given += 1
                q.student_view += answer.nr_asked
                q.student_bad += answer.nr_bad_answer
                q.student_indice += answer.indice + 1
                if answer.indice >= 0:
                    try:
                        for i in xrange(answer.indice+1):                    
                            q.student_indices[i] += 1
                    except IndexError:
                        import sys
                        sys.stderr.write("""Problem indice overflow
question: %s
student: %s
""" % (q, s.filename))
                q.student_time_searching += answer.time_searching
                q.student_time_after += answer.time_after
                q.student_good += answer.answered != False
                q.student_nr_comment += len(answer.comments)

        for q in questions.questions.values():
            q.student_time = q.student_time_searching + q.student_time_after

        # Search correct response time within a time window
        t = []
        for s in self.all_students:
            s.nr_answer_same_time = {}
            for answer in s.answers.values():
                if answer.answered:
                    t.append( (answer.question, answer.last_time, s) )
        t.sort()
        for i in xrange(len(t)):
            current_time = t[i][1]
            current_question = t[i][0]
            j = i - 1
            while ( j >= 0
                    and current_time - t[j][1] < 60
                    and current_question == t[j][0]
                    ) :
                try:
                    t[i][2].nr_answer_same_time[t[j][2].name] += 1
                except KeyError:
                    t[i][2].nr_answer_same_time[t[j][2].name] = 1
                    
                try:
                    t[j][2].nr_answer_same_time[t[i][2].name] += 1
                except KeyError:
                    t[j][2].nr_answer_same_time[t[i][2].name] = 1
                j -= 1
        for s in self.all_students:
            s.nr_of_same_time = sum( s.nr_answer_same_time.values() )
            if s.the_number_of_given_questions:
                s.nr_of_same_time_normed = (
                    s.nr_of_same_time
                    / float(s.the_number_of_given_questions) )
            else:
                s.nr_of_same_time_normed = 0



        # Compute the list of students by number of good answers
        t = [(s.the_number_of_good_answers, s)  for s in self.all_students]
        t.sort()
        t.reverse()
        self.sorted_students = [ b for a,b in t]

        if normed_count:
            # Compute some means
            self.nr_good_answers         /= float(len(self.all_students))
            self.nr_bad_answers          /= float(len(self.all_students))
            self.nr_given_indices        /= float(len(self.all_students))
            self.nr_given_comment        /= float(len(self.all_students))
            self.time_after              /= float(len(self.all_students))
            self.nr_given_indices_normed /= float(normed_count)
            self.nr_bad_answers_normed   /= float(normed_count)
            self.time_after_normed       /= float(normed_count)
                                               
        # Warning
        # 2 : very good
        # 1 : good
        # 0 : nothing to say
        # -1 : bad
        # -2 : very bad
        for s in self.all_students:
            s.warning_nr_good_answers = 0
            s.warning_nr_bad_answers = 0
            s.warning_nr_given_indices = 0
            s.warning_time_after = 0
    
        if self.nr_good_answers > 10:
            for s in self.all_students:
                if s.the_number_of_good_answers == 0:
                    continue ;

                s.warning_nr_good_answers = select(
                    s.the_number_of_good_answers,
                    self.nr_good_answers,
                    (0.7, 0.8, 1.3, 1.5)
                    )

                s.warning_nr_bad_answers = - select(
                    s.the_number_of_bad_answers_normed,
                    self.nr_bad_answers_normed,
                    (0.5, 0.6, 1.5, 2)
                    )

                s.warning_nr_given_indices = - select(
                    s.the_number_of_given_indices_normed,
                    self.nr_given_indices_normed,
                    (0.5, 0.6, 1.5, 2)
                    )

                s.warning_time_after = - select(
                    s.the_time_after_normed,
                    self.time_after_normed,
                    (0.4, 0.6, 1.6, 1.8)
                    )
        if self.all_students:
            self.max_good_answers  = max([s.the_number_of_good_answers
                                          for s in self.all_students])
            self.max_bad_answers   = max([s.the_number_of_bad_answers
                                          for s in self.all_students])
            self.max_given_indices = max([s.the_number_of_given_indices
                                          for s in self.all_students])
        else:
            self.max_good_answers  = 0
            self.max_bad_answers  = 0
            self.max_given_indices  = 0

        # Compute statistics a percentage of the time
        # If we compute statistics in one second and we want
        # to allocate 1% of time to statistics computing.
        # then we must compute statistics every 100 seconds.
        self.ttl = (101-configuration.statistics_cpu_allocation)*(time.time() - self.last_time)




_stats = Stats()


def histogram_level():
    histogram = [0]*(questions.sorted_questions[-1].level+1)
    for q in questions.sorted_questions:
        histogram[q.level] += 1
    return histogram


def question_stats():
    return _stats

def forget_stats():
    _stats.last_time = 0

def update_stats():
    _stats.update()

def merge(t):
    return [ (a[0], a[1]+b[1], a[2]+b[2], a[3]+b[3], a[4]+b[4])
             for a,b in zip(t[0::2], t[1::2]) ]

def compute_plot():
    stats = statistics.question_stats()

    x_size = len(questions.questions) * 10

    nr_good_answers = [0] * x_size
    nr_bad_answers = [0] * x_size
    nr_indices = [0] * x_size
    nr_time_searching = [0] * x_size
    nr_time_after = [0] * x_size
    for s in stats.all_students:
        nr_good_answers[ s.number_of_good_answers() ] += 1
        nr_bad_answers[ s.number_of_bad_answers() ] += 1
        nr_indices[ s.number_of_given_indices() ] += 1
        # Par interval de 1 minutes
        nr_time_searching[ int(s.time_searching() / 60) ] += 1
        nr_time_after[ int(s.time_after() / 60) ] += 1

    while  nr_good_answers[-1] == 0 and  nr_bad_answers[-1] == 0 \
          and nr_time_searching[-1] == 0:
        nr_good_answers.pop()
        nr_bad_answers.pop()
        nr_time_searching.pop()
        nr_time_after.pop()
        x_size -= 1

    data = zip(range(x_size), nr_good_answers, nr_bad_answers, nr_time_searching, nr_time_after, nr_indices)
    data = merge(data)
    data = merge(data)
    data = merge(data)

    if 0:
        d = []
        for s in data:
            d.append(s)
            d.append( [s[0]+8] + list(s[1:]) )
        data = d


    for i in data:
        for j in i:
            print j,
        print

    try:
        # from pychart import *

        x = axis.X(tic_interval = 16, label="#Questions, #Indices, #Minutes")
        y = axis.Y(tic_interval = 1, label="Nombre d'etudiants")
        ar =  area.T(x_axis=x, x_range=(0,data[-1][0]),
                     y_axis=y, y_range=(0,None),
                     size=(500,500))
        p1 =  bar_plot.T(label="Bonne reponse",
                         data=data,
                         hcol=1,
                         cluster=(0,5),
                         line_style=line_style.T(color=color.green),
                         fill_style=fill_style.Diag(
            line_style=line_style.T(width=1, color=color.green),
            line_interval=6,
            )
                         )
        p2 =  bar_plot.T(label="Mauvaise reponse", data=data,
                          hcol=2, cluster=(1,5),
                          line_style=line_style.T(color=color.red),
                         fill_style=fill_style.Rdiag(
            line_style=line_style.T(width=1, color=color.red),
            line_interval=6,
                         ),
                         )
        p3 =  bar_plot.T(label="Temps de reflexion", data=data,
                         hcol=4,  cluster=(2,5),
                         line_style=line_style.T(color=color.black),
                         fill_style=fill_style.Horiz(
            line_style=line_style.T(width=1, color=color.black),
            line_interval=6),
                         )
        p4 =  bar_plot.T(label="Temps apres", data=data,
                         hcol=5,  cluster=(3,5),
                         line_style=line_style.T(color=color.black),
                         fill_style=fill_style.Horiz(
            line_style=line_style.T(width=1, color=color.black),
            line_interval=6),
                         )
        p5 =  bar_plot.T(label="Indices donnes", data=data,
                          hcol=4, cluster=(4,5),
                          line_style=line_style.T(color=color.blue),
                         fill_style=fill_style.Vert(
            line_style=line_style.T(width=1, color=color.black),
            line_interval=6),
                         )  
        ar.add_plot(p1)
        ar.add_plot(p2)
        ar.add_plot(p3)
        ar.add_plot(p4)
        ar.add_plot(p5)
        the_canvas = canvas.init("xxx_decomptes.pdf")
        ar.draw(the_canvas)


        x = axis.X(label="Temps de reflexion")
        y = axis.Y(label="Nombre d'essais")
        ar =  area.T(x_axis=x, x_range=(1,600), x_coord=log_coord.T(),
                     y_axis=y, y_range=(1,5), y_coord=log_coord.T(),
                     size=(500,500))
        ar.add_plot(line_plot.T(data=((0.5,0.5),)))
        the_canvas = canvas.init("xxx_questions.pdf")
        
        question_stats()
        nr_students = float(len(stats.all_students))

        dat = []
        for q in questions.questions.values():
            norme = float(q.student_given)
            if norme == 0:
                continue
            if q.student_time == 0:
                continue

            dat.append((
                ar.x_pos(q.student_time / norme),
                ar.y_pos(1 + (q.student_good + q.student_bad) / norme),
                q.student_indice/norme,
                q.student_given,
                q.name
                ))
        for x, y, indice, given, name in dat:
            the_canvas.ellipsis(
                line_style.T(color=color.black),
                fill_style.Plain(bgcolor=color.white),
                x, y,
                radius=(given)**0.5,
                )
        for x, y, indice, given, name in dat:
            the_canvas.show(x, y,
                "/05/a%d{}" % int(90*float(indice))
                + name.replace("/", "//")
                )

        ar.draw(the_canvas)


    except ImportError:
        return "'pychart' python module is missing"


def graph():
    f = open("xxx.vcg", "w")
    f.write("""
    graph: { title: "questions"
    """)
    d = {}
    i = 0
    
    nb = float(len(questions.worlds()))
    for w in questions.worlds():
        rvb = colorsys.hls_to_rgb(i/nb, 0.8, 0.99)
        f.write("colorentry %d: %d %d %d\n" % (
            i + 64, rvb[0]*256, rvb[1]*256, rvb[2]*256))
        d[w] = i + 64
        i += 1
    
    for q in questions.questions:
        name = q.translate(utilities.flat)
        color = d[ q.split(":")[0] ]
        f.write("""
        node: { title: "%s" 
		label: "%s"
                color: %d
                }
                """ % (name, name.replace(':',' ').replace(' ', '\\n'), color))

    for q in questions.questions.values():
        name = q.name.translate(utilities.flat)        
        for qq in q.required.names():
            if not questions.questions.has_key(qq):
                print "%s est requis par %s" % (qq, q)
                raise ValueError
            f.write("""edge: { sourcename: "%s" targetname: "%s" }\n""" % (
                qq.translate(utilities.flat), name))

    f.write("}\n")
    f.close()

    os.system("""
    (
    rm xxx.ppm HTML/xxx.ps HTML/xxx.png 2>/dev/null
    xvcg -color -silent -psoutput HTML/xxx.ps xxx.vcg
    xvcg -silent -xdpi 300 -ydpi 300 -ppmoutput xxx.ppm xxx.vcg
    convert -resize 900x900 xxx.ppm HTML/xxx.png
    rm xxx.ppm
    echo 'Image computing done'
    ) &""")

def translate_dot_(name):
    return name.translate(utilities.flat).replace(':',' ').replace('"',' ').replace(' ', '\\n')

def translate_dot(name):
    return '"' + translate_dot_(name) + '"'


def graph_dot():
    stats = question_stats()
    if len(stats.all_students) == 0:
        return

    f = open("HTML/xxx_graphe.dot", "w")
    f.write("""
    digraph "questions" {
    node[shape=none];
    graph[nodesep="0",ranksep="0",charset="Latin1", size="11.50,7.5", orientation="L",mclimit="10",nslimit="10"];
    """)
    nb = float(len(questions.worlds()))
    rvb = {}
    i = 0
    for w in questions.worlds():
        c = colorsys.hls_to_rgb(i/nb, 0.8, 0.99)
        rvb[w] = '"#%02x%02x%02x"' % (c[0]*256, c[1]*256, c[2]*256)
        i += 1
    n = float(len(stats.all_students))
    for q in questions.questions.values():
        name = q.name.translate(utilities.flat)
        color = rvb[ q.name.split(":")[0] ]
        if q.student_given == 0:
            q.student_given = 1
        f.write("""%s [ label=<<TABLE BORDER="0" CELLPADDING="0"><TR><TD COLSPAN="3" BGCOLOR="blue" HEIGHT="%d"></TD></TR><TR><TD BGCOLOR="red" WIDTH="%d"></TD><TD BGCOLOR=%s>%s</TD><TD BGCOLOR="green" WIDTH="%d"></TD></TR><TR><TD COLSPAN="3" BGCOLOR="black" HEIGHT="%d"></TD></TR></TABLE>>] ;\n""" % (
            translate_dot(q.name),
            int(50*q.student_indice/q.student_given),
            int(50*q.student_bad/n),
            color,
            translate_dot_(q.name).replace("\\n", "<BR/>"),
            int(50*q.student_good/n),
            int(q.student_time/q.student_given/10),
            )
                )

    for q in questions.questions.values():
        for qq in q.required.names():
            if not questions.questions.has_key(qq):
                print "%s est requis par %s" % (qq, q)
                raise ValueError
            f.write("%s -> %s ;\n" % (translate_dot(qq), translate_dot(q.name)))

    f.write("}\n")
    f.close()

    os.system("dot -oHTML/xxx_graphe.ps -Tps HTML/xxx_graphe.dot &")

def graph2_dot():
    f = open("HTML/xxx_graphe2.dot", "w")
    f.write("""
    digraph "questions" {
    node[shape=none];
    graph[charset="Latin1", size="11.50,7.5", orientation="L"];
    """)
    stats = question_stats()
    nb = float(len(questions.worlds()))
    n = float(len(stats.all_students))
    for w in questions.worlds():
        f.write("""subgraph cluster_%s {
     label="%s";
     """ % (w, w))
        for q in questions.questions.values():
            if not q.name.startswith(w + ":"):
                continue
            name = q.name.translate(utilities.flat)
            if q.student_given == 0:
                q.student_given = 1
            f.write("""%s [ label=<<TABLE BORDER="0" CELLPADDING="0"><TR><TD COLSPAN="3" BGCOLOR="blue" HEIGHT="%d"></TD></TR><TR><TD BGCOLOR="red" WIDTH="%d"></TD><TD>%s</TD><TD BGCOLOR="green" WIDTH="%d"></TD></TR><TR><TD COLSPAN="3" BGCOLOR="black" HEIGHT="%d"></TD></TR></TABLE>>] ;\n""" % (
                translate_dot(q.name),
                int(50*q.student_indice/q.student_given),
                int(50*q.student_bad/n),
                translate_dot_(q.name).replace("\\n", "<BR/>"),
                int(50*q.student_good/n),
                int(q.student_time/q.student_given/10),
                )
                    )
        f.write("}\n")

    for q in questions.questions.values():
        for qq in q.required.names():
            if not questions.questions.has_key(qq):
                print "%s est requis par %s" % (qq, q)
                raise ValueError
            f.write("%s -> %s ;\n" % (translate_dot(qq), translate_dot(q.name)))

    f.write("}\n")
    f.close()

    os.system("dot -oHTML/xxx_graphe2.ps -Tps HTML/xxx_graphe2.dot &")

##############################################################################

import server

class Svg:
    def __init__(self, text_height, bar_text_height, border_width,
                 width, height, url_base=''):
        self.url_base = url_base

        self.content = '''<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<!-- <?xml-stylesheet href="/%ssvg.css" type="text/css"?> -->
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 20010904//EN" 
  "http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd">
<svg
   xmlns:dc="http://purl.org/dc/elements/1.1/"
   xmlns:cc="http://web.resource.org/cc/"
   xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
   xmlns:svg="http://www.w3.org/2000/svg"
   xmlns="http://www.w3.org/2000/svg"
   xmlns:xlink="http://www.w3.org/1999/xlink"
   width="%d"
   height="%d"
   >
<script xlink:href="/%ssvg.js"/>
<defs>
    <style type="text/css"><![CDATA[
      text.title { font-size:%dpx;}
      text.title_m { font-size:%dpx; text-anchor: middle;}
      text.bar { font-size:%dpx; }
      rect.back { stroke-width: %dpx; }
      %s
    ]]></style>
  </defs>
     ''' % (configuration.prefix, width, height,configuration.prefix,text_height,text_height, bar_text_height, border_width, server.get_file('svg.css').content)

    def end(self):
        return self.content + '''</svg>'''
    def rect(self, x, y, w, h, svg_class="", style=""):
        if svg_class:
            svg_class = 'class="' + svg_class + '" '
        self.content += '''<rect %sx="%g" y="%g" width="%g" height="%g" style="%s" />\n''' % (svg_class,x,y,w,h,style)
    def text(self, x,y, text, svg_class="", style=""):
        if svg_class:
            svg_class = ' class="' + svg_class + '"'
        if style:
            style = ' style="' + style + '"'
        self.content += '''<text%s%s><tspan x="%d" y="%d">%s</tspan></text>\n''' % (svg_class, style, x,y,cgi.escape(text))
    def g_start(self, x, y):
        self.content += """<g onclick=\"mouseclick(this);\" transform=\"translate(%d,%d)\">\n""" % (x, y)
    def g_end(self):
        self.content += """</g>\n"""

def split_text(text):
    middle = len(text)/2
    for i in xrange(middle):
        if text[middle+i] == ' ':
            return [text[0:middle+i], text[middle+i+1:]]
        if text[middle-i] == ' ':
            return [text[0:middle-i], text[middle-i+1:]]
    return [text, '']

class BarPlot:
    def __init__(self, svg, width, height, nr_bars, text_height):
        self.svg = svg
        self.width = width
        self.height = height
        self.nr_bars = nr_bars
        self.bar_height = height / nr_bars
        self.text_height = text_height
        self.half_width = width / 2

    def background(self, border_width):
        self.svg.rect(-border_width/2., -border_width/2.,
                 self.width + border_width, self.height + border_width,
                 svg_class='back', style=self.opacity)

    def bar(self, nr, value, max, svg_class, pixel=False):
        if isinstance(value, types.ListType):
            if not value:
                return
            dy = self.bar_height / float(len(value))
            for i in range(len(value)):
                self.svg.rect(0, self.bar_height * nr + i * dy,
                              self.width * (value[i]/float(max)), dy, 
                              svg_class=svg_class + str(i),
                              style=self.opacity)
            self.svg.text(1,
                          self.bar_height * nr + (self.text_height + self.bar_height)/2.,
                          str(value),
                          style=self.opacity,
                          svg_class='bar s',
                          )
            return

        if pixel:
            n = int(value/max)
            for i in range(n):
                self.svg.rect(0, self.bar_height * nr + i*pixel,
                              self.width, pixel, 
                              svg_class=svg_class,
                              style=self.opacity)
            self.svg.rect(0, self.bar_height * nr + n*pixel,
                          self.width * (value % max) / float(max), pixel, 
                          svg_class=svg_class,
                          style=self.opacity)
            comment = '%g' % value
        else:
            self.svg.rect(0, self.bar_height * nr,
                          self.width * (value/float(max)), self.bar_height, 
                          svg_class=svg_class,
                          style=self.opacity)
            comment = '%g / %g' % (value, max)

        self.svg.text(1,
                      self.bar_height * nr + (self.text_height + self.bar_height)/2.,
                      comment,
                      style=self.opacity,
                      svg_class='bar',
                      )

    def nr_comments(self, nr):
        self.svg.text(self.width, self.height/2,
                      str(nr),
                      style=self.opacity,
                      svg_class='nr_comments e',
                      )
        
    def textes(self, textes, line_decal, svg_class='title_m', align='m',
               opacity=None, url=None):
        if opacity == None:
            opacity = self.opacity
        if align == 'm':
            x = self.half_width
        elif align == 's':
            x = self.width + self.text_height
        else:
            x = -self.text_height

        if url:
            self.svg.content += '<a xlink:href="%s%s">' % (
                self.svg.url_base, url.replace('&','&amp;'))
        
        for i in range(len(textes)):
            if textes[i]:
                self.svg.text(x, line_decal*(i+1),
                              unicode(textes[i], 'latin1').encode('utf8'),
                              style=opacity,
                              svg_class=svg_class
                              )
        if url:
            self.svg.content += '</a>'
        
def plot_svg(url_base):
    width = 55
    height = 24
    text_height = 6
    bar_text_height = text_height / 2
    x_spacing = 3
    y_spacing = 3
    line_spacing = (height - 3*text_height)/4
    line_decal = text_height + line_spacing
    border_width = 1
    x_margin = width * 1.5
    x_margin += 2*width # Name of required and used_by list of questions
    y_margin = height * 1.4

    stats = question_stats()

    h = histogram_level()
    h.sort()
    
    svg = Svg(text_height, bar_text_height, border_width,
              h[-1] * (width + x_spacing) + 2*x_margin,
              (2+questions.sorted_questions[-1].level) * (height + y_spacing) + 2 * y_margin,
              url_base
              )
    barplot = BarPlot(svg, width, height, 4, bar_text_height)

    level = -1
    y = y_margin
    for q in questions.sorted_questions:
        if q.level != level:
            x = x_margin
            y += height + y_spacing
            level = q.level

        opacity = (q.student_given+1.) / (len(stats.all_students)+1)
        barplot.opacity = "opacity:%5.3f;fill-opacity:%5.3f;" % (
            opacity, opacity)

        svg.g_start(x, y)
        barplot.background(border_width)

        if q.student_given:
            barplot.bar(0,q.student_good, q.student_given, svg_class='good')
            barplot.bar(1,q.student_bad, q.student_given, svg_class='bad', pixel=2)
            barplot.bar(2,q.student_indices, q.student_given, svg_class='indice')
            barplot.bar(3,int(q.student_time_searching/q.student_given),300,svg_class='time', pixel=2)

        if q.student_nr_comment:
            barplot.nr_comments(q.student_nr_comment)
            
        t = [q.world] + split_text(q.short_name)
        barplot.textes(t, line_decal, url=q.url())

        t = q.required.names()
        barplot.textes(t, bar_text_height, opacity='fill-opacity:0;', align='e', svg_class='bar e')

        t = q.used_by
        barplot.textes(t, bar_text_height, opacity='fill-opacity:0;', align='s', svg_class='bar s')
        
        x += width + x_spacing
        svg.g_end()

    return svg.end()


def troncate_question(q):
    if True:
        q = q.split('<br>')[0]
        q = q[:min(len(q),90)]
        q = cgi.escape(q)
    else:
        q = cgi.escape(q)
            
    return q

# CSS :  style="overflow: hidden; white-space:nowrap;"
def html_simple():
    all = list(questions.questions)
    all.sort()
    t = ['<table><colgroup><col width="*"><col width="5*"></colgroup><tbody>']
    w = ''
    for q in all:
        q = questions.questions[q]
        if q.world != w:
            t.append( '<tr><th colspan="2">' + q.world + '</th></tr>')
            w = q.world
        t.append('<tr><td>' + q.short_name + '</td><td><p>' +
                 troncate_question(q.question()) + '</td></tr>')
    t.append('</tbody></table>')
           
    return '\n'.join(t)


def question_pixel_map_other(state):
    student = state.student_stat
    if student == None:
        student = state.student
    filename = os.path.join(student.file,'map.gif')

    try:
        f = open(filename,'r')
        image = f.read()
        f.close()
        return 'image/gif', image
    except IOError:
        return 'text/plain', '?'


def question_color(q, state, answerables):
    if (state.question != None and state.question.name == q.name):
        p = '\377\377\0'
    elif state.student.answered_question(q.name):
        p = '\0\377\0'
    elif state.student.bad_answer_question(q.name):
        p = '\377\0\0'
    elif q.name in answerables:
        if state.student.given_question(q.name):
            p = '\0\377\377'
        else:
            p = '\0\0\377'
    else:
        p = '\200\200\200'
    indice = state.student.nr_indices_question(q.name)
    if indice > 3:
        indice = 3
    ppp = ''
    for pp in p:
        if pp == '\0':
            pp = chr( int(indice**0.5  *  80) )
        ppp += pp
    return ppp

def question_pixel_map(state):

    if state.student.acls['question_pixel_map_circle']:
        return question_pixel_map_circle(state)
    
    m = []
    level = None
    answerables = [q.name for q in state.student.answerables()]
    for q in questions.sorted_questions:
        if q.level != level:
            m.append([])
            level = q.level
        m[-1].append(question_color(q, state, answerables))
    
    width = max([len(x) for x in m])
    image = 'P6\n%d %d\n255\n' % (len(m), width)
    for col in range(width):
        for line in range(len(m)):
            try:
                image += m[line][col]
            except IndexError:
                image += '\377\377\377'
                

    filename = os.path.join(state.student.file,'map.gif')
    f = os.popen('ppmtogif >"%s" 2>/dev/null' % filename, 'w')
    f.write(image)
    f.close()
    f = open(filename,'r')
    image = f.read()
    f.close()
    return 'image/gif', image

def question_pixel_map_circle(state):
    answerables = [q.name for q in state.student.answerables()]
    width = max([questions.questions[q].coordinates[0]
                 for q in questions.questions]) + 1
    height = max([questions.questions[q].coordinates[1]
                  for q in questions.questions]) + 1
    picture = []
    for i in range(height):
        picture.append(['\377\377\377']*width)

    for q in questions.questions:
        q = questions.questions[q]
        picture[q.coordinates[1]][q.coordinates[0]] = question_color(q, state, answerables)

    print width, height
    image = 'P6\n%d %d\n255\n' % (width, height)
    for line in picture:
        image += ''.join(line)

    filename = os.path.join(state.student.file,'map.gif')
    f = os.popen('ppmtogif >"%s" 2>/dev/null' % filename, 'w')
    f.write(image)
    f.close()
    f = open(filename,'r')
    image = f.read()
    f.close()
    return 'image/gif', image


def display_no_more_valid_answers():
    import sys
    update_stats()
    stats = question_stats()
    messages = {}
    print
    for s in stats.all_students:
        sys.stdout.write('*')
        sys.stdout.flush()
        for answer in s.answers.values():
            try:
                q = questions.questions[answer.question]
            except KeyError:
                continue
            if not answer.answered:
                continue
            ok, comment = s.check_answer(q,answer.answered,None)
            if not ok:
                if answer.question not in messages:
                    messages[answer.question] = {}
                messages[answer.question][answer.answered] = True
    print
    print "Answers no more accepted:"
    for m in messages:
        print m
        for v in messages[m]:
            print '\t' + v

