#!/usr/bin/env python

#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2006 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr
#

import os
import urllib2
import time
import sre
import sys
import cgi
import traceback

for i in ('http_proxy', 'https_proxy'):
    if os.environ.has_key(i):
        del os.environ[i]

def create_students():
    if os.path.exists('Students'):
        if os.path.exists('Students/regtest'):
            os.system('rm -r Students/regtest')
    else:
        os.mkdir('Students')
    try:
        os.mkdir("Students/regtest")
        os.mkdir("Students/regtest/Logs")
        os.mkdir("Students/regtest/Logs/guest0")
        os.mkdir("Students/regtest/Logs/guest1")
    except OSError:
        pass

    f = open("Students/regtest/Logs/guest0/acl", "w")
    f.write("acl_change\n!question_shuffle\n")
    f.close()
    
    f = open("Students/regtest/Logs/guest1/acl", "w")
    f.write("stat_questions_see\nstat_others_see\nstat_others_csv_see\nstat_histogram_good_see\nacl_change\n!question_shuffle\n")
    f.close()

def create_default_acls():
    f = open("Students/regtest/acls.default", "w")
    f.write("gui_times\nquestion_see\nquestion_answerable\nquestion_indice_see\naction_comment_make\n")
    f.close()
    f = open("Students/regtest/acls.student", "w")
    f.write("\n")
    f.close()

class Table:
    def __init__(self, content):
        self.table = []
        for line in content.split('\n'):
            if line.startswith('<tr><td>'):
                self.table.append( line[8:-10].split('</td><td>') )
        self.nr_columns = len(self.table[0])
        for line in self.table:
            if self.nr_columns != len(line):
                raise ValueError("Table is not a table")
        self.nr_lines = len(self.table)

    def column(self, col):
        t = []
        for line in self.table:
            t.append(line[col])
        return t
        
class Question:
    def __init__(self, name, active=False):
        self.name = name
        self.active = active
        self.html_class = []

    def __str__(self):
        s = "<A"
        if self.active:
            s += " ID=1"
        s += " HREF=\"?action=question_see&question=%s\"" \
             % cgi.urllib.quote(self.name)
        s += " CLASS=\""
        for c in ('current_question', 'question_given',
                  'bad_answer_given', 'indice_given', 'resigned'
                  ):
            if c in self.html_class:
                s += c + " "
        s += "\">" + self.name + "</A>"
        return s

class State:
    def __init__(self):
        self.nr_given_questions = 0
        self.nr_bad_answers = 0
        self.nr_good_answers = 0
        self.nr_indices = 0
        self.nr_urls_in_header = 2
        self.nr_urls = 0
        self.urls = ()
        self.content = ()
        self.reject_content = ()
        self.title = ""
        self.question = None            # Text of the question
        self.questions = {}             # Questions URLs
        self.table = None
        self.display_times = True
        self.no_question_displayed()

    def add_question(self, question):
        if self.questions.has_key(question.name):
            raise ValueError("Question %s yet in list" % question.name)
        self.questions[question.name] = question

    def no_question_displayed(self, h1='Guest0'):
        self.h1 = h1
        self.question = None
        self.bad_answer = None
        self.good_answer = None
        self.answer_form = False
        self.default_answer = None
        
class Page:
    def __init__(self, content, state):
        self.content = content
        self.state = state
        problems = sre.findall("[hH][rR][Ee][Ff]=[^'\"]([^'>\"]*)[^'\"]",
                               content)
        if problems:
            print problems
            raise ValueError("Bad HREF")
    
        self.urls = sre.findall("[hH][rR][Ee][Ff]=['\"]?([^'>\"]*)['\"]",
                                content)

        self.urls = [ urllib2.unquote(u) for u in self.urls ]

        if self.urls:
            self.base = self.urls[0]
        else:
            self.base = None

    def cpu(self):
        s = self.content.split("<p class=\"time\">")
        if len(s) == 1:
            s = 0
        else:
            try:
                s = float(s[1].split('<br>')[2].split('<')[0])
            except:
                print s
        return s


    def check(self):
        state = self.state
        
        if state.nr_urls != None \
               and state.nr_urls + state.nr_urls_in_header != len(self.urls):
            return "The page does not contains %d URL but %d :\n%s" % (
                state.nr_urls, len(self.urls)-state.nr_urls_in_header, self.urls)
        if self.content.find("nr_given_indices tips\">%d"%state.nr_indices) == -1:
            return "Bad number of given indices"
        if self.content.find("nr_bad_answers tips\">%d"%state.nr_bad_answers) == -1:
            return "Bad number of bad answers (waiting:%d)" % (
                state.nr_bad_answers, )
        if self.content.find("nr_good_answers tips\">%d"%state.nr_good_answers)== -1:
            return "Bad number of good answers"
        if self.content.find("<title>%s</title>" % state.title)== -1:
            return "Bad title (%s) (%s)" % (state.title, self.content)
        if state.h1 and self.content.find("<h1>%s</h1>" % state.h1)== -1:
            return "Bad H1 : (%s) (%s)" % (state.h1, self.content)
        if state.question \
               and self.content.find("<TD>%s</TD>" % state.question)== -1:
            return "Bad Question (%s) (%s)" %  (state.question, self.content)
        if state.answer_form \
           == (self.content.find("ID=\"2\"") == -1):
            return "FORM displayed or not ?"
        if state.bad_answer \
           and self.content.find(state.bad_answer) == -1:
            return "Can't find bad answer message"
        if state.good_answer \
           and self.content.find(state.good_answer) == -1:
            return "Can't find good answer message"
        if state.default_answer != None \
           and  self.content.find("VALUE=\"%s\"" % state.default_answer) == -1:
            return "Can't find default value (%s) in %s" % (
                state.default_answer, self.content)
        for q in state.questions.values():
            if self.content.find(str(q)) == -1:
                print "\n" + str(q)
                for line in self.content.split('\n'):
                    if line.find(q.name) != -1:
                        print line[line.find('<A'):]
                return "Can't find question link: " + q.name
        if state.display_times != (
            self.content.find('end_date') != -1
            and self.content.find('time_left') != -1 ):
                return "Problem : no time display"
            
        for url in state.urls:
            if url not in self.urls:
                print self.urls
                return "Can't find URL: %s" % url

        if state.table:
            table = Table(self.content)
            for i in state.table:
                if table.column(i[0]) != i[1]:
                    return 'column: %s, expected: %s, in the file: %s' % (
                        i[0], i[1], table.column(i[0]))

        for c in state.content:
            if self.content.find(c) == -1:
                return "Can't find text in the page: %s\n%s" % (c,self.content)

        for c in state.reject_content:
            if self.content.find(c) != -1:
                return "Rejected text in the page: %s\n%s" % (c,self.content)

        return ("", len(self.content), self.cpu())


class Server:
    def __init__(self, port):

        create_students()
        create_default_acls()

        self.state = State()
        self.port = port
        os.system('''
        main.py regtest create Questions/regtest %d
        main.py regtest nocache percentage-time-for-stat 100 start 2>Students/regtest/logs >&2 &
        ''' % port)
        
        # Wait server start
        while True:
            try:
                urllib2.urlopen("http://localhost:%d/" % self.port).close()
                break
            except:
                print 'Waiting server...'
                time.sleep(0.1)

        self.nr = 0
        self.cpu = 0.
        self.bytes = 0

    def stop(self):
        os.system('''main.py regtest stop''')
        print "Total server 'CPU' time:", self.cpu, "s"
        print "Total kilo bytes from server: %.3fkb" % (self.bytes/1000.)
        print "Page average size: %.3fkb" % ((self.bytes/1000.)/self.nr)


    def display_log(self):
        f = open("Students/regtest/logs","r")
        
        for line in f.readlines():
            if line[0] == ' ':
                print line,
        f.close()

    def get(self, url, base=None):
        if base == None:
            base = "http://localhost:%d/" % self.port
        f = urllib2.urlopen(base + url)
        p = Page(f.read(), self.state)
        if p.base:
            self.base = p.base
        f.close()        
        self.nr += 1

        self.cpu += p.cpu()
        self.bytes += len(p.content)
        
        p.name = "Students/regtest/xxx.%03d" % self.nr
        f = open(p.name, "w")
        f.write(p.content)
        f.write("\n" + url + "\n")
        f.close()

        if '<html>' in p.content:
            os.system('%s %s' % (weblint, p.name))


        return p

    def get_next(self, url):
        return self.get(url, base=self.base)


# Tests  return False is ok.

def check_connection(server):
    server.state.no_question_displayed()
    server.state.title = 'Guest0'
    server.state.nr_urls = 3
    server.state.urls = ['../../css.fr.css']
    server.state.add_question(Question('a:a', active=True))
    server.state.content = ("<P CLASS=\"choose\"></P>",)
    server.state.reject_content = ['Indice B', 'a:b']
    return server.get("guest0").check()

def check_css(server):
    page = server.get("css.css")
    if page.content.find('TABLE.menu') == -1:
        return 'Not the CSS'
    return (None, len(page.content), 0)

def check_figures(server):
    page = server.get("regtest.html")
    if page.content != '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN">':
        return "Can't load figure"
    return (None, len(page.content), 0)

def check_question(server):
    server.state.questions['a:a'].html_class = ['current_question', 'question_given']
    server.state.nr_given_questions += 1
    server.state.h1 = 'a:a'
    server.state.question = 'a'
    server.state.answer_form = True
    server.state.content = []
    return server.get_next("?action=question_see&question=a:a").check()

def check_reconnect(server):
    return server.get("guest0").check()

def check_answer_a0(server):
    server.state.nr_bad_answers += 1
    server.state.questions['a:a'].html_class += ['bad_answer_given']
    server.state.default_answer = "a0"
    return server.get_next("?action=question_answerable&value=a0").check()

def check_answer_a1(server):
    server.state.nr_bad_answers += 1
    server.state.bad_answer = "bad_a1"
    server.state.default_answer = "a1"
    return server.get_next("?action=question_answerable&value=a1").check()

def check_answer_a3(server):
    server.state.nr_bad_answers += 1
    server.state.bad_answer = "bad_a3"
    server.state.default_answer = "a3"
    return server.get_next("?action=question_answerable&value=a3").check()

def check_answer_empty(server):
    server.state.nr_bad_answers += 1
    server.state.bad_answer = None
    server.state.default_answer = ""
    return server.get_next("?action=question_answerable&value=").check()

def check_unauthorized_question(server):
    server.state.no_question_displayed()
    return server.get_next("?action=question_see&question=a%3Ab").check()
    
def check_unauthorized_answer(server):
    return server.get_next("?action=question_answerable&value=a").check()

def check_answer_a(server):
    server.get_next("?action=question_see&question=a%3Aa")
    server.state.nr_urls = 4
    server.state.nr_good_answers += 1
    server.state.h1 = 'a:a'
    del server.state.questions['a:a']
    server.state.add_question(Question('a:b', active=True))
    server.state.add_question(Question('a:c'))
    server.state.reject_content.remove('a:b')
    return server.get_next("?action=question_answerable&value=a").check()

def check_question_b(server):
    server.state.reject_content.append('a:a')
    server.state.answer_form = True
    server.state.nr_urls = 5
    server.state.h1 = "a:b"
    server.state.question = "b"
    server.state.nr_given_questions += 1
    server.state.default_answer = None
    server.state.questions['a:b'].html_class += ['current_question',
                                                 'question_given']
    return server.get_next("?action=question_see&question=a%3Ab").check()

def check_question_b_ask_indice(server):
    server.state.nr_indices += 1
    server.state.content = ['Indice A']
    server.state.questions['a:b'].html_class += ['indice_given']
    return server.get_next("?action=question_indice_see").check()    

def check_question_c(server):
    server.state.content = []    
    server.state.nr_urls = 4
    server.state.nr_given_questions += 1
    server.state.h1 = "a:c"
    server.state.question = "c"
    server.state.questions['a:b'].active = False
    server.state.questions['a:b'].html_class.remove('current_question')
    server.state.questions['a:b'].html_class += ['resigned']
    server.state.questions['a:c'].html_class += ['current_question',
                                                 'question_given']
    server.state.questions['a:c'].active = True
    return server.get_next("?action=question_see&question=a%3Ac").check()

def check_answer_c0(server):
    server.state.nr_bad_answers += 1
    server.state.bad_answer = "bad_c"
    server.state.questions['a:c'].html_class += ['bad_answer_given']
    server.state.default_answer = "c0"
    return server.get_next("?action=question_answerable&value=c0").check()

def check_allow_show_acl(server):
    server.state.urls += ['?action=acl_see']
    server.state.nr_urls += 1
    server.state.default_answer = None
    return server.get_next("?action=acl_change&value=Yacl_see").check()

def check_see_acl(server):
    server.state.nr_urls = None
    server.state.urls += [
                          '?sort_column=-1',
                          ]
    # a(value,"answers_reload");'
    server.state.no_question_displayed()
    return server.get_next("?action=acl_see").check()

def check_question_c_once_more(server):
    server.state.h1 = "a:c"
    server.state.question = "c"
    server.state.bad_answer = "bad_c"
    server.state.answer_form = True
    server.state.urls.remove('?sort_column=-1')
    # server.state.urls.remove('?action=Xanswerable_any')
    server.state.default_answer = "c0"
    return server.get_next("?action=question_see&question=a%3Ac").check()

def check_comment_c(server):
    server.state.no_question_displayed()
    server.state.content = ['<PRE>cComment</PRE>']    
    server.state.default_answer = None
    return server.get_next("?action=action_comment_make&value=cComment").check()
    
def check_see_stat(server):
    server.state.content = []    
    server.state.no_question_displayed()
    server.state.urls += ['?action=stat_see']
    server.state.nr_urls = 17
    server.state.table = [
        [ 1, ['False', 'False', 'True'] ],
        [ 2, ['1', '1', '1'] ],
        [ 3, ['1', '0', '4'] ],
        [ 4, ['0/0', '1/2', '0/0'] ],
        [ 5, ['1', '0', '0'] ],
        ]
    server.state.reject_content.remove('a:a')
    server.get_next("?action=acl_change&value=Ystat_see")
    server.get_next("?action=stat_see")
    return server.get_next("?sort_column=-8").check()

def check_answer_a_again(server):
    server.state.nr_urls = 6
    server.state.h1 = 'a:a'
    server.state.table = None
    server.state.questions['a:b'].active = True
    server.state.questions['a:c'].active = False
    server.state.questions['a:c'].html_class.remove('current_question')
    server.state.questions['a:c'].html_class += ['resigned']
    return server.get_next("?action=question_see&question=a%3Aa").check()

def check_any_question(server):
    server.state.answer_form = True
    server.state.add_question(Question('a:a'))
    server.state.questions['a:a'].html_class += ['question_given',
                                                 'bad_answer_given',
                                                 'current_question',
                                                ]
    server.state.questions['a:a'].active = True
    server.state.questions['a:b'].active = False
    server.state.questions['a:c'].active = False
    server.state.nr_urls = 10
    server.get_next("?action=acl_change&value=Yquestion_answerable_any")
    return server.get_next("?action=acl_change&value=Yquestion_display_any").check()

def check_see_comments(server):
    server.state.no_question_displayed()    
    server.state.nr_urls = 17
    server.state.table = [
        [ 1, ['cComment'] ],
        ]
    server.get_next("?action=acl_change&value=Yanswered_comments_see")
    return server.get_next("?action=answered_comments_see").check()

def check_answer_a_again_again(server):
    server.state.nr_urls = 11 # should be 12
    server.state.table = None
    server.state.h1 = 'a:a'
    server.state.question = 'a'
    server.state.answer_form = True
    server.state.questions['a:a'].html_class += ['current_question']
    server.state.questions['a:a'].active = True
    server.state.questions['a:c'].active = False
    server.state.questions['a:a'].html_class.remove('current_question')
    server.state.default_answer = "a"
    return server.get_next("?action=question_see&question=a%3Aa").check()

def check_see_bad_answers(server):
    server.state.nr_urls = 22 # should be 23
    server.state.table = [
        [ 1, ['<tt>a3</tt>', '<tt>a1</tt>',
              "<span class=\"uncommented\"><tt>a0</tt></span>",
              "<span class=\"uncommented\"><tt></tt></span>",
              ] ],
        ]    
    server.state.urls += ['?student=guest0&action=stat_see_other',
                          '?action=question_see&question=a:a',
                          ]
    server.get_next("?sort_column=-1")
    return server.get_next("?action=acl_change&value=Yanswered_bad_see").check()

def check_comments(server):
    server.state.nr_urls = 16 # should be 17
    server.state.table = [
        [ 1, ['cComment'] ],
        ]
    server.state.urls.remove('?student=guest0&action=stat_see_other')
    server.state.h1 = 'a:c'
    server.state.question = 'c'
    server.state.answer_form = True
    server.state.default_answer = 'c0'
    server.state.bad_answer = 'bad_c'
    server.state.questions['a:a'].html_class.remove('current_question')
    server.state.questions['a:c'].html_class += ['current_question']
    server.get_next("?action=acl_change&value=-answered_bad_see")
    return server.get_next("?action=question_see&question=a:c").check()

def check_required(server):
    server.state.nr_urls = 10
    server.state.table = None
    server.state.content = ['<TD>a</TD>', '<tt>a</tt>']
    server.get_next("?action=acl_change&value=-answered_comments_see")
    return server.get_next("?action=acl_change&value=Yquestion_required_see").check()
    
def check_see_answered(server):
    server.state.nr_urls = 11
    server.state.no_question_displayed()
    server.state.content = ['a:a', '<TD>a</TD>', '<tt>a</tt>',
                            '<tt>a0</tt>', '<tt>a1</tt>', '<tt>a3</tt>',
                            'a:b', '<TD>b</TD>',
                            'a:c', '<TD>c</TD>', '<tt>c0</tt>',
                            '<PRE>cComment</PRE>',
                            '<li>Indice A</li>',
                            ]
    server.get_next("?action=acl_change&value=Yanswered_see")
    return server.get_next("?action=answered_see").check()

def check_stop_server_1(server):
    server.state.nr_urls += 1
    return server.get_next("?action=acl_change&value=Yaction_server_stop").check()

def check_stop_server_2(server):
    server.state.nr_urls -= 1
    server.state.h1 = None
    server.state.display_times = False
    server.state.reject_content.append('<h1>Guest0</h1>')
    return server.get_next("?action=action_server_stop").check()

def check_question_b_again(server):
    server.state.content = []
    return server.get_next("?action=question_see&question=a:b").check()
    
def check_stop_server_3(server):
    server.state.h1 = 'Guest0'
    server.state.reject_content.remove('<h1>Guest0</h1>')
    return server.get_next("?action=acl_change&value=Yquestion_answerable_anytime").check()

def check_question_b_again_again(server):
    server.state.nr_urls += 1 # Indice
    server.state.h1 = 'a:b'
    server.state.question = 'b'
    server.state.answer_form = True
    server.state.content = ['Indice A', '<TD>a</TD>', '<tt>a</tt>']
    server.state.questions['a:b'].html_class += ['current_question']
    server.state.questions['a:c'].html_class.remove('current_question')
    return server.get_next("?action=question_see&question=a:b").check()

def check_see_answered_2(server):
    server.state.nr_urls -= 1
    server.state.no_question_displayed()
    server.state.content = ['a:a', '<TD>a</TD>', '<tt>a</tt>',
                            '<tt>a0</tt>', '<tt>a1</tt>', '<tt>a3</tt>',
                            'a:b', '<TD>b</TD>',
                            'a:c', '<TD>c</TD>', '<tt>c0</tt>',
                            '<PRE>cComment</PRE>',
                            '<li>Indice A</li>',
                            ]
    return server.get_next("?action=answered_see").check()


def check_reset_stat(server):
    server.state.nr_urls += 0 # Because no pointer (its on the stat page)
    server.state.no_question_displayed()
    server.state.nr_indices = 0
    server.state.nr_bad_answers = 0
    server.state.nr_good_answers = 0
    server.state.nr_given_questions = 0
    server.state.content = []
    server.state.questions['a:a'].html_class = []
    server.state.questions['a:b'].html_class = []
    server.state.questions['a:c'].html_class = []
    server.state.h1 = None
    server.state.reject_content.append('<h1>Guest0</h1>')
    server.state.questions['a:a'].active = True
    # server.state.questions['a:b'].active = True
    server.get_next("?action=acl_change&value=-question_answerable_anytime")
    server.get_next("?action=acl_change&value=Yanswered_reset")
    server.get_next("?action=answered_reset")
    return server.get_next("?action=answered_see").check()

def check_make_comment(server):
    server.state.content = ['<PRE>COMent</PRE>']    
    return server.get_next("?action=action_comment_make&value=COMent").check()

def check_restart_server(server):
    server.state.nr_urls += 1
    server.get_next("?action=acl_change&value=Yaction_server_start")
    server.state.h1 = 'Guest0'
    server.state.display_times = True
    server.state.reject_content.remove('<h1>Guest0</h1>')
    return server.get_next("?action=action_server_start").check()
    
def check_question_a_after_reset(server):
    server.state.questions['a:a'].html_class = ['current_question', 'question_given']
    server.state.nr_given_questions += 1
    server.state.h1 = 'a:a'
    server.state.question = 'a'
    server.state.answer_form = True
    server.state.content = []
    return server.get_next("?action=question_see&question=a:a").check()

def check_reload(server):
    server.get_next("?action=acl_change&value=Yanswers_reload")
    server.state.nr_urls += 1
    server.state.h1 = "<span style=\"font-size:50%\"><a class=\"answers_reload tips\" href=\"?action=answers_reload\"><span></span></a>\n</span> a:a"
    server.state.questions['a:a'].active = True
    server.state.questions['a:b'].active = False
    a = server.get_next("?action=answers_reload").check()
    server.state.nr_urls -= 1
    server.state.h1 = 'a:a'
    server.get_next("?action=acl_change&value=-answers_reload")
    return a

def check_some_answers_after_reset(server):
    server.get_next("?action=question_answerable&value=a")
    server.get_next("?action=question_see&question=a:b")
    server.get_next("?action=question_indice_see")
    server.get_next("?action=question_indice_see")
    server.state.nr_indices += 2
    server.state.nr_good_answers += 2
    server.state.h1 = 'a:b'
    server.state.question = 'b'
    server.state.questions['a:a'].html_class.remove('current_question')
    server.state.questions['a:b'].html_class += [ 'current_question',
                                                  'question_given',
                                                  'indice_given'
                                                  ]
    return server.get_next("?action=question_answerable&value=b").check()

def check_see_acl_2(server):
    server.state.nr_urls += 3 # URL for columns title
    server.state.urls += [ '?sort_column=2' ]
    server.state.no_question_displayed()
    return server.get_next("?action=acl_see").check()

def check_see_acl_3(server):
    server.state.nr_urls += 3 # URL for students guest0 and guest1 and yours
    server.state.urls += [ '?student=guest0&action=acl_see_other' ]        
    return server.get_next("?action=acl_change&value=Yacl_see_other").check()

def check_see_acl_student(server):
    server.state.content = [ "class=\"acl_button\"" ]
    return server.get_next("?student=guest0&action=acl_see_other").check()

def check_see_acl_student_2(server):
    server.state.title = 'Guest1'
    server.state.h1 = 'Guest1'
    server.state.content = [ 'class="r1"' ]
    return server.get_next("?student=guest1&action=acl_see_other").check()

def check_see_acl_student_3(server):
    server.state.content = [ "class=\"acl_button\"" ]
    return server.get_next("?action=acl_change&value=Yacl_change_other").check()

def check_change_acl_student_give(server):
    page = server.get_next("?action=acl_change_other&value=-answer")
    f = open("Students/regtest/Logs/guest1/acl", "r")
    if not '\n!answer\n' in '\n' + f.read() + '\n':
        return "BUG"
    f.close()
    return page.check()

def check_change_acl_student_take(server):
    page = server.get_next("?action=acl_change_other&value=Yanswer")
    f = open("Students/regtest/Logs/guest1/acl", "r")
    if not '\nanswer\n' in '\n' + f.read() + '\n':
        return "BUG"
    f.close()
    return page.check()

def check_change_acl_student_default(server):
    page = server.get_next("?action=acl_change_other&value=.answer")
    f = open("Students/regtest/Logs/guest1/acl", "r")
    if '\nanswer\n' in '\n' + f.read() + '\n':
        return "BUG"
    f.close()
    return page.check()

def check_see_default_acls(server):
    server.state.nr_urls += 1
    return server.get_next("?action=acl_change&value=Yacl_default_see").check()
    
def check_default_acls(server):
    server.state.title = 'Guest0'
    server.state.h1 = 'Guest0'
    server.state.content = [ 'class="r1"',  'class="r2"' ]

    return server.get_next("?action=acl_default_see").check()

def check_change_default_acls(server):
    server.state.content = [ '<button' ]
    return server.get_next("?action=acl_change&value=Yacl_default_change").check()

def check_change_default_acls_give(server):
    server.state.nr_urls += 1 # action_help_see
    page = server.get_next("?action=acl_default_change&value=Yaction_help_see")
    f = open("Students/regtest/acls.default", "r")
    if not '\naction_help_see\n' in '\n' + f.read() + '\n':
        return "BUG"
    f.close()
    return page.check()

def check_change_default_acls_take(server):
    server.state.nr_urls -= 1 # action_help_see
    page = server.get_next("?action=acl_default_change&value=-action_help_see")
    f = open("Students/regtest/acls.default", "r")
    if '\naction_help_see\n' in '\n' + f.read() + '\n':
        return "BUG"
    f.close()
    return page.check()


def check_statistics_students(server):
    server.state.urls.remove('?sort_column=2')
    server.state.urls.remove('?student=guest0&action=acl_see_other')
    server.state.no_question_displayed()
    server.state.content = [ ]
    server.state.nr_urls = 26
    server.state.urls += [ '?student=guest0&action=stat_see_other' ]
    server.state.urls += [ '?student=guest1&action=stat_see_other' ]
    server.get_next("?action=acl_change&value=Ystat_see_other")
    return server.get_next("?action=stat_see").check()



def check_remove_change_acl(server):
    server.get_next("?action=acl_default_see")
    server.state.urls = ['../../css.fr.css']
    server.state.nr_urls = 14
    server.get_next("?action=acl_change&value=-answered_see")
    server.get_next("?action=acl_change&value=-stat_see")
    server.get_next("?action=acl_change&value=-acl_see")
    server.get_next("?action=acl_change&value=-action_server_stop")
    server.get_next("?action=acl_change&value=-action_server_start")
    server.get_next("?action=acl_change&value=-answered_reset")
    return server.get_next("?action=acl_change&value=-acl_change").check()
    
def check_add_change_acl(server):
    server.get_next("?action=acl_change&value=Yacl_change")
    server.get_next("?action=acl_change&value=Yquestion_display_any")
    return server.get_next("?action=acl_change&value=Yquestion_answerable_any").check()

def check_connection_guest1(server):
    server.state.no_question_displayed(h1='Guest1')
    server.state.title = 'Guest1'
    server.state.nr_urls = 3
    server.state.urls = ['../../css.fr.css']
    server.state.questions = {}
    server.state.add_question(Question('a:a', active=True))
    server.state.content = ("<P CLASS=\"choose\"></P>",)
    server.state.reject_content = ['Indice B', 'a:b']
    server.state.nr_indices = 0
    server.state.nr_good_answers = 0
    server.state.nr_bad_answers = 0
    return server.get("guest1").check()

def check_guest1_answers(server):
    server.get_next("?action=question_see&question=a:a")
    server.get_next("?action=question_answerable&value=a")
    server.get_next("?action=question_see&question=a:c")
    server.get_next("?action=question_see&question=a:b")
    server.get_next("?action=question_answerable&value=bb")
    server.get_next("?action=action_comment_make&value=bcom")
    server.get_next("?action=question_see&question=a:c")
    server.get_next("?action=question_see&question=a:b")
    server.get_next("?action=question_indice_see")
    server.state.nr_good_answers = 2
    server.state.nr_bad_answers = 1
    server.state.nr_given_questions = 2
    server.state.nr_indices = 1
    server.state.h1 = 'a:b'
    server.state.nr_urls = 4
    del server.state.questions['a:a']
    server.state.add_question(Question('a:c'))
    server.state.add_question(Question('b:B', active=True))
    server.state.reject_content = ['a:a']
    server.state.questions['a:c'].html_class = ['question_given', 'resigned']
    server.state.content = [ "good_b",
                             "question_given\">2",
                             "question_view\">3 (1.5)",
                             "question_good\">2 (1)",
                             "question_bad\">1 (0.5)",
                             "question_indice\">3 (1.5)",
                             "question_comment\">1 (0.5)",
                             ]
    return server.get_next("?action=question_answerable&value=b").check()

def check_guest0_answers(server):
    return server.get_next("?action=answered_see_other&student=guest0").check()

def check_guest0_answers_2(server):
    server.get_next("?action=acl_change&value=Yanswered_see_other")
    server.state.content = ['Indice A']
    server.state.h1 = 'Guest0'
    server.state.title = 'Guest0'
    server.state.reject_content = []
    return server.get_next("?action=answered_see_other&student=guest0").check()


def check_reset_stat_student(server):
    return server.get_next("?action=answered_reset_other&student=guest0").check()

def check_reset_stat_student_2(server):
    server.state.reject_content = ['Indice A']
    server.state.content = []
    server.get_next("?action=acl_change&value=Yanswered_reset_other")
    return server.get_next("?action=answered_reset_other&student=guest0").check()


def check_question_stats(server):
    server.state.no_question_displayed()
    server.state.content = [ ]
    server.state.reject_content = []
    server.state.nr_urls = 19
    server.state.h1 = 'Guest1'
    server.state.title = 'Guest1'
    server.state.table = [
        [ 1, [' 0.500', ' 0.500', ' 0.500'] ],
        [ 2, [' 1.000', ' 2.000', ' 2.000'] ],
        [ 3, [' 1.000', ' 1.000', ' 0.000'] ],
        [ 4, [' 0.000', ' 1.000', ' 0.000'] ],
        [ 5, [' 0.000', ' 1.000', ' 0.000'] ],
        [ 7, [' 0.000', ' 1.000', ' 0.000'] ],
        ]

    server.get_next("?action=stat_questions_see")
    return server.get_next("?sort_column=0").check()



def check_csv(server):
    page = server.get_next("?action=stat_others_csv_see")
    if page.content != """Nom,Bonne,Mauvaise,Indices,Heures
,1:beaucoup,1:beaucoup,1:beaucoup,
guest0, 0.00, 0.00, 0.00,  0.0
guest1, 1.00, 1.00, 1.00,  0.0
""":
        return "Bad content for CSV:" + page.content
    return None, len(page.content), 0

def check_histogram_good(server):
    server.state.nr_urls = 8
    server.state.table = None
    
    server.state.content = ["""  0-  0 : *
  1-  1 : 
  2-  2 : *
"""]
    return server.get_next("?action=stat_histogram_good_see&value=1").check()

def check_connection_guest2(server):
    "Connection to an user without acl and log files"
    
    server.state.no_question_displayed()
    server.state.title = 'Guest2'
    server.state.h1 = 'Guest2'
    server.state.nr_urls = 3
    server.state.urls = ['../../css.fr.css']
    server.state.questions = {}
    server.state.add_question(Question('a:a', active=True))
    server.state.content = ("<P CLASS=\"choose\"></P>",)
    server.state.reject_content = ['Indice B', 'a:b']
    server.state.nr_indices = 0
    server.state.nr_good_answers = 0
    server.state.nr_bad_answers = 0
    return server.get("guest2").check()

def check_same_time(server):

    return None, 1, 0


# Check Weblint options
weblint = "weblint"

f = os.popen("%s --help" % weblint, "r")
c = f.read()
f.close()
if "-d " in c:
    weblint += " -d empty-container"



server = Server(34000)

try:
    for check in (
        check_connection,
        check_css,
        check_figures,
        check_question,
        check_reconnect,
        check_answer_a0,
        check_answer_a1,
        check_answer_a3,
        check_answer_empty,
        check_unauthorized_question,
        check_unauthorized_answer,
        check_answer_a,
        check_question_b,
        check_question_b_ask_indice,
        check_question_c,
        check_answer_c0,
        check_allow_show_acl,
        check_see_acl,
        check_question_c_once_more,
        check_comment_c,
        check_see_stat,
        check_answer_a_again,
        check_any_question,
        check_see_comments,
        check_answer_a_again_again,
        check_see_bad_answers,
        check_comments,
        check_required,
        check_see_answered,
        check_stop_server_1,
        check_stop_server_2,
        check_question_b_again,
        check_stop_server_3,
        check_question_b_again_again,
        check_see_answered_2,
        check_reset_stat,
        check_make_comment,
        check_restart_server,
        check_question_a_after_reset,
        check_reload,
        check_some_answers_after_reset,
        check_see_acl_2,
        check_see_acl_3,
        check_see_acl_student,
        check_see_acl_student_2,
        check_see_acl_student_3,
        check_change_acl_student_give,
        check_change_acl_student_take,
        check_change_acl_student_default,
        check_see_default_acls,
        check_default_acls,
        check_change_default_acls,
        check_change_default_acls_give,
        check_change_default_acls_take,
        check_statistics_students,

        check_remove_change_acl,        
        check_add_change_acl,
        check_connection_guest1,
        check_guest1_answers,
        check_guest0_answers,
        check_guest0_answers_2,
        check_reset_stat_student,
        check_reset_stat_student_2,
        check_question_stats,
        check_csv,
        check_histogram_good,
        check_connection_guest2,
        check_same_time,
        ):
        sys.stdout.write( "%32s" % check.__name__ )
        sys.stdout.flush()
        v = check(server)
        sys.stdout.write( " %03d" % server.nr )
        if isinstance(v, basestring):
            sys.stdout.write(" Problem:\n" + v + "\n")
            break
        else:
            try:
                sys.stdout.write("%7.1fkb %6.2fms OK\n" % (v[1]/1000., v[2]*1000))
            except:
                sys.stdout.write(str(v))



finally:
#except IOError:
    server.display_log()
    traceback.print_exc()
    try:
        if 'nowait' not in sys.argv:
            time.sleep(1000)
    finally:
        server.stop()



