#!/usr/bin/env python

#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2006 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr

import types
import random
import os
import utilities
import inspect
import sys
import cgi
import sre

current_evaluate_answer = None

class Required:
    def __init__(self, world, string):
        w = string.split(":")
        if len(w) == 2:
            self.world = w[0]
            self.question_name = w[1]
        elif len(w) == 1:
            self.world = world
            self.question_name = string
        else:
            raise ValueError("Do not use ':' in question name")

        q = self.question_name.split('(')
        if len(q) != 1:
            self.question_name = q[0]
            
            if q[1][-1] != ')':
                raise ValueError("Bad required name: " + string)
                        
            self.answer = sre.compile(q[1][:-1])
        else:
            self.answer = False

        self.name = self.world + ":" + self.question_name

    def question(self):
        return questions[self.name]

    def full_name(self):
        if self.answer:
            return self.name + '(' + self.answer + ')'
        else:
            return self.name


class Requireds:
    def __init__(self, requireds):
        self.requireds = requireds

    def __iter__(self):
        return self.requireds.__iter__()

    def answered(self, answered):
        for r in self.requireds:
            if not answered.get(r.name, False):
                return False
            if r.answer and not sre.match(r.answer, answered[r.name]):
                return False
        return True

    def names(self):
        return [r.name for r in self.requireds]

class Question:
    def __init__(self, world, arg, previous_question=[]):
        self.name = arg["name"]
        self.world = world
        self.short_name = self.name.split(':')[1]
        for c in self.name:
            if ord(c) >= 32 and c not in "/":
                continue
            raise ValueError("Bad question name (%s)" % c)

        self.question = arg["question"]
        if not isinstance(self.question, types.FunctionType):
            q = self.question
            def tmp():
                return q
            self.question = tmp
        self.tests = arg.get("tests", ())
        self.before = arg.get("before", None)
        self.good_answer = arg.get("good_answer", "")
        if not isinstance(self.good_answer, basestring):
            raise ValueError("good_answer must be a string")
        self.bad_answer = arg.get("bad_answer", "")
        if not isinstance(self.bad_answer, basestring):
            raise ValueError("bad_answer must be a string")
        self.nr_lines = int(arg.get("nr_lines", "1"))
        self.comment = []
        self.required = arg.get("required", previous_question)
        self.required = Requireds([Required(world, i) for i in self.required])
            
        self.indices = arg.get("indices", ())
        self.default_answer = arg.get("default_answer", "")
        if not self.tests:
            print self.name, "no tests defined !"
        self.evaluate_answer = current_evaluate_answer

    def html(self, menu="heart"):
        s = ""
        if self.before:
            s += utilities.div("before", self.before, menu)
        q = self.question()
        s += utilities.div("question", q, menu)
        return s

    def answers_html(self):
        s = ""
        s += "<table class=\"an_answer\"><caption>" + self.name + "</caption>" +\
             "<tbody>"
        if self.before:
            s += "<tr class=\"test_info\"><td colspan=\"6\">%s</td></tr>" % \
                 self.before
        s += "<tr class=\"test_info\"><td colspan=\"6\">%s</td></tr>" % self.question()
        s += "<tr class=\"test_header\">"
        for i in range(6):
            s += "<th class=\"c%d\"></th>" % i
        s += "</tr>"
        for t in self.tests:
            try:
                s += t('', get_answer=True)
            except TypeError:
                s += "<tr class=\"test_unknown\"><td colspan=6>" + utilities.answer_format(str(t)) + "</td></tr>"
        for i in self.indices:
            s += "<tr class=\"test_indice\"><td colspan=\"6\">%s</td></tr>" % i
        if self.bad_answer:
            s += "<tr class=\"test_bad_comment\"><td colspan=\"6\">%s</td></tr>" % self.bad_answer
        if self.good_answer:
            s += "<tr class=\"test_good_comment\"><td colspan=\"6\">%s</td></tr>" % self.good_answer
                
        s += "<tbody></table>"
        return s

    def check_answer(self, answer, state):
        answer = answer.strip(" \n\t\r").replace("\r\n", "\n")
        if self.nr_lines == 1 and answer.find("\n") != -1:
            return False, "VOTRE REPONSE CONTIENT DES RETOURS A LA LIGNE"

        full_comment = ''
        for t in self.tests:
            try:
                result, comment = t(answer, state=state)
            except TypeError:
                try:
                    result, comment = t(answer)
                except TypeError:
                    print t
                    raise
            full_comment += comment
            if result != None:
                return result, full_comment
        return False, full_comment

    # Comments starting by a white space are not
    # considered as usefull for the student.
    # It is used by questions.py/comment function.
    def answer_commented(self, answer):
        a = self.check_answer(answer, None)
        if a[0]:
            return "*" # Correct answer, so commented.
        c = a[1]
        c = sre.sub("<sequence.*</sequence>", "", c.replace("\n",""))
        return c != '' and c[0] != ' '

    def url(self):
        return "?action=question_see&question=%s" % cgi.urllib.quote(self.name)

    def a_href(self):
        return "<A HREF=\"%s\">%s</A>" % (self.url(), cgi.escape(self.name))

    def nr_indices(self):
        return len(self.indices)

    def __str__(self):
        return self.name

##    def __cmp__(self, other):
##        if other == None:
##            return 1
##        return cmp(self.name, other.name)

questions = {}
previous_question = ""

def add(**arg):
    """Add a question to the question base"""

    attributs = {
    "name": "Nom court de la question. Ne pas donner la reponse dans le nom de la question.",
    "required": "Liste des noms des questions auxquels l'etudiant doit avoir deja repondu",
    "before": "Ce que doit faire l'etudiant avant de lire la question",
    "question": "Texte de la question en HTML ou fonction retournant le texte",
    "indices": "Une liste d'indices pour aider l'etudiant a repondre",
    "tests": "Les tests de verification des reponses",
    "good_answer": "Texte a afficher si l'etudiant donne une bonne reponse",
    "bad_answer": "Texte a afficher si l'etudiant donne une mauvaise reponse",
    "nr_lines": "Nombre de lignes pour la reponse",
    "default_answer": "Reponse par defaut",
    }

    sys.stdout.write("*")
    sys.stdout.flush()
    world = inspect.currentframe().f_back.f_globals["__name__"].split(".")[-1]
    for a in arg.keys():
        if a not in attributs.keys():
            print "'%s' n'est pas un attribut de question" % a
            print "Les attributs possibles de questions sont:"
            for i in attributs.keys():
                print "\t%s: %s" % (i, attributs[i]) 
            raise KeyError("Voir stderr")
    arg["name"] = world + ":" + arg["name"]
    if arg["name"] in questions.keys():
        print "Une question porte deja le nom", arg["name"]
        raise KeyError("Voir stderr")
    global previous_question
    if previous_question and previous_question.split(":")[0] == world:
        pq = [previous_question]
    else:
        pq = []
    questions[arg["name"]] = Question(world, arg, previous_question=pq)
    previous_question = arg["name"]

def answerable(answered={}):
    """Returns the authorized question list.
    The parameter is the names of the questions yet answered.
    The prequired are checked here.
    """
    answerable = []
    for q in questions.values():
        if not answered.get(q.name,False) and q.required.answered(answered):
            answerable.append(q)

    return answerable

def nr_indices(question_name):
    try:
        return questions[question_name].nr_indices()
    except KeyError:
        return -1

def a_href(question_name):
    try:
        return questions[question_name].a_href()
    except KeyError:
        return question_name

def worlds():
    d = {}
    for q in questions.values():
        d[q.world] = 1
    return d.keys()

sorted_questions = []

def compare_questions(x, y):
    c = cmp(x.level, y.level)
    if c != 0:
        return c
    return cmp( len(y.used_by), len(x.used_by) ) # Most used first

# Very far from optimal algorithm
def sort_questions():
    for q in questions.values():
        q.level = 0
        q.used_by = []

    # Compute 'used_by'
    try:
        for q in questions.values():
            for r in q.required.names():
                questions[r].used_by.append(q.name)
    except KeyError:
        print 'FROM QUESTION:', q.name
        raise

    # Compute 'descendants'
    leave = []
    for q in questions.values():
        q.nr_childs = len(q.used_by)
        if q.used_by:
            q.descendants = dict.fromkeys(q.used_by)
        else:
            q.descendants = {}
            leave.append(q)

    while leave:
        node = leave.pop()
        for q in node.required.names():
            q = questions[q]
            q.descendants.update(node.descendants)
            q.nr_childs -= 1
            if q.nr_childs == 0:
                del q.nr_childs
                leave.append(q)

    # Compute 'level' (not optimal)
    while True:
        change = False
        for q in questions.values():
            max_level = [ questions[r].level for r in q.required.names() ]
            if max_level:
                max_level = max(max_level)
            else:
                max_level = 0
            if q.level != max_level + 1:
                q.level = max_level + 1
                change = True
        if not change:
            break

    global sorted_questions
    sorted_questions = questions.values()
    sorted_questions.sort(compare_questions)

    # Compute coordinates.
    # Questions without prerequisites are in the center.
    # Others are the nearer possible from their requisite.
    for q in questions.values():
        q.nr_parents = len(q.required.names())
    nodes = [q for q in questions.values() if q.level == 1]
    pixels = {}    
    while nodes:
        n = nodes.pop()
        r = n.required.names()
        if r:
            cxs = [questions[q].coordinates[0] for q in r]
            cys = [questions[q].coordinates[1] for q in r]
            cx = sum(cxs) / len(cxs)
            cy = sum(cys) / len(cxs)
        else:
            cx = cy = 0
        for d in spiral(cx,cy):
            if d not in pixels:
                n.coordinates = d
                pixels[d] = True
                break
        for q in n.used_by:
            q = questions[q]
            q.nr_parents -= 1
            if q.nr_parents == 0:
                nodes.append(q)
                del q.nr_parents
    minx = min([q.coordinates[0] for q in questions.values()])
    miny = min([q.coordinates[1] for q in questions.values()])
    for q in questions.values():
        q.coordinates = (q.coordinates[0] - minx, q.coordinates[1] - miny)
        


def spiral(x,y):
    yield x,y
    w = 3
    while True:
        y += 1
        yield x,y
        for i in range(w-2):
            x += 1
            yield x,y
        for i in range(w-1):
            y -= 1
            yield x,y
        for i in range(w-1):
            x -= 1
            yield x,y
        for i in range(w-1):
            y += 1
            yield x,y
        w += 2

# These functions returns the result and a comment
#    - True is the answer is correct
#    - False is the answer is bad
#    - None is it is neither good nor bad, but the comment can be != ''

def _format(s):
    r = ""
    for i in s:
        r += utilities.answer_format(str(i)) + "<BR>"
    return r[:-4]

def get_answer_html(classe, strings, comment):
    return ("<TR CLASS=\"a_test " + classe + "\"><td class=\"a\"></td>" +
            "<td class=\"b\"></td>" +
            "<td class=\"c\">" + _format(strings) + "</td>" +
            "<td class=\"d\">" + comment + "</td></tr>\n")

def get_str_answer(classe, strings, comment):
    return get_answer_html(classe + " test_string", strings, comment)

def good(strings, comment=''):
    strings = utilities.rewrite_string(strings)
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_good test_is', strings, comment)
        for string in strings:
            if v == string:
                return True, comment
        return None, ""
    return f

def good_if_contains(strings, comment=''):
    strings = utilities.rewrite_string(strings)
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_good test_is_in', strings, comment)
        for string in strings:
            if string in v:
                return True, comment
        return None, ""
    return f

def bad(strings, comment=''):
    strings = utilities.rewrite_string(strings)
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_is', strings, comment)
        for string in strings:
            if v == string:
                return False, comment
        return None, ""
    return f

def require(strings, comment=''):
    strings = utilities.rewrite_string(strings)
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_require', strings, comment)
        for string in strings:
            if v.find(string) == -1:
                return False, comment
        return None, ""
    return f

def require_startswith(string, comment=''):
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_startswith', string, comment)
        if not v.startswith(string):
            return False, comment
        return None, ""
    return f

def require_endswith(string, comment=''):
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_endswith', string, comment)
        if not v.endswith(string):
            return False, comment
        return None, ""
    return f

def reject(strings, comment=''):
    strings = utilities.rewrite_string(strings)
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_reject', strings, comment)
        for string in strings:
            if v.find(string) != -1:
                return False, comment
        return None, ""
    return f

def reject_startswith(string, comment=''):
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_notstartswith', string, comment)
        if v.startswith(string):
            return False, comment
        return None, ""
    return f

def reject_endswith(string, comment=''):
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad test_notendswith', string, comment)
        if v.endswith(string):
            return False, comment
        return None, ""
    return f


def comment(comment, require=None):
    def f(v, get_answer=False):
        if get_answer:
            return get_str_answer('test_bad', (), comment)
        # This is a dirty way to program,
        # but the white space is here to indicate that
        # it is not a real contextual comment on the student answer.
        # This white space is tested by questions.py:answer_commented
        if require != None:
            for string in require:
                if v.find(string) != -1:
                    break
            else:
                return None, ''
                
        return None, ' ' + comment + '<br>'
        
    return f

def require_int(v, get_answer=False):
    m = "<p class='int_required'></p>"
    if get_answer:
        return "<tr CLASS=\"test_bad\"><td colspan=6>" + m + "</td></tr>"
    try:
        v = int(v)
        return None, ""
    except ValueError:
        return False, m

def answer_length_is(length, comment=''):
    def f(v, get_answer=False):
        if get_answer:
            return get_answer_html('test_bad test_require test_length',
                                   (str(length),), comment)
        if len(v) != length:
            return False, comment
        return None, ""
    return f
    

def number_of_is(character, number, comment=''):
    def f(v, get_answer=False):
        if get_answer:
            return get_answer_html('test_bad test_require test_number_of',
                                   ("%s %d"%(character,number),), comment)
        if v.count(character) != number:
            return False, comment
        return None, ""
    return f




    
