#!/usr/bin/env python

#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2006 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr

import actions
import pages
import statistics

###############################################################################
# URL behaviours for actions
###############################################################################

all_acls = {}

class Acl:
    def __init__(self, name, action=None, page=None,
                 visible=False, unstopped=True):
        all_acls[name] = self
        self.name = name
        self.action = action   # Function modifying the 'state'
        self.page = page       # Page to display (DO NOT MODIFY THE STATE)
        self.visible = visible # In the action menu
        self.unstopped = unstopped # Authorized action when system stopped
        self.weight = len(all_acls)

    def authorized(self, stopped):
        return self.unstopped or not stopped

def initialize():
    Acl('question_shuffle'           )
    Acl('answered_bad_see'           )
    Acl('gui_classement'             )
    Acl('question_next'              )
    Acl('question_answerable_any'    )
    Acl('question_display_any'       )
    Acl('gui_smiley'                 )
    Acl('question_answerable_anytime')
    Acl('question_required_see'      )
    Acl('question_max_descendants'   )
    Acl('gui_times'                  )
    Acl('gui_autofocus'              )
    Acl('answers_source_see'         )
    Acl('question_pixel_map_circle'  )
    Acl('question_time_see'          )
    
    Acl('stat_see'           ,page  =  pages.page_stat          , visible=True)
    Acl('acl_see'            ,page  =  pages.page_acl           , visible=True)
    Acl('answered_see'       ,page  =  pages.page_answered      , visible=True)
    Acl('action_help_see'    ,page  =  pages.page_help          , visible=True)
    Acl('action_server_stop' ,action=actions.action_server_stop , visible=True)
    Acl('action_server_start',action=actions.action_server_start, visible=True)
    Acl('role_teacher_i_have',action=actions.role_teacher_i_have, visible=True)
    Acl('role_student_i_have',action=actions.role_student_i_have, visible=True)
    Acl('role_author_i_have' ,action=actions.role_author_i_have , visible=True)
    Acl('role_admin_i_have'  ,action=actions.role_admin_i_have  , visible=True)
    Acl('action_deconnection',action=actions.action_deconnection,
        page=pages.page_deconnection, visible=True)

    Acl('acl_default_see'         , page  =  pages.page_default_acls         )
    Acl('role_student_see'        , page  =  pages.page_role_student         )
    Acl('role_author_see'         , page  =  pages.page_role_author          )
    Acl('role_admin_see'          , page  =  pages.page_role_admin           )
    Acl('role_teacher_see'        , page  =  pages.page_role_teacher         )
    Acl('question_see'            , page  =  pages.page_question             )
    Acl('stat_see_other'          , page  =  pages.page_stat_student         )
    Acl('acl_see_other'           , page  =  pages.page_acl                  )
    Acl('stat_questions_see'      , page  =  pages.page_question_stats       )
    Acl('answers_graph'           , page  =  pages.page_graph                )
    Acl('answered_comments_see'   , page  =  pages.page_comments             )
    Acl('answers_html_see'        , page  =  pages.page_answers              )
    Acl('stat_others_see'         , page  =  pages.page_student_stats        )
    Acl('stat_histogram_good_see' , page  =  pages.page_histogram_good       )
    Acl('stat_histogram_level_see', page  =  pages.page_histogram_level      )
    Acl('stat_same_time_see'      , page  =  pages.page_see_same_time        )
    Acl('acl_see_other'           , page  =  pages.page_acl_student          )
    Acl('answered_see_other'      , page  =  pages.page_answered_student     )
    Acl('stat_others_csv_see'     , action=  pages.page_student_csv          )
    Acl('question_pixel_map_see'  , action=  statistics.question_pixel_map   )
    Acl('question_pixel_map_see_other',action=statistics.question_pixel_map_other)
    Acl('stat_plot_see'           , page  =  pages.plot_svg                  )
    Acl('answered_reset'          , action=actions.student_make_reset        )
    Acl('answered_reset_other'    , action=actions.student_make_reset_student)
    Acl('acl_change'              , action=actions.acl_change                )
    Acl('acl_default_change'      , action=actions.acl_default_change        )
    Acl('acl_change_other'        , action=actions.acl_change_other          )
    Acl('answers_reload'          , action=actions.answers_reload            )
    Acl('role_student_change'     , action=actions.role_student_change       )
    Acl('role_teacher_change'     , action=actions.role_teacher_change       )
    Acl('role_admin_change'       , action=actions.role_admin_change         )
    Acl('role_author_change'      , action=actions.role_author_change        )

    Acl('question_indice_see', action=actions.student_asks_indice, unstopped=False)
    Acl('question_answerable'       , action=actions.student_answers    , unstopped=False)

    Acl('action_comment_make', action=actions.student_make_comment,page=pages.page_comment)

###############################################################################

class Acls:
    def __init__(self, filename, default="", role=None):
        self.filename = filename
        self.state = None
        self.system_stopped = None
        self.set_role(role)
        self.read(default)

    def read(self, default=""):
        try:
            f = open(self.filename, 'r')            
            c = f.read()
            f.close()
        except IOError:
            c = default        
        c = '\n' + c + '\n'
        self.acls = {}
        for acl in all_acls.keys():
            if c.find('\n' + acl + '\n') != -1:
                self.acls[acl] = True
            if c.find('\n!' + acl + '\n') != -1:
                self.acls[acl] = False

    def write(self):
        f  = open(self.filename, "w")
        f.write( str(self) )
        f.close()

    def __iter__(self):
        return self.acls.__iter__()

    def keys(self):
        return self.acls.keys()

    def __getitem__(self, k):
        role = self
        while role:
            a = role.acls.get(k, None)
            if a == None:
                role = role.role
            elif a:
                if self.state == None:
                    raise ValueError('state should not be None')
                return all_acls[k].authorized(self.state.stopped)
            else:
                break
        return False

    def get(self, k):
        """Returns the real value as an integer
        0: False
        1: None
        2: True
        """
        acl = self.acls.get(k, None)
        if acl == None:
            return 1
        if acl:
            return 2
        else:
            return 0

    def __setitem__(self, k, v):
        self.action_menu_cached = None
        self.acls[k] = v

    def set_role(self, role):
        self.role = role
        self.action_menu_cached = None

    def __str__(self):
        s = ''
        for acl in self.acls:
            if self.acls[acl] == True:
                s += acl + '\n'
            elif self.acls[acl] == False:
                s += '!' + acl + '\n'
                
        return s

    def a_href(self, key, br='<br>', only_if_visible=False):
        if not self[key]:
            return ''
        if only_if_visible and not all_acls[key].visible:
            return ''
            
        return "<a class=\"%s tips\" href=\"?action=%s\"><span></span></a>%s\n" % (key, key, br)
        return "<a class=\"%s\" href=\"?action=%s\"></a>%s\n" % (key, key, br)

    def action_menu(self, system_stopped):

        if system_stopped != self.system_stopped:
            self.action_menu_cached = None
            self.system_stopped = system_stopped

        if self.action_menu_cached:
            return self.action_menu_cached

        k = all_acls.keys()
        k.remove("action_deconnection")
        k.remove("action_server_stop")
        k.remove("action_server_start")
        k.append("action_deconnection")
        if system_stopped:
            k.append("action_server_start")
        else:
            k.append("action_server_stop")
        action = ""
        for act in k:
            action += self.a_href(act, only_if_visible = True)

        self.action_menu_cached = action
        return self.action_menu_cached

