#!/usr/bin/env python

#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2007 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr

import os
import sys
import statistics
import time
import configuration
import questions
import acls
import server
import socket
import utilities

# To make casauth work we should not use a proxy
# And the mailcheck speed down shell startup.
for i in ('http_proxy', 'https_proxy', 'MAIL', 'MAILCHECK'):
    if os.environ.has_key(i):
        del os.environ[i]

def search_command(command, comment):
    f = os.popen('which %s' % command, 'r')
    if f.read() == '':
        sys.stderr.write('WARNING: "%s" not found. %s\n' % (command, comment))
    f.close()

search_command('ppmtogif',
               'So no the graphical question map for the student (netpbm)')
search_command('highlight',
               'So no Python source highlighting for the author')
search_command('weblint',
               'So no HTML testing in the regression tests')
search_command('xvcg',
               'So some picture of the question graph are not computed')
search_command('dot',
               'So some picture of the question graph are not computed')


def mkdir(name):
    if not os.path.exists(name):
        os.mkdir(name)

def write(filename, text, overwrite=True):
    if not overwrite and os.path.exists(filename):
        return
    f = open(filename, 'w')
    f.write(text)
    f.close()

def read(filename):
    try:
        f = open(filename, 'r')
    except IOError:
        return ''
    c = f.read()
    f.close()
    return c.strip()

def date_to_seconds(date):
    return time.mktime( time.strptime(date.strip(), "%H:%M %d/%m/%Y") )

class Session:
    def __init__(self, name):
        self.name = name
        self.cache = {}
        self.nr_requests = 0
        self.percentage_time_for_stat = None
        self.dir = os.path.join('Students', name, '')
    def set_questions_directory(self, dirname):
        if not os.path.exists(dirname):
            sys.stderr.write(dirname + ' question directory does not exists\n')
            sys.exit(1)
        write(self.dir + 'questions', dirname)
    def set_port(self, port):
        write(self.dir + 'port', port)
    def set_begin_date(self, date, overwrite=True):
        write(self.dir + 'begin_date', date, overwrite)
    def set_end_date(self, date, overwrite=True):
        write(self.dir + 'end_date', date, overwrite)
    def set_url(self, url, overwrite=True):
        write(self.dir + 'url', url.strip('/'), overwrite)

    def init(self):
        try:
            configuration.port = int(read(self.dir + 'port'))
        except ValueError:
            sys.stderr.write(
                "You should 'create' the session before starting it\n")
            sys.exit(0)
        configuration.dates = [
            date_to_seconds(read(self.dir + 'begin_date')),
            date_to_seconds(read(self.dir + 'end_date')),
            ]
        if self.percentage_time_for_stat:
            configuration.statistics_cpu_allocation = \
            self.percentage_time_for_stat
        configuration.questions = read(self.dir + 'questions')
        configuration.url = read(self.dir + 'url')
        if configuration.url == '':
            configuration.url = 'http://%s:%d' % (socket.getfqdn(),
                                                  configuration.port)
        url = configuration.url.split('/')
        if len(url) == 3:
            configuration.prefix = ''
        else:
            configuration.prefix = '/'.join(url[3:]) + '/'

        questions.modules = {}
        for i in  os.listdir(configuration.questions):
            if i == '__init__.py':
                continue
            if i.endswith('.py'):
                print '\n%20s' % str(i),
                i = i[:-3]
                name = configuration.questions.replace(os.path.sep, '.') + '.' + i
                # Get module object
                module = __import__(name)
                for directory in name.split('.')[1:]:
                    module = module.__dict__[directory]
                questions.modules[i] = module

        questions.sort_questions()

        os.chdir(self.dir)
        acls.initialize()
        configuration.acls_default = acls.Acls(
            'acls.default', default=configuration.acl_default_see)
        configuration.acls_student = acls.Acls(
            'acls.student', default=configuration.student_role,
            role=configuration.acls_default)
        configuration.acls_teacher = acls.Acls(
            'acls.teacher', default=configuration.teacher_role,
            role=configuration.acls_default)
        configuration.acls_author = acls.Acls(
            'acls.author', default=configuration.author_role,
            role=configuration.acls_default)
        configuration.acls_admin = acls.Acls(
            'acls.admin', default=configuration.admin_role,
            role=configuration.acls_default)
        os.chdir(configuration.root)

    def start(self):
        self.init()
        write(self.dir + 'pid', str(os.getpid()))
        write(self.dir + 'hostname', socket.gethostname())
        os.chdir(self.dir)
        try:
            server.run(self.nr_requests, self.cache)
        except KeyboardInterrupt:
            sys.stderr.write("A signal has stopped the server\n")
        os.chdir(configuration.root)
        os.remove(self.dir + 'pid')

    def stop(self):
        if read(self.dir + 'hostname') != socket.gethostname():
            sys.stderr.write("Can stop server, it does not run on this host\n")
            return
        pid = read(self.dir + 'pid')
        if pid != '':
            try:
                os.kill( int(pid), 15)
            except OSError:
                pass
            os.remove(self.dir + 'pid')

    def plot(self):
        import pages

        self.init()
        os.chdir(self.dir)
        write('HTML/simple.html', statistics.html_simple())
        statistics.update_stats()
        write('HTML/state.svg', statistics.plot_svg(''))
        statistics.update_stats()
        statistics.graph()
        statistics.graph_dot()
        statistics.graph2_dot()
        pages.page_plot(None)
        sys.exit(0)

    def display_no_more_valid_answers(self):
        self.init()
        os.chdir(self.dir)
        statistics.display_no_more_valid_answers()
        sys.exit(0)

# Analyse command line options

name = sys.argv[0]
args = sys.argv[1:]
args.reverse()

try:
    session = Session(args.pop())
except IndexError:
    sys.stderr.write(name + """ the_session_name action1 action2 ...

SET PERSISTENT SESSION OPTIONS: 
    'create TheQuestionDirectory ThePortNumber'
        Create the session. For example: 'create Questions/unix 9999'
    'begin-date "%%H:%%M %%d/%%m/%%Y"'
        Set the date after which the students can answers questions.
        For example: 'create begin-date "09:00 1/1/2005"'
    'end-date "%%H:%%M %%d/%%m/%%Y"'
        Set the date after which the students can NOT answers questions.
    'admin login_name'
        Give to 'login_name' the administrator role
    'url public_URL_of_the_server'
        Use it if there is URL rewrite. For example, server run on :
           http://intranet.univ.org:7777/
        And students have only access to another server:
           http://www.univ.org/quenlig
        Apache configuration example:
           RewriteEngine On
           RewriteRule ^/quenlig(.*) http://intranet.univ.org:7777/$1 [P]

ACTIONS:
    'start'
        Start the server
    'stop'
        Stop the server
    'plot'
        Creation of all the plots related to this session
    'problems'
        Show the good answers given by the students that are no more
        valid because the question testing was modified.
SET TEMPORARY SESSION OPTIONS:
     'nocache'
         Does not cache HTML, CSS, PS, ... files
     'nr-requests-served #requests'
         The server stops after the number of request indicated
     'percentage-time-for-stat'
         Percentage of the time used by the server to compute stats.
         The default is %s%%
""" % configuration.statistics_cpu_allocation)

    if os.path.isdir('Students'):
        format = "%-12s %-5s %5s@%-8s %-17s %16s %s\n"

        sys.stderr.write(format % (
            "SESSION NAME", "PORT", "PID", "HOSTNAME", "QUESTIONS BASE",
            "START DATE", "DURATION"))

        for n in os.listdir('Students'):
            fn = os.path.join('Students', n)
            if not os.path.isdir(fn):
                continue
            begin_date = read(os.path.join('Students', n, 'begin_date'))
            end_date = read(os.path.join('Students', n, 'end_date'))
            duration = date_to_seconds(end_date) - date_to_seconds(begin_date)
            duration = utilities.duration(int(duration))
            sys.stderr.write(format % (
                n,
                read(os.path.join('Students', n, 'port')),
                read(os.path.join('Students', n, 'pid')),
                read(os.path.join('Students', n, 'hostname')),
                read(os.path.join('Students', n, 'questions')),
                begin_date,
                duration,
                ))    
    sys.exit(1)

mkdir('Students')
mkdir(session.dir)
session.set_begin_date('1:1 1/1/1970', overwrite=False)
session.set_end_date('3:3 3/3/2033', overwrite=False)    
mkdir(session.dir + 'Logs' )
mkdir(session.dir + 'HTML' )
configuration.root = os.getcwd()

configuration.version = os.path.basename(os.getcwd())

if len(args) == 0:
    sys.stderr.write("""
You indicate no action to do about your session named '%s'

Run without parameters to see the help.

Or create and run session, for example :

main.py unix2007 create Questions/unix 55555
main.py unix2007 admin  guestadmin
main.py unix2007 start
""" % session.name )
    sys.exit(1)

while args:
    action = args.pop()
    if action == 'start':
        session.start()
    elif action == 'stop':
        session.stop()
    elif action == 'nocache':
        session.cache = None
    elif action == 'plot':
        session.plot()
    elif action == 'problems':
        session.display_no_more_valid_answers()
    elif action == 'nr-requests-served':
        session.nr_requests = int(args.pop())
    elif action == 'percentage-time-for-stat':
        session.percentage_time_for_stat = int(args.pop())
    elif action == 'begin-date':
        session.set_begin_date(args.pop())        
    elif action == 'end-date':
        session.set_end_date(args.pop())        
    elif action == 'url':
        session.set_url(args.pop())        
    elif action == 'create':
        session.set_questions_directory(args.pop())
        session.set_port(args.pop())
    elif action == 'admin':
        user = os.path.join(session.dir, 'Logs', args.pop())
        mkdir(user)
        user = os.path.join(user, 'acl')
        write(user, read(user) + '\nrole_admin_i_have\n')
    else:
        sys.stderr.write("""Unknown action : %s\n""" % action)
        sys.exit(2)

