#!/usr/bin/env python
#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2007 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr
#

"""Replay on server 6666 the Unix question as answered
by the students with the server 9999.

This is done in order to simulate exactly the reality, except for:
  - The data files (CSS, images and so on)
  - The proxy usage

Script arguments are:
  - The number of simulated students (50)
  - The time slice for statistics (60 seconds)
  - The directory of existing student data (Students/L2unix2005s2/Logs)
  - The directory with the questions (Questions/unix)
    This must be the same than for the collected student data.
  - Acceleration of the reality (1)
  - Compute server profiling, if 0 NO profiling,
    else it give the number of retrieved URL before exiting
  - The port number
  - The name of the session


It prints the mean page load time and the number of loaded page.
"""

import os
import urllib2
import time
import sre
import sys
import cgi
import thread

for i in ('http_proxy', 'https_proxy'):
    if os.environ.has_key(i):
        del os.environ[i]

# Is this thread safe ?
class Stats:
    def __init__(self):
        self.nr = 0
        self.sum = 0
    def add(self, value):
        self.sum += value
        self.nr += 1
    def reset(self):
        a = str(self)
        self.__init__()
        return a
    def __str__(self):
        if self.nr:
            return "%f %d" % (self.sum/self.nr, self.nr)
        else:
            return "0 0"

def init_student_rights(name):
    f = open('Students/%s/acls.student' % name, 'w')
    f.write('''
question_answerable_any
question_shuffle
gui_smiley
gui_classement
''')
    f.close()
    
class Server:
    def __init__(self, port, questions, profiling, name):
        self.port = port
        self.name = name
        self.stats = Stats()
        self.clean()
        if profiling:
            profiling = 'nr-requests-served %s' % profiling
        else:
            profiling = ''
        os.system(
            'main.py %s stop ;' % name +
            'rm -r Students/%s || true ; ' % name +
            'main.py %s create %s %d begin-date "1:1 1/1/1970" end-date "1:1 1/1/2020" url "http://localhost:%d/" ;' % (
            name, questions, port, port) +
            'main.py %s %s start 2>Students/%s/logs >&2 &' % (
            name, profiling, name)
            )
        
        # Wait server start
        while True:
            try:
                urllib2.urlopen("http://localhost:%d/" % self.port).close()
                break
            except urllib2.URLError:
                time.sleep(0.4)

    def stop(self):
        f = open("Students/%d/pid" % self.port,"r")
        pid = int(f.read())
        f.close()
        os.kill(pid, 1)
        self.clean()

    def clean(self):
        os.system("rm -r Students/%s/Logs || true" % self.name)

    def get(self, url, trace=False):
        if trace:
            print 'GET', url
        start = time.time()
        f = urllib2.urlopen(url)
        p = f.read()
        f.close()
        if trace:
            print 'GET', time.time() - start
        self.stats.add(time.time() - start)
        return p

def unquote(s):
    "Anybody has a better idea for this complex function?"
    return '\\'.join([w.replace("\\r","\r").replace("\\n","\n").replace("\\a", "") for w in s.split('\\\\')])

class Action:
    def __init__(self, line):
        line = line.split('')
        if len(line) == 5:
            del line[2] # Seed in the old logs
        self.question = line[0]
        self.date = float(line[1])
        self.action = line[2]
        self.value = unquote(line[3][:-1])

    def __str__(self):
        return "%s;%f;%s;%s" % (
            self.question, self.date, self.action, self.value)

class Student:
    """Read real student data and replay it.
    Memorize the time to receive the page."""
    def __init__(self, server, student_log, student_name):
        self.server = server
        self.student_log = student_log
        self.student_name = student_name
        self.index = 0
        self.date = None
        self.base = "http://localhost:%d/%s" % (
            self.server.port, self.student_name)
        self.get('')

    def get_action(self):
        if self.index == len(self.student_log):
            return None
            
        action = Action(self.student_log[self.index])
        self.index += 1

        return action

    def get(self, url, trace=False):
        page = self.server.get(self.base + url, trace)
        if page.find('<base href='):
            self.base = sre.sub('(?s).*<base href="', '', page)
            self.base = sre.sub('(?s)".*', '', self.base)
        else:
            sys.stderr.write("Problem in the page\n")

    def do_an_action(self):
        action = self.get_action()
        if action == None:
            return
        if self.date:
            if action.date - self.date < 3600:
                time.sleep((action.date - self.date)/time_acceleration)
        self.date = action.date
        if action.action == 'asked':
            self.get('?action=question&question=%s' % cgi.urllib.quote(action.question))
        elif action.action == 'good' or action.action == 'bad':
            self.get('?action=answer&value=%s' % cgi.urllib.quote(action.value))
        elif action.action == 'question_indice_see':
            self.get('?action=question_indice_see')
        elif action.action == 'action_comment_make':
            self.get('?action=action_comment_make&value=%s#' % cgi.urllib.quote(action.value))
        elif action.action == 'None':            
            pass
        else:
            pass
            # print action

    def do_all_actions(self):
        while self.index != len(self.student_log):
            self.do_an_action()

def simulate_client(server, logs, name):
    Student(server, logs, name).do_all_actions()
    threads.pop()

try:
    port = int(sys.argv[7])
except:
    port = 6666
        

try:
    server = Server(port, sys.argv[4], int(sys.argv[6]), sys.argv[8])

    dirname = sys.argv[3]
    logs = []
    for filename in os.listdir(dirname):
        f = open(dirname + '/' + filename + '/log', 'r')
        logs.append( f.readlines() )
        f.close()

    nr_students = int(sys.argv[1])
    time_slice = int(sys.argv[2])
    time_acceleration = int(sys.argv[5])

    threads = []
    for i in range(nr_students):
        threads.append(i)
        thread.start_new_thread(simulate_client,(server,
                                                 logs[i%len(logs)],
                                                 'guest%d' % i)
                                                 )
        time.sleep(0.1)

    i = 0
    while threads:
        time.sleep(time_slice)
        print i, server.stats.reset()
        i += time_slice

finally:
    server.stop()


