#include "timeseries.h"

#define SHFT(a,b,c,d) (a)=(b); (b) = (c); (c) = (d);

double brent_angle(time_series ts, data_kernel dk, noise_model nm, options op, double *params_mle, double *cov_mle, int return_cov) {

	int j, jj, k, i, iter;
	int n_params, lwork, liwork, info;

	int imax       = 100;
	int count      = 0;
	int got_answer = 0;

	int *iwork, *ipiv;

	double a, b, d, etemp, fu, fv, fw, fx, p, q, r, tol1, tol2, u, v, w, x, xm;
	double wh_mle, MLE, xmin;
	double cn_mle, min_mle;
	double F1, F2, F3, F4, second_partial, cross_partial;

	double e         = 0.0;
	double zeps      = 1.0e-10;
	double cgold     = 0.3819660;
	double tolerance = 0.01;

	double *Eig_vectors, *Eig_values, *C_unit, *C;
	double *residuals, *Ioo;
	double *cov_wh, *params_wh, *cov_cn, *params_cn;
	double *fit, *cov_fit, *data_hat, *work, *radius;

	double *start, *params, *cov, *start_value;
	double *value, *wh, *cn, *mle, *par;

	clock_t time1, time2;

        /*****************************************************************/
        /* This is for a coloured noise plus white noise so n_params = 2 */
        /*****************************************************************/

        n_params = 2;

        /*******************************************************************************/
        /* First of all generate all the relevant matrices for the rest of the simplex */
        /*******************************************************************************/

        Eig_vectors = (double *) calloc((size_t)(dk.n_data*dk.n_data), sizeof(double));
        C           = (double *) calloc((size_t)(dk.n_data*dk.n_data), sizeof(double));
        C_unit      = (double *) calloc((size_t)(dk.n_data*dk.n_data), sizeof(double));
        Eig_values  = (double *) calloc((size_t) dk.n_data,            sizeof(double));
        residuals   = (double *) calloc((size_t) dk.n_data,            sizeof(double));

        fit         = (double *) calloc((size_t) dk.n_par,             sizeof(double));
        cov_fit     = (double *) calloc((size_t)(dk.n_par*dk.n_par),   sizeof(double));
        data_hat    = (double *) calloc((size_t) dk.n_data,            sizeof(double));

        lwork = 35 * dk.n_data;
        work        = (double *) calloc((size_t) lwork,                sizeof(double));

        liwork = 5 * dk.n_data;
        iwork       = (int *)    calloc((size_t) liwork,               sizeof(int));

        /*************************************************************************/
        /* Create the unit power law covariance matrix and find the eigen values */
        /* and the eigen vectors                                                 */
        /*************************************************************************/

	
	time1 = clock();

        create_covariance(nm, ts, op.cov_method, Eig_vectors, 1);

        dlacpy_("Full", &dk.n_data, &dk.n_data, Eig_vectors, &dk.n_data, C_unit, &dk.n_data);
        dsyev_("V","L",&dk.n_data,Eig_vectors,&dk.n_data,Eig_values,work,&lwork,&info);

        if (op.verbose) fprintf(op.fpout, " work(1)  = %f\n", work[0]);
        if (op.verbose) fprintf(op.fpout, " info     = %d\n", info);

        free(iwork);
        free(work);

        time2 = clock();

        if (op.verbose) {
                fprintf(op.fpout, " Time taken to create covariance matrix and compute eigen value and vectors : ");
                fprintf(op.fpout, "%f seconds\n", ((double) (time2-time1)) / CLOCKS_PER_SEC );
        }

        /***********************************************************/
        /* Find the white noise only and coloured noise only mle's */
        /***********************************************************/

        i = dk.n_par + 1;
        cov_wh    = (double *) calloc((size_t)(i*i), sizeof(double));
        cov_cn    = (double *) calloc((size_t)(i*i), sizeof(double));
        params_wh = (double *) calloc((size_t) i,    sizeof(double));
        params_cn = (double *) calloc((size_t) i,    sizeof(double));

        wh_mle = est_line_WH_only(dk, op, params_wh, cov_wh);

        cn_mle = cats_CN_only(dk, C_unit, Eig_values, Eig_vectors, params_cn, cov_cn, op.speed);

        if (op.verbose) {
		fprintf(op.fpout, " wh_only = %.8f (%.4f),",  params_wh[dk.n_par],  wh_mle); 
		fprintf(op.fpout, " cn_only = %.8f (%.4f)\n", params_cn[dk.n_par], -cn_mle);
	}

	/***********************************************************************/
	/* Now Begin the Brent one-dimensional minimisation based on the angle */
	/***********************************************************************/

	start_value  = (double *) calloc((size_t)(3),       sizeof(double));
	
	start_value[0] = 0.0;
	start_value[1] = 45.0;
	start_value[2] = 90.0;

	if (op.verbose) {
		fprintf(op.fpout, " Starting a one-dimensional minimisation : initial angle %5.2f\n", start_value[1]);
	}

	a = (start_value[0] < start_value[2] ? start_value[0] : start_value[2]);
	b = (start_value[0] > start_value[2] ? start_value[0] : start_value[2]);

	x = w = v = start_value[1];
	
	start      = (double *) calloc((size_t)(2),       sizeof(double));
	radius     = (double *) calloc((size_t)(1),       sizeof(double));
	mle        = (double *) calloc((size_t)(200),     sizeof(double));
	value      = (double *) calloc((size_t)(200),     sizeof(double));
	wh         = (double *) calloc((size_t)(200),     sizeof(double));
	cn         = (double *) calloc((size_t)(200),     sizeof(double));

	/* params     = (double *) calloc((size_t)(dk.n_par+1),                sizeof(double));
	cov        = (double *) calloc((size_t)((dk.n_par+1)*(dk.n_par+1)), sizeof(double)); */

	j = 0;

	if (op.speed == 3) {
		for (k = 0; k < dk.n_data; k++) residuals[k] = dk.d[k];
	} else {
		start[0] = tan(x*M_PI/180.0);
		start[1] = 1.0;
		cov_scale(C_unit,dk.n_data,start,C);
		linefit_all_fast(dk.A,dk.d,C,dk.n_data,dk.n_par,fit,cov_fit,data_hat);
		for (k = 0; k < dk.n_data; k++) residuals[k] = dk.d[k] - data_hat[k];
	}

	MLE = exact_radius(dk.n_data, Eig_values, Eig_vectors, residuals, x*M_PI/180.0, radius); 


	value[j] = x; mle[j] = MLE; wh[j] = radius[0] * sin(x*M_PI/180.0); cn[j] = radius[0] * cos(x*M_PI/180.0); j++;
	
	if (op.verbose) {
		fprintf(op.fpout, " angle = %9.6f ", x); 
		fprintf(op.fpout, "mle = %13.8f ", MLE);
		fprintf(op.fpout, "radius = %12.6f ", radius[0]*1000.0);
		fprintf(op.fpout, "wh = %12.6f ",  wh[j-1]*1000.0);
		fprintf(op.fpout, "cn = %12.6f\n", cn[j-1]*1000.0);
	}

	fw=fv=fx=-MLE;

	for (iter=0; iter < imax; iter++) {
		xm   = 0.5*(a+b);
		tol2 = 2.0*(tol1=tolerance*fabs(x)+zeps);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			got_answer = 1;
			xmin = x;
			break;
		}
		if (fabs(e) > tol1) {
			r = (x-w)*(fx-fv);
			q = (x-v)*(fx-fw);
			p = (x-v)*q-(x-w)*r;
			q = 2.0 * (q-r);
			if (q > 0.0) p = -p;
			q = fabs(q);
			etemp = e;
			e = d;
			if (fabs(p) >= fabs(0.5 * q* etemp) || p <= q*(a-x) || p >= q * (b-x)) {
				d = cgold*(e=(x >= xm ? a-x : b-x));
			} else {
				d = p / q;
				u = x+d;
				if (u-a < tol2 || b-u < tol2) {
					d = (xm-x < 0.0 ? -fabs(tol1) : fabs(tol1) );
				}
			}
		} else {
			d = cgold*(e=(x >= xm ? a-x : b-x));
		}

		u = (fabs(d) >= tol1 ? x+d : x + (d < 0.0 ? -fabs(tol1) : fabs(tol1) ));

		if (op.verbose) fprintf(op.fpout, " Next choice of angle = %9.6f\n", u);

		if (op.speed == 3) {
			for (k = 0; k < dk.n_data; k++) residuals[k] = dk.d[k];
		} else if (op.speed <= 1) {
			start[0] = tan(u*M_PI/180.0);
			start[1] = 1.0;
			cov_scale(C_unit,dk.n_data,start,C);
			linefit_all_fast(dk.A,dk.d,C,dk.n_data,dk.n_par,fit,cov_fit,data_hat);
			for (k = 0; k < dk.n_data; k++) residuals[k] = dk.d[k] - data_hat[k];
		}

		MLE = exact_radius(dk.n_data, Eig_values, Eig_vectors, residuals, u*M_PI/180.0, radius); 

		value[j] = u; mle[j] = MLE; wh[j] = radius[0] * sin(u*M_PI/180.0); cn[j] = radius[0] * cos(u*M_PI/180.0); j++;

		if (op.verbose) {
			fprintf(op.fpout, " angle = %9.6f ", u); 
			fprintf(op.fpout, "mle = %13.8f ", MLE);
			fprintf(op.fpout, "radius = %12.6f ", radius[0]*1000.0);
			fprintf(op.fpout, "wh = %12.6f ",  wh[j-1]*1000.0);
			fprintf(op.fpout, "cn = %12.6f\n", cn[j-1]*1000.0); 
		}

		fu = -MLE;

		if (fu <= fx) {
			if (u >= x) a = x; else b = x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a = u; else b = u;
			if (fu <= fw || w == x) {
				v = w;
				w = u;
				fv = fw;
				fw = fu;
			} else if (fu <= fv || v == x || v == w) {	
				v = u;
				fv = fu;
			}
		}
	}

	if (got_answer) {

		if (op.verbose) fprintf(op.fpout, " Finished : \n\n");
		for (k = 0; k < j; k++) {
			if (op.verbose) {
				fprintf(op.fpout, " Angle = %9.6f ", value[k]); 
				fprintf(op.fpout, "mle = %13.8f ", mle[k]);
				fprintf(op.fpout, "wh = %12.6f ",  wh[k]*1000.0);
				fprintf(op.fpout, "cn = %12.6f\n", cn[k]*1000.0); 
			}
			if (value[k] == xmin) {
				start[0] = wh[k];
				start[1] = cn[k];
			}
		}

		if (op.speed < 3) {
			cov_scale(C_unit,dk.n_data,start,C);
			linefit_all_fast(dk.A,dk.d,C,dk.n_data,dk.n_par,fit,cov_fit,data_hat);

			for (j = 0; j < dk.n_par; j++)  params_mle[j]          = fit[j];
			for (j = 0; j < n_params; j++) params_mle[j+dk.n_par] = start[j];
		} else {
			for (j = 0; j < dk.n_par; j++) params_mle[j]          = 0.0;
			for (j = 0; j < n_params; j++) params_mle[j+dk.n_par] = start[j];
		}

		i = dk.n_par + n_params;

		min_mle = MLE_nparam_CN(params_mle,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values,Eig_vectors);


		if (-min_mle < -cn_mle) {

			/***************************************************************/
			/* Check whether min_mle is lower than coloured_noise only mle */
			/***************************************************************/

			if (op.fpout != NULL) {
                                fprintf(op.fpout, " brent_angle : ");
                                fprintf(op.fpout, "mle (%f) is less ", -min_mle);
                                fprintf(op.fpout, "than cn only mle (%f)\n", -cn_mle);
                        }       

                        for (j = 0; j < dk.n_par; j++) params_mle[j] = params_cn[j];
                        params_mle[dk.n_par] = 0.0;
                        params_mle[dk.n_par+1] = params_cn[dk.n_par];

                        min_mle = cn_mle;

                        if (return_cov) {
                                for (j = 0; j < dk.n_par; j++) {
                                        for (k = 0; k < dk.n_par; k++) {
                                                cov_mle[j + k * i] = cov_cn[j + k * (dk.n_par + 1)];
                                        }
                                }
                                cov_mle[i*i-1] = cov_cn[(dk.n_par+1)*(dk.n_par+1)];
                        }

                } else if (-min_mle < wh_mle) {

			/***************************************************************/
			/* Check whether min_mle is lower than white_noise only mle */
			/***************************************************************/

                        if (op.fpout != NULL) {
                                fprintf(op.fpout, " brent_angle : ");
                                fprintf(op.fpout, "mle (%f) is less ", -min_mle);
                                fprintf(op.fpout, "than wh only mle (%f)\n", wh_mle);
                        }

                        for (j = 0; j < dk.n_par; j++) params_mle[j] = params_wh[j];
                        params_mle[dk.n_par] = params_wh[dk.n_par];
                        params_mle[dk.n_par+1] = 0.0;

                        min_mle = -wh_mle;

                        if (return_cov) {
                                for (j = 0; j < dk.n_par+1; j++) {
                                        for (k = 0; k < dk.n_par+1; k++) {
                                                cov_mle[j + k * i] = cov_wh[j + k * (dk.n_par + 1)];
                                        }
                                }
                        }

		/**************************************************/
		/* Okay the minimisation was the best so carry on */
		/**************************************************/

                } else if (return_cov) {

                        cov_scale(C_unit,dk.n_data,start,C);
                        linefit_all_fast(dk.A,dk.d,C,dk.n_data,dk.n_par,fit,cov_fit,data_hat);

                        /*********************************************/
                        /* Invert cov_fit and substitute it into Ioo */
                        /*********************************************/

                        i = dk.n_par + n_params;
                        Ioo = (double *) calloc((size_t)(i*i), sizeof(double) );

                        lwork = dk.n_par;
                        ipiv = (int *)    calloc((size_t) dk.n_par, sizeof(int));
                        work = (double *) calloc((size_t) lwork,    sizeof(double));

                        dgetrf_(&dk.n_par, &dk.n_par, cov_fit, &dk.n_par, ipiv, &info);
                        dgetri_(&dk.n_par, cov_fit, &dk.n_par, ipiv, work, &lwork, &info);

                        free(ipiv);
                        free(work);

                        /* free(params); */

                        i = dk.n_par + n_params;
                        work   = (double *) calloc( (size_t) i, sizeof(double) );
                        params = (double *) calloc( (size_t) i, sizeof(double) );

                        for (j = 0; j < dk.n_par; j++)  work[j]          = params[j]          = fit[j];
                        for (j = 0; j < n_params; j++)  work[j+dk.n_par] = params[j+dk.n_par] = start[j];

                        /*******************************************************************/
                        /* Now calculate an estimate of the covariance matrix of the MLE   */
                        /* based on the Fisher Information Matrix for the non-linear parts */
                        /*******************************************************************/

                        for (j = 0; j < dk.n_par; j++) {
                                for (k = 0; k < dk.n_par; k++) Ioo[j+k*i] = -cov_fit[j+k*dk.n_par];
                        }

                        /*************************************/
                        /* Now for the non-linear parameters */
                        /*************************************/

                        F2 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);

			for (j = dk.n_par; j < i; j++) {
                                params[j] = work[j] + 1e-3 * fabs(work[j]);
                                F1 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);
                                params[j] = work[j] - 1e-3 * fabs(work[j]);
                                F3 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);

                                second_partial = -(F1-2.0*F2+F3)/pow(1e-3*fabs(work[j]),2.0);
                                Ioo[j + j * i] = second_partial;
                                params[j] = work[j];
                        }

                        for (j = dk.n_par; j < i; j++) {
                                for (k = 0; k < j; k++) {
                                        params[j] = work[j] + 1e-3 / 2.0 * fabs(work[j]);
                                        params[k] = work[k] + 1e-3 / 2.0 * fabs(work[k]);
                                        F1 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);

                                        params[j] = work[j] - 1e-3 / 2.0 * fabs(work[j]);
                                        params[k] = work[k] + 1e-3 / 2.0 * fabs(work[k]);
                                        F2 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);

                                        params[j] = work[j] + 1e-3 / 2.0 * fabs(work[j]);
                                        params[k] = work[k] - 1e-3 / 2.0 * fabs(work[k]);
                                        F3 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);

                                        params[j] = work[j] - 1e-3 / 2.0 * fabs(work[j]);
                                        params[k] = work[k] - 1e-3 / 2.0 * fabs(work[k]);
                                        F4 = MLE_nparam_CN(params,i,dk.n_par,dk.n_data,dk.d,dk.A,Eig_values, Eig_vectors);

                                        cross_partial = -(F1-F2-F3+F4) / (1e-3 * fabs(work[j]) * 1e-3 * fabs(work[k]));
                                        Ioo[j + k * i] = Ioo[k + j * i] = cross_partial;

                                        params[j] = work[j];
                                        params[k] = work[k];
                                }
                        }

                        free(work);


                        /**********************/
                        /* Calculate inv(Ioo) */
                        /**********************/

                        lwork = i;
                        ipiv = (int *) calloc((size_t) i, sizeof(int));
                        work = (double *) calloc((size_t) lwork, sizeof(double));

                        dgetrf_(&i, &i, Ioo, &i, ipiv, &info);
                        dgetri_(&i, Ioo, &i, ipiv, work, &lwork, &info);

                        free(ipiv);
                        free(work);

                        /***************************/
                        /* Store -Ioo into cov_mle */
                        /***************************/

                        for (j = 0; j < i; j++) {
                                for (k = 0; k < i; k++) cov_mle[j + k * i] = -Ioo[j + k * i];
                        }
			free(params);
                        free(Ioo);
                }
	} else {
		fprintf(stderr, " brent_angle : Minimization did not converge!!\n");

                min_mle = 1e10;
                for (j = 0; j < i; j++)        params_mle[j] = 0.0;
                for (j = 0; j < n_params; j++) start[j]      = 0.0;
	}

	free(mle);
	free(residuals);
	free(start_value);
	free(Eig_vectors);
	free(C);
	free(C_unit);
	free(Eig_values);
	free(fit);
	free(cov_fit);
	free(data_hat);

	free(start);
	free(radius);
        free(value);
	free(wh);
	free(cn);

	free(cov_wh);
	free(cov_cn);
	free(params_wh);
	free(params_cn);

	return(-min_mle);
}
