

#include "App.h"
#include "nvwidgets/nvSdlWidgets.h"

#include "Transform.h"
#include "Camera.h"
#include "Orbiter.h"

#include "MeshIO.h"
#include "EffectIO.h"
#include "EffectShaderManager.h"
#include "BufferManager.h"

#ifndef GK_OPENGL4
#error opengl4 non supporte par gKit, definir GK_OPENGL4 dans premake4.lua, re-generer le projet et recompiler.
#endif


class Tesselation4 : public gk::App
{
    gk::EffectShaderManager *manager;
    gk::GLShaderProgram *program;
    gk::GLAttributeBuffer *positions;
    gk::GLIndexBuffer *indices;

    nv::SdlUIContext ui;
    float inner_factor;
    gk::Point edges_factor;

    std::string filename;
    gk::FirstPersonCamera camera;
    gk::Orbiter orbiter;
    
public:
    // creation de la fenetre avec les reglages par defaut.
    Tesselation4( )
        :
        gk::App()
    {
        gk::AppSettings settings;
        //~ settings.setSamples(4);
        createWindow(1024, 768, settings);
    }
    
    ~Tesselation4( ) {}
    
    // a redefinir pour utiliser les widgets.
    void processWindowResize( SDL_ResizeEvent& event )
    {
        ui.reshape(event.w, event.h);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseButtonEvent( SDL_MouseButtonEvent& event )
    {
        ui.processMouseButtonEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseMotionEvent( SDL_MouseMotionEvent& event )
    {
        ui.processMouseMotionEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processKeyboardEvent( SDL_KeyboardEvent& event )
    {
        ui.processKeyboardEvent(event);
    }
    
    int init( )
    {
        ui.init(windowWidth(), windowHeight());
        
        // creer l'utilitaire de creation de shader a partir d'un fichier .gkfx
        manager= new gk::EffectShaderManager("tessellation.gkfx");
        if(manager == NULL)
            return -1;
        
        // creer le shader program
        program= manager->createShaderProgram4("tessellation_triangle");
        if(program == NULL || program->createGLResource() < 0)
            // erreur de chargement des fichiers sources ou de compilation des shaders, sortir de l'application
            return -1;
        
        // charger un objet
        filename= "bigguy.obj";
        gk::Mesh *mesh= gk::MeshIO::read(filename);
        if(mesh == NULL)
            return -1;
        
        // creer un buffer pour stocker les positions des sommets
        positions= gk::createAttributeBuffer(mesh->positionCount(), 
            sizeof(gk::Point) * mesh->positionCount(), &mesh->positions().front());
        if(positions == NULL || positions->createGLResource() < 0)
            return -1;
        
        // creer un buffer pour stocker les indices des sommets de chaque triangle
        indices= gk::createIndexBuffer(mesh->indiceCount(),
            sizeof(unsigned int) * mesh->indiceCount(), &mesh->indices().front());
        if(indices == NULL || indices->createGLResource() < 0)
            return -1;
        
        // parametres par defaut de la tessellation
        inner_factor= 2.f;
        edges_factor.x= 2.f;
        edges_factor.y= 2.f;
        edges_factor.z= 2.f;
        
        // init camera
        camera= gk::FirstPersonCamera(50.f, 1.f, 1.f, 1000.f, windowWidth(), windowHeight());
        camera.move( gk::Point(0.f, 0.f, 50.f) );
        
        // init object manipulator
        orbiter= gk::Orbiter();
        
        return 0;       // tout c'est bien passe, sinon renvoyer -1
    }
    
    int quit( ) 
    {
        return 0;
    }
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            // fermer l'application si l'utilisateur appuie sur ESCAPE
            Close();

        // camera
        camera.setViewport(windowWidth(), windowHeight());
        
        // controle de la camera a la souris
        int x, y;
        int button= SDL_GetRelativeMouseState(&x, &y);
        if(button & SDL_BUTTON(1))
        {
            // move camera
            camera.rotateUp(-x);
            camera.rotateRight(-y);
        }
        if(button & SDL_BUTTON(3))
        {
            // re-orient object
            orbiter.rotateUp(x);
            orbiter.rotateRight(y);
        }
        
        // controle au clavier de la camera
        if(SDL_GetModState() & KMOD_CTRL)
        {
            //~ if(key(SDLK_UP))
                //~ camera.rotateRight(1.f);
            //~ if(key(SDLK_DOWN))
                //~ camera.rotateRight(-1.f);
            //~ if(key(SDLK_LEFT))
                //~ camera.rotateUp(1.f);
            //~ if(key(SDLK_RIGHT))
                //~ camera.rotateUp(-1.f);
        }
        else
        {
            if(key('z'))
                camera.moveForward(-1.f);
            if(key('s'))
                camera.moveForward(1.f);
            if(key('q'))
                camera.moveRight(-1.f);
            if(key('d'))
                camera.moveRight(1.f);
            if(key(SDLK_PAGEUP))
                camera.moveUp(1.f);
            if(key(SDLK_PAGEDOWN))
                camera.moveUp(-1.f);
            
            if(key('c'))        // copier / coller la position de la camera
            {
                key('c')= 0;
                camera.write(filename);
            }
            if(key('v'))
            {
                key('v')= 0;
                camera.read(filename);
            }
            
            if(key(SDLK_PLUS))
                camera.setSpeed(camera.speed() * 1.1f);
            if(key(SDLK_MINUS))
                camera.setSpeed(camera.speed() * .9f);
        }
        
        glViewport(0, 0, windowWidth(), windowHeight());
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // dessiner les aretes sans remplir les triangles
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        
        // activer le shader
        glUseProgram(program->name());
        
        // parametrer le shader, par exemple, mvpMatrix
        GLint location= program->attribute("position").location();
        glBindBuffer(GL_ARRAY_BUFFER, positions->name());
        glVertexAttribPointer(location, 3, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(location);
        
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, indices->name());
        
        gk::setUniform(program->uniform("inner_factor"), inner_factor, 0.f);
        gk::setUniform(program->uniform("edge_factor"), edges_factor.x, edges_factor.y, edges_factor.z, 0.f);

        gk::Transform model= orbiter.transform();
        gk::Transform view= camera.view();
        gk::Transform projection= camera.projection();
        gk::Transform mvp= projection * view * model;

        gk::setUniform(program->uniform("mvpMatrix"), mvp.matrix());

        //~ glLineWidth(2.f);
        glPatchParameteri(GL_PATCH_VERTICES, 3);
        glDrawElements(GL_PATCHES, indices->count(), GL_UNSIGNED_INT, 0);
        
        // nettoyage
        glDisableVertexAttribArray(location);
        glUseProgram(0);
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        
        // interface
        char tmp[1024]= { 0 };
        ui.begin();
            ui.beginGroup(nv::GroupFlags_GrowDownFromLeft);

            sprintf(tmp, "inner level %.1f", inner_factor);
            ui.doLabel(nv::Rect(), tmp);
            ui.doHorizontalSlider(nv::Rect(0, 0, 400, 0), 1.f, 32.f, &inner_factor);
            
            sprintf(tmp, "edge1 level %.1f", edges_factor.x);
            ui.doLabel(nv::Rect(), tmp);
            ui.doHorizontalSlider(nv::Rect(0, 0, 400, 0), 1.f, 32.f, &edges_factor.x);
            
            sprintf(tmp, "edge2 level %.1f", edges_factor.y);
            ui.doLabel(nv::Rect(), tmp);
            ui.doHorizontalSlider(nv::Rect(0, 0, 400, 0), 1.f, 32.f, &edges_factor.y);
            
            sprintf(tmp, "edge3 level %.1f", edges_factor.z);
            ui.doLabel(nv::Rect(), tmp);
            ui.doHorizontalSlider(nv::Rect(0, 0, 400, 0), 1.f, 32.f, &edges_factor.z);

            ui.endGroup();
        ui.end();
        
        if(key(SDLK_LEFT))
        {
            key(SDLK_LEFT)= 0;
            // bouge toutes les valeurs
            inner_factor= std::max(1.f, (inner_factor -1.f));
            edges_factor.x= std::max(1.f, (edges_factor.x -1.f));
            edges_factor.y= std::max(1.f, (edges_factor.y -1.f));
            edges_factor.z= std::max(1.f, (edges_factor.z -1.f));
        }
        if(key(SDLK_RIGHT))
        {
            key(SDLK_RIGHT)= 0; 
            // bouge toutes les valeurs
            inner_factor= std::min(32.f, (inner_factor +1.f));
            edges_factor.x= std::min(32.f, (edges_factor.x +1.f));
            edges_factor.y= std::min(32.f, (edges_factor.y +1.f));
            edges_factor.z= std::min(32.f, (edges_factor.z +1.f));
        }
            
        // afficher le buffer de dessin
        SDL_GL_SwapBuffers();
        return 1;       // continuer
    }
};


int main( int argc, char **argv )
{
    Tesselation4 app;
    app.run();
    
    return 0;
}

