
#include "GL/GLPlatform.h"
#include "IOFileSystem.h"

#include "EffectIO.h"
#include "EffectShaderManager.h"

#include "Mesh.h"
#include "MeshIO.h"
#include "BufferManager.h"

#include "SamplerManager.h"
#include "GLTexture2DIO.h"

#include "App.h"

#include "Camera.h"
#include "Orbiter.h"

class TP : public gk::App
{
    gk::Effect *m_effect;
    gk::GLShaderProgram *m_program;
    gk::GLAttributeBuffer *m_positions;
    gk::GLAttributeBuffer *m_normals;
    gk::GLAttributeBuffer *m_texcoords;
    gk::GLIndexBuffer *m_index;
    gk::GLSampler *m_sampler;
    gk::Mesh *m_mesh;
    std::vector<gk::GLTexture *> m_diffuse_textures;
    std::vector<gk::GLTexture *> m_specular_textures;
    
    std::string m_filename;
    gk::FirstPersonCamera m_camera;
    gk::Orbiter m_orbiter;
    
public:
    TP( const int w= 1024, const int h= 768 )
        :
        gk::App(w, h),
        m_effect(NULL),
        m_program(NULL),
        m_positions(NULL),
        m_normals(NULL),
        m_texcoords(NULL),
        m_index(NULL),
        m_sampler(NULL),
        m_mesh(NULL),
        m_diffuse_textures(),
        m_specular_textures(),
        m_filename(),
        m_camera(),
        m_orbiter()
    {}
    
    ~TP( ) {}
    
    int init( )         // charger les objets, les vertex buffers, etc.
    {
        // init camera
        m_camera= gk::FirstPersonCamera(50.f, 1.f, 1.f, 1000.f, windowWidth(), windowHeight());
        m_camera.move( gk::Point(0.f, 0.f, 50.f) );
        
        // init object manipulator
        m_orbiter= gk::Orbiter();
        
        m_effect= gk::EffectIO::read("core_simple.gkfx");
        if(m_effect == NULL)
            return -1;
        
        m_program= gk::EffectShaderManager(m_effect).createShaderProgram("diffuse.program");
        if(m_program == NULL || m_program->createGLResource() < 0)
            return -1;
        
        m_filename= m_effect->find("bigguy")->asFile();
        m_mesh= gk::MeshIO::read( m_filename );
        if(m_mesh == NULL)
            return -1;
        
        if(m_mesh->normalCount() != m_mesh->positionCount())
            m_mesh->buildNormals();
        
        m_positions= gk::createAttributeBuffer(
            m_mesh->positionCount(), 
            m_mesh->positionCount() * sizeof(float [3]), &m_mesh->positions().front());
        if(m_positions == NULL || m_positions->createGLResource() < 0)
            return -1;
        
        m_normals= gk::createAttributeBuffer(
            m_mesh->normalCount(),
            m_mesh->normalCount() * sizeof(float [3]), &m_mesh->normals().front());
        if(m_normals == NULL || m_normals->createGLResource() < 0)
            return -1;
        
        if(m_mesh->texCoordCount() > 0)
        {
            m_texcoords= gk::createAttributeBuffer( 
                m_mesh->texCoordCount(),
                m_mesh->texCoordCount() * sizeof(float [2]), &m_mesh->texCoords().front());
            if(m_texcoords == NULL || m_texcoords->createGLResource() < 0)
                return -1;
        }
            
        if(m_mesh->indiceCount() > 0)
        {
            m_index= gk::createIndexBuffer(
                m_mesh->indiceCount(), 
                m_mesh->indiceCount() * sizeof(unsigned int), &m_mesh->indices().front());
            if(m_index == NULL || m_index->createGLResource() < 0)
                return -1;
        }
    
        // charger les textures utilisees par les matieres
        if(m_mesh->materialCount() > 0 && m_texcoords != NULL)
        {
            const std::vector<gk::MeshMaterial *>& materials= m_mesh->materials();
            const int count= (int) materials.size();
            for(int i= 0; i < count; i++)
            {
                const gk::MeshMaterial *material= materials[i];
                assert(material != NULL);
                gk::GLTexture *diffuse= NULL;
                if(material->diffuse_texture.empty() == false)
                    diffuse= gk::GLTexture2DIO::read(gk::UNIT0, material->diffuse_texture);
                m_diffuse_textures.push_back(diffuse);
                
                gk::GLTexture *specular= NULL;
                if(material->specular_texture.empty() == false)
                    specular= gk::GLTexture2DIO::read(gk::UNIT0, material->specular_texture);
                m_specular_textures.push_back(specular);
            }
            
            m_sampler= gk::createAnisotropicSampler(1);
            if(m_sampler == NULL || m_sampler->createGLResource() < 0)
                return -1;
        }
        
        return 0;       // tout c'est bien passe
    }
    
    int quit( )         
    {
        // toutes les ressources seront liberees par shaderManager et bufferManager, etc.
        return 0;
    }
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            // sortir si l'utilisateur appuye sur ESC
            Close();

        // camera
        m_camera.setViewport(windowWidth(), windowHeight());
        
        // controle de la camera a la souris
        int x, y;
        int button= SDL_GetRelativeMouseState(&x, &y);
        if(button & SDL_BUTTON(1))
        {
            // move camera
            m_camera.rotateUp(-x);
            m_camera.rotateRight(-y);
        }
        if(button & SDL_BUTTON(3))
        {
            // re-orient object
            m_orbiter.rotateUp(x);
            m_orbiter.rotateRight(y);
        }
        
        // controle au clavier de la camera
        if(SDL_GetModState() & KMOD_CTRL)
        {
            if(key(SDLK_UP))
                m_camera.rotateRight(1.f);
            if(key(SDLK_DOWN))
                m_camera.rotateRight(-1.f);
            if(key(SDLK_LEFT))
                m_camera.rotateUp(1.f);
            if(key(SDLK_RIGHT))
                m_camera.rotateUp(-1.f);
        }
        else
        {
            if(key('z') || key(SDLK_UP))
                m_camera.moveForward(-1.f);
            if(key('s') || key(SDLK_DOWN))
                m_camera.moveForward(1.f);
            if(key('q') || key(SDLK_LEFT))
                m_camera.moveRight(-1.f);
            if(key('d') || key(SDLK_RIGHT))
                m_camera.moveRight(1.f);
            if(key(SDLK_PAGEUP))
                m_camera.moveUp(1.f);
            if(key(SDLK_PAGEDOWN))
                m_camera.moveUp(-1.f);
            
            if(key('c'))        // copier / coller la position de la camera
            {
                key('c')= 0;
                m_camera.write(m_filename);
            }
            if(key('v'))
            {
                key('v')= 0;
                m_camera.read(m_filename);
            }
            
            if(key(SDLK_PLUS))
                m_camera.setSpeed(m_camera.speed() * 1.1f);
            if(key(SDLK_MINUS))
                m_camera.setSpeed(m_camera.speed() * .9f);
        }
        
        // redimensionner la fenetre, si necessaire.
        glViewport(0, 0, windowWidth(), windowHeight());
        // effacer le buffer de dessin.
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // dessiner quelque chose.
        glUseProgram(m_program->name());

        gk::Transform model= m_orbiter.transform();
        gk::Transform view= m_camera.view();
        gk::Transform projection= m_camera.projection();
        gk::Transform mvp= projection * view * model;
        gk::Transform mv= view * model;
        
        // passer la matrice model view projection au shader
        gk::setUniform(m_program->uniform("mvpMatrix"), mvp.matrix());
        
        // parametrer les autres uniforms du shader
        // recuperer la valeur dans le script
        gk::Vec4Value color= m_effect->find("uniform.color")->asVec4();
        gk::setUniform(m_program->uniform("color"), color.x, color.y, color.z, color.w);

        // activer les buffers d'attributs de sommets
        GLint positions= m_program->attribute("position").location();
        glBindBuffer(GL_ARRAY_BUFFER, m_positions->name());
        glVertexAttribPointer(positions, 3, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(positions);
        
        
        //~ GLint normals= m_program->attribute("normal").location();
        //~ glBindBuffer(GL_ARRAY_BUFFER, m_normals->name());
        //~ glVertexAttribPointer(normals, 3, GL_FLOAT, GL_FALSE, 0, 0);
        //~ glEnableVertexAttribArray(normals);
        
        if(m_texcoords != NULL)
        {
            GLint texcoords= m_program->attribute("texcoord").location();
            glBindBuffer(GL_ARRAY_BUFFER, m_texcoords->name());
            glVertexAttribPointer(texcoords, 2, GL_FLOAT, GL_FALSE, 0, 0);
            glEnableVertexAttribArray(texcoords);
        }
        
        if(m_index != NULL)
        {
            // draw indexe, activer le buffer d'indexation
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_index->name());
            
            // parcourir les submeshes
            const std::vector<gk::SubMesh>& submeshes= m_mesh->subMeshes();
            const int count= (int) submeshes.size();
            for(int i= 0 ; i < count; i++)
            {
                const gk::SubMesh& submesh= submeshes[i];
                const gk::MeshMaterial& material= m_mesh->subMeshMaterial(i);
                gk::GLTexture *texture= m_diffuse_textures[submesh.material_id];
                if(texture != NULL)
                {
                    glBindTexture(texture->target(), texture->name());
                    gk::setSamplerUniform(m_program->uniform("diffuse_texture"), 0);
                    glBindSampler(0, m_sampler->name());
                    glSamplerParameteri(m_sampler->name(), GL_TEXTURE_WRAP_S, GL_REPEAT);
                    glSamplerParameteri(m_sampler->name(), GL_TEXTURE_WRAP_T, GL_REPEAT);
                }
                
                glDrawElements(GL_TRIANGLES, submesh.end - submesh.begin, 
                    GL_UNSIGNED_INT, (const GLvoid *) (submesh.begin * sizeof(unsigned int)));
            }
            
            glDrawElements(GL_TRIANGLES, m_index->count(), GL_UNSIGNED_INT, 0);
            
            // nettoyer
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
        }
        else
        {
            // pas d'index, dessiner tout les points
            glDrawArrays(GL_POINTS, 0, m_positions->count());
        }

        // nettoyer
        glDisableVertexAttribArray(positions);
        //~ glDisableVertexAttribArray(normals);
        glUseProgram(0);
        
        // afficher.
        SDL_GL_SwapBuffers();
        
        // continuer.
        return 1;
    }
};


int main( int argc, char *argv[] )
{
    TP app;
    app.run();
    
    return 0;
}
