
#ifndef _TP_TEXTURE_H
#define _TP_TEXTURE_H

#include "GL/GLPlatform.h"
#include "GLResource.h"
#include "Image.h"
#include "ImageArray.h"
#include "GL/TPProgramName.h"


namespace gk {

enum 
{
    UNIT0= 0,
    UNIT1= 1,
    UNIT2= 2,
    UNIT3= 3,
    UNIT4= 4,
    UNIT5= 5,
    UNIT6= 6,
    UNIT7= 7
};

//! representation du format des pixels d'une texture
struct TextureFormat
{
    GLenum internal;
    GLenum data_format;
    GLenum data_type;
    
    TextureFormat( )
        :
        internal(GL_RGBA),
        data_format(GL_RGBA),
        data_type(GL_UNSIGNED_BYTE)
    {}
    
    TextureFormat( const GLenum _internal, const GLenum _format, const GLenum _type )
        :
        internal(_internal),
        data_format(_format),
        data_type(_type)
    {}
};

extern TextureFormat TextureRGBA;
extern TextureFormat TextureRGBA16F;
extern TextureFormat TextureRGBA32F;
extern TextureFormat TextureDepth;
extern TextureFormat TextureDepth24;
extern TextureFormat TextureDepth32;
extern TextureFormat TextureR32UI;
extern TextureFormat TextureR16UI;
extern TextureFormat TextureRG16UI;

//! utilisation interne. representation d'une texture openGL.
class GLTexture : public GLResource
{
    // non copyable
    GLTexture();
    GLTexture( const GLTexture& );
    GLTexture& operator= ( const GLTexture& );

protected:
    GLenum m_target;
    TextureFormat m_format;
    int m_width;
    int m_height;
    int m_depth;
    
public:
    //! constructeur.
    GLTexture( const GLenum target )
        :
        GLResource(),
        m_target(target),
        m_format(),
        m_width(0),
        m_height(0),
        m_depth(0)
    {
        glGenTextures(1, &m_name);
    }
    
    //! desctructeur.
    virtual ~GLTexture( ) 
    {
        glDeleteTextures(1, &m_name);
    }
    
    //! gestion objet opengl.
    int createGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }
    
    //! gestion objet opengl.
    int releaseGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }

    //! renvoie le type de texture : GL_TEXTURE_2D, etc.
    GLenum target( ) const
    {
        return m_target;
    }
    
    //! renvoie le format de la texture GL_RGBA, etc.
    const TextureFormat& format( ) const
    {
        return m_format;
    }
    
    //! renvoie la largeur de la texture.
    int width( ) const
    {
        return m_width;
    }
    
    //! renvoie la hauteur de la texture 2d ou 3d, ou le nombre de textures dans un texture1DArray.
    int height( ) const
    {
        return m_height;
    }

    //! renvoie la profondeur de la texture 3d, ou le nombre de textures dans un texture2DArray.
    int depth( ) const
    {
        return m_depth;
    }

    //! renvoie le nombre de textures dans un textureArray.
    int count( ) const
    {
        if(m_target == GL_TEXTURE_1D_ARRAY)
            return m_height;
        else if(m_target == GL_TEXTURE_2D_ARRAY)
            return m_depth;
        else if(m_target == GL_TEXTURE_CUBE_MAP)
            return 6;
        // else if(m_target == GL_TEXTURE_CUBE_MAP_ARRAY) return m_depth * 6;
        
        // texture normale.
        return 1;
    }
    
    //! renvoie une image contenant les pixels de la texture. 'level' permet de choisir quel niveau de mipmap recuperer.
    //! l'application est proprietaire de l'image renvoyee, et de sa liberation.
    Image *getImage( const int unit, const int level= 0 ) const;

    //! renvoie une image contenant les pixels de la texture. 'level' permet de choisir quel niveau de mipmap recuperer.
    //! l'application est proprietaire de l'image renvoyee, et de sa liberation.
    HDRImage *getHDRImage( const int unit, const int level= 0 ) const;
    
    //! renvoie un ensemble d'images contenant les pixels de la texture. 'level' permet de choisir quel niveau de mipmap recuperer.
    //! l'application est proprietaire des images renvoyees, et de leur liberations.
    ImageArray *getImageArray( const int unit, const int level= 0 ) const;
    
    //! renvoie un ensemble d'images contenant les pixels de la texture. 'level' permet de choisir quel niveau de mipmap recuperer.
    //! l'application est proprietaire des images renvoyees, et de leur liberations.
    HDRImageArray *getHDRImageArray( const int unit, const int level= 0 ) const;
    
    //! \todo getImageCube, cas particulier de ImageArray.
    //! \todo channels( ), renvoie le nombre de composantes des pixels de la texture. rgba= 4, rgb= 3, red= 1, depth= 1, etc.
};

class GLTexture1D : public GLTexture
{
public:
    GLTexture1D( )
        :
        GLTexture(GL_TEXTURE_1D)
    {}
    
    ~GLTexture1D( ) {}
};

class GLTexture1DArray : public GLTexture
{
public:
    GLTexture1DArray( )
        :
        GLTexture(GL_TEXTURE_1D_ARRAY)
    {}
    
    ~GLTexture1DArray( ) {}
};

class GLTexture2D : public GLTexture
{
public:
    GLTexture2D( const GLenum target )
        :
        GLTexture(target)
    {}
    
    virtual ~GLTexture2D( ) {}

    GLTexture2D( const int unit, const int w, const int h, const TextureFormat& format= TextureRGBA );
    
    GLTexture2D( const int unit, const HDRImage *image, const TextureFormat& format= TextureRGBA32F );
    
    GLTexture2D( const int unit, const Image *image, const TextureFormat& format= TextureRGBA );
};

class GLDepthTexture : public GLTexture2D
{
public:
    GLDepthTexture( )
        :
        GLTexture2D(GL_TEXTURE_2D)
    {}
    
    GLDepthTexture( const int unit, const int w, const int h, const TextureFormat& format= TextureDepth );
    
    ~GLDepthTexture( ) {}
    
    HDRImage *getHDRImage( const int unit ) const;
};

class GLTexture2DArray : public GLTexture2D
{
public:
    GLTexture2DArray( )
        :
        GLTexture2D(GL_TEXTURE_2D_ARRAY)
    {}
    
    ~GLTexture2DArray( ) {}
    
    GLTexture2DArray( const int unit, const int w, const int h, const int count, const TextureFormat& format= TextureRGBA );
    
    GLTexture2DArray( const int unit, const HDRImageArray *images, const TextureFormat& format= TextureRGBA32F );
    
    GLTexture2DArray( const int unit, const ImageArray *images, const TextureFormat& format= TextureRGBA );
};

class GLTextureCube : public GLTexture2D
{
public:
    GLTextureCube( )
        :
        GLTexture2D(GL_TEXTURE_CUBE_MAP)
    {}
    
    ~GLTextureCube( ) {}
    
    GLTextureCube( const int unit, const int w, const int h, const TextureFormat& format= TextureRGBA );
    
    GLTextureCube( const int unit, const HDRImageCube *faces, const TextureFormat& format= TextureRGBA32F );
    
    GLTextureCube( const int unit, const ImageCube *faces, const TextureFormat& format= TextureRGBA );
};

class GLDepthTextureCube : public GLTexture2D
{
public:
    GLDepthTextureCube( )
        :
        GLTexture2D(GL_TEXTURE_CUBE_MAP)
    {}
    
    ~GLDepthTextureCube( ) {}

    GLDepthTextureCube( const int unit, const int w, const int h, const TextureFormat& format= TextureDepth );
};

class GLTextureCubeArray : public GLTexture
{
public:
    GLTextureCubeArray( )
        :
        GLTexture(GL_TEXTURE_CUBE_MAP_ARRAY_ARB)
    {
        //! \todo verifier le support de gl_arb_texture_cube_map_array
    }
    
    ~GLTextureCubeArray( ) {}
};


class GLTexture3D : public GLTexture
{
public:
    GLTexture3D( )
        :
        GLTexture(GL_TEXTURE_3D)
    {}
    
    ~GLTexture3D( ) {}
};

}

#endif
