
#ifndef _GK_CAMERA_H
#define _GK_CAMERA_H

#include "Transform.h"
#include "IOFileSystem.h"

namespace gk {

//! camera generique.
class Camera
{
protected:
    Transform m_view;
    Transform m_projection;
    Transform m_viewport;

    float m_width;
    float m_height;
    
public:
    //! renvoie la position de l'origine.
    Point origin( ) const
    {
        return m_view.inverse( Point(0.f, 0.f, 0.f) );
    }
    
    //! renvoie le vecteur 'vers le haut'.
    Vector up( ) const // Y
    {
        return m_view.inverse( Vector(0.f, 1.f, 0.f) );
    }
    
    //! renvoie le vecteur 'vers la droite'.
    Vector right( ) const      // X
    {
        return m_view.inverse( Vector(1.f, 0.f, 0.f) );
    }
    
    //! renvoie le vecteur 'devant'.
    Vector forward( ) const    // Z
    {
        return m_view.inverse( Vector(0.f, 0.f, 1.f) );
    }

public:
    //! constructeur par defaut.
    Camera( )
        :
        m_view(),
        m_projection(),
        m_viewport( Viewport(100.f, 100.f) ),
        m_width(100.f),
        m_height(100.f)
    {}
    
    //! constructeur, definit la projection utilisee par la camera.
    Camera( const Transform& projection, const int width= 100, const int height= 100 )
        :
        m_view(),
        m_projection(projection),
        m_viewport( Viewport(width, height) ),
        m_width(width),
        m_height(height)
    {}
    
    //! destructeur.
    ~Camera() {}
    
    //! modifie le viewport de la camera.
    void setViewport( const int width, const int height )
    {
        m_width= width;
        m_height= height;
        m_viewport= Viewport(width, height);
    }
    
    //! modifie le viewport de la camera.
    void setViewport( int viewport[4] )
    {
        m_width= viewport[2];
        m_height= viewport[3];
        m_viewport= Viewport(viewport[2], viewport[3]);
    }
    
    //! modifie la projection de la camera.
    void setProjection( const Transform& projection )
    {
        m_projection= projection;
    }
    
    //! renvoie la transformation de projection de la camera.
    const Transform& projection( )
    {
        return m_projection;
    }

    //! renvoie la transforamtion view de la camera.
    const Transform& view( )
    {
        return m_view;
    }
    
    //! renvoie la transformation viewport.
    const Transform& viewport( )
    {
        return m_viewport;
    }
};

//! camera perspective.
class PerspectiveCamera : public gk::Camera
{
public:
    //! constructeur par defaut.
    PerspectiveCamera( )
        :
        gk::Camera()
    {}
    
    //! constructeur, parametre la projection perspective, cf gk::Perspective().
    PerspectiveCamera( const float fov, const float aspect, const float znear, const float zfar, const int width= 100, const int height= 100 )
        :
        Camera( Perspective(fov, aspect, znear, zfar), width, height)
    {}

    //! constructeur, parametre la projection perspective, cf gk::Perspective().
    PerspectiveCamera( const Transform& projection, const int width= 100, const int height= 100 )
        :
        Camera(projection, width, height)
    {}
    
    float znear( ) const
    {
        gk::Point p= m_projection.inverse( gk::Point(.0f, .0f, -1.f) );
        return -p.z;
    }
    
    float zfar( ) const
    {
        gk::Point p= m_projection.inverse( gk::Point(.0f, .0f, 1.f) );
        return -p.z;
    }
    
    //! destructeur.
    ~PerspectiveCamera( ) {}
};

//! camera first person.
class FirstPersonCamera : public PerspectiveCamera
{
protected:
    Point m_position;
    float m_rotation_x;
    float m_rotation_y;
    float m_rotation_z;
    float m_speed;
    
public:
    FirstPersonCamera( )
        :
        PerspectiveCamera(),
        m_position(),
        m_rotation_x(0.f),
        m_rotation_y(0.f),
        m_rotation_z(0.f),
        m_speed(1.f)
    {}

    //! constructeur, parametre la projection perspective, cf gk::Perspective().
    FirstPersonCamera( const float fov, const float aspect, const float znear, const float zfar, const int width= 100, const int height= 100 )
        :
        PerspectiveCamera(fov, aspect, znear, zfar, width, height),
        m_position(),
        m_rotation_x(0.f),
        m_rotation_y(0.f),
        m_rotation_z(0.f),
        m_speed(1.f)
    {}
        
    FirstPersonCamera( const Transform& projection, const int width= 100, const int height= 100 )
        :
        PerspectiveCamera(projection, width, height),
        m_position(),
        m_rotation_x(0.f),
        m_rotation_y(0.f),
        m_rotation_z(0.f),
        m_speed(1.f)        
    {}
    
    ~FirstPersonCamera( ) {}
    
    //! definit la 'vitesse' de deplacement == l'amplitude des deplacements.
    void setSpeed( const float v )
    {
        m_speed= v;
    }
    
    //! renvoie la 'vitesse' de deplacement == l'amplitude des deplacements.
    float speed( ) const
    {
        return m_speed;
    }
    
    //! deplace l'objet, vers l'avant, +Z.
    void moveForward( const float v )
    {
        m_position= m_position + forward() * v * m_speed;
    }
    
    //! deplace l'objet, vers la droite, +X.
    void moveRight( const float v )
    {
        m_position= m_position + right() * v * m_speed;
    }
    
    //! deplace l'objet, en haut, +Y.
    void moveUp( const float v )
    {
        m_position= m_position + up() * v * m_speed;
    }
    
    //! repositionne l'objet.
    void move( const Point& p )
    {
        m_position= p;
    }
    
    //! rotation gauche-droite (autour de up / Y local).
    void rotateUp( const float v )
    {
        m_rotation_y+= v;
    }
    
    //! rotation haut-bas (autour de right / X local).
    void rotateRight( const float v )
    {
        m_rotation_x+= v;
    }
    
    //! rotation autour de l'axe optique (forward / Z).
    void rotateForward( const float v )
    {
        m_rotation_z+= v;
    }
    
    //! renvoie la 'position' de la camera.
    const Point& position( ) 
    {
        return m_position;
    }
    
    //! renvoie la transformation view de la camera.
    const Transform& view( )
    {
        m_view= (Translate( Vector(m_position) ) * RotateZ(m_rotation_z) * RotateX(m_rotation_x) * RotateY(m_rotation_y)).getInverse();
        return m_view;
    }

    //! enregistre la description de la camera dans un fichier.
    int read( const std::string& filename )
    {
        std::string pref= gk::IOFileSystem::changeType(filename, ".gkcamera");
    #ifdef VERBOSE
        printf("loading camera prefs '%s'...\n", pref.c_str());
    #endif
        
        FILE *in= fopen(pref.c_str(), "rt");
        if(in == NULL)
        {
        #ifdef VERBOSE
            printf("failed.\n");
        #endif
            return -1;
        }
        
        float x, y, z;
        float speed;
        float angle1, angle2, angle3;
        if(fscanf(in, "x %f y %f z %f v %f a %f b %f c %f", 
            &x, &y, &z, 
            &speed, 
            &angle1, &angle2, &angle3) == 7)
        {
        #ifdef VERBOSE  // _DEBUG
            printf("camera %f %f %f, rotate %f %f %f, speed %f\n",
                x, y, z, angle1, angle2, angle3, speed);
        #endif
            
            m_position= Point(x, y, z);
            m_rotation_y= angle1;
            m_rotation_x= angle2;
            m_rotation_z= angle3;
            m_speed= speed;

        #ifdef VERBOSE
            printf("done.\n");
        #endif
            fclose(in);
            return 0;
        }

    #ifdef VERBOSE
        printf("failed.\n");
    #endif
        fclose(in);
        return -1;
    }
    
    //! recharge une description de camera depuis un fichier.
    int write( const std::string& filename )
    {
        std::string pref= gk::IOFileSystem::changeType(filename, ".gkcamera");
    #ifdef VERBOSE
        printf("writing camera prefs '%s'...\n", pref.c_str());
    #endif
        
        FILE *out= fopen(pref.c_str(), "wt");
        if(out == NULL)
        {
        #ifdef VERBOSE
            printf("failed.\n");
        #endif
            return -1;
        }
        
        if(fprintf(out, "x %f y %f z %f v %f a %f b %f c %f", 
            m_position.x, m_position.y, m_position.z,
            m_speed, 
            m_rotation_y, m_rotation_x, m_rotation_z) < 0)
        {
        #ifdef VERBOSE
            printf("failed.\n");
        #endif
            fclose(out);
            return -1;
        }

        fclose(out);
        return 0;
    }
};


//! camera orthographique.
class OrthographicCamera : public gk::Camera
{
public:
    //! constructeur par defaut.
    OrthographicCamera( )
        :
        gk::Camera()
    {}
    
    //! constructeur, parametre la projection orthographique, cf gk::Orthographic().
    OrthographicCamera( const float znear, const float zfar, const int width= 100, const int height= 100 )
        :
        Camera( Orthographic(znear, zfar), width, height )
    {}
    
    //! destructeur.
    ~OrthographicCamera( ) {}
};

}

#endif
