
#include "App.h"
#include "nvwidgets/nvSdlWidgets.h"

#include "ProfilerClock.h"

#include "MeshIO.h"
#include "GLTexture2DIO.h"
#include "EffectIO.h"
#include "EffectShaderManager.h"
#include "TextureManager.h"
#include "BufferManager.h"
#include "QueryManager.h"


namespace gk {

//! mesure du temps d'execution d'une operation openGL.
struct StatsGLTimerData : IStatsUserData
{
    GLTimer *start;
    GLTimer *stop;
    
    StatsGLTimerData( )
        :
        IStatsUserData( name() ),
        start(NULL),
        stop(NULL)
    {
        start= new GLTimer;
        stop= new GLTimer;
        if(start != NULL)
            start->createGLResource();
        if(stop != NULL)
            stop->createGLResource();
    }
    
    ~StatsGLTimerData( )
    {
        if(start != NULL)
            start->releaseGLResource();
        delete start;
        if(stop != NULL)
            stop->releaseGLResource();
        delete stop;
    }
    
    static const char *name( )
    {
        return "GLTimer";
    }
};

//! mesure du temps d'execution d'une operation openGL realisee dans un bloc de code c++.
class ScopedGLTimer
{
    StatsCounter *m_counter;
    StatsGLTimerData *m_timer;
    int m_last;
    
    // non copyable
    ScopedGLTimer( const ScopedGLTimer& );
    ScopedGLTimer& operator=( const ScopedGLTimer& );
    
    ScopedGLTimer( );
    
public:
    //! constructeur, nom et taille de l'historique du compteur (gk::StatsCounterIO) associe au timer.
    ScopedGLTimer( const std::string& name, const int n= 100 )
        :
        m_counter(NULL),
        m_timer(NULL),
        m_last(0)
    {
        assert(glGetError() == GL_NO_ERROR);
        
        StatsCounter *counter= StatsCounterIO::manager().find(name);
        if(counter == NULL)
        {
            counter= StatsCounterIO::manager().create(name, n);
            m_timer= new StatsGLTimerData;
            counter->attachUserData(m_timer);
        }
        else
        {
            m_timer= static_cast<StatsGLTimerData *>( counter->findUserData(StatsGLTimerData::name()) );
            if(m_timer != NULL)
            {
                // stocker le resultat de la mesure precedente : retarder le plus possible la demande de la mesure
                int time= (m_timer->stop->result64() - m_timer->start->result64()) / 1000;       // delai en micro secondes
                assert(glGetError() == GL_NO_ERROR);
                counter->push(time);
                m_last= time;
            }
        }
        
        // demarrer le timer
        if(m_timer != NULL)
            glQueryCounter(m_timer->start->name(), GL_TIMESTAMP);
        
        m_counter= counter;
        assert(glGetError() == GL_NO_ERROR);
    }
    
    //! arrete la mesure de temps, et l'enregistre dans le compteur, 
    //! ne renvoie pas la mesure, elle n'est pas encore disponible.
    int stop( )
    {
        if(m_timer == NULL)
            return -1;
        if(m_counter == NULL)
            return -1;
        
        // arreter le timer
        glQueryCounter(m_timer->stop->name(), GL_TIMESTAMP);
        assert(glGetError() == GL_NO_ERROR);
        
        // la mesure est relevee le plus tard possible, avant le debut de prochaine mesure, cf ScopedGLTimer( )
        m_counter= NULL;
        return 0;
    }
    
    int last( ) const
    {
        return m_last;
    }
    
    //! destructeur, termine la mesure et l'insere dans le compteur.
    ~ScopedGLTimer( )
    {
        stop();
    }
};

}       // namespace



struct BasicGLMesh
{
    gk::Mesh *mesh;
    gk::GLAttributeBuffer *positions;
    gk::GLAttributeBuffer *normals;
    gk::GLAttributeBuffer *texcoords;
    gk::GLIndexBuffer *indices;
    
    std::vector<gk::SubMesh> submeshes;
    std::vector<gk::MeshMaterial *> materials;
    
    gk::BBox bbox;
    
    gk::BufferLayout positions_layout;
    gk::BufferLayout normals_layout;
    gk::BufferLayout texcoords_layout;
    gk::BufferLayout indices_layout;
    
    BasicGLMesh( )
        :
        mesh(NULL),
        positions(NULL),
        normals(NULL),
        texcoords(NULL),
        indices(NULL),
        submeshes(),
        materials(),
        bbox()
    {}
    
    ~BasicGLMesh( ) {}
    
    int read( const std::string& filename )
    {
        mesh= gk::MeshIO::read(filename);
        return buildFromMesh(mesh);
    }
    
    int buildFromMesh( gk::Mesh *_mesh )
    {
        if(_mesh == NULL)
            return -1;
        mesh= _mesh;
        
        bbox= mesh->getBBox();
        
        submeshes= mesh->subMeshes();
        materials= mesh->materials();
        
        positions= gk::createAttributeBuffer(mesh->positionCount(),
            mesh->positionCount() * sizeof(float [3]), &mesh->positions().front());
        if(positions == NULL || positions->createGLResource() < 0)
            return -1;
        positions_layout= gk::BufferLayout(3, GL_FLOAT, sizeof(float [3]));

        if(mesh->normalCount() > 0)
        {
            normals= gk::createAttributeBuffer(mesh->normalCount(),
                mesh->normalCount() * sizeof(float [3]), &mesh->normals().front());
            if(normals == NULL || normals->createGLResource() < 0)
                return -1;
            
            normals_layout= gk::BufferLayout(3, GL_FLOAT, sizeof(float [3]));
        }
    
        if(mesh->texCoordCount() > 0)
        {
            texcoords= gk::createAttributeBuffer(mesh->texCoordCount(),
                mesh->texCoordCount() * sizeof(float [2]), &mesh->texCoords().front());
            if(texcoords == NULL || texcoords->createGLResource() < 0)
                return -1;
            
            texcoords_layout= gk::BufferLayout(2, GL_FLOAT, sizeof(float [2]));
        }
    
        if(mesh->indiceCount() > 0)
        {
            indices= gk::createIndexBuffer(mesh->indiceCount(),
                mesh->indiceCount() * sizeof(int), &mesh->indices().front());
            if(indices == NULL || indices->createGLResource() < 0)
                return -1;
            
            indices_layout= gk::BufferLayout(1, GL_UNSIGNED_INT, sizeof(unsigned int));
        }
        
        return 0;
    }
};


class Batch : public gk::App
{
    BasicGLMesh m_mesh;
    BasicGLMesh m_mesh_alt;
    
    gk::GLShaderProgram *m_program;
    gk::GLShaderProgram *m_program_alt;
    gk::GLTexture2D *m_diffuse_texture;
    gk::GLTexture2D *m_diffuse_texture_alt;
    gk::GLQuery *m_time_query;
    gk::GLQuery *m_triangles_query;

    unsigned int m_draw_calls_stat;
    unsigned int m_triangles_stat;
    nv::SdlUIContext m_ui;
    
    int m_draw_count;
    bool m_change_shader;
    bool m_change_buffer;
    bool m_change_texture;
    bool m_draw_0;
    
public:    
    Batch( )
        :
        gk::App(),
        m_mesh(),
        m_program(NULL),
        m_program_alt(NULL),
        m_diffuse_texture(NULL),
        m_diffuse_texture_alt(NULL),
        m_time_query(NULL),
        m_triangles_query(NULL),
        m_draw_calls_stat(0),
        m_triangles_stat(0),
        m_ui(),
        m_draw_count(100),
        m_change_shader(false),
        m_change_buffer(false),
        m_change_texture(false),
        m_draw_0(false)
    {
        gk::AppSettings settings;
        settings.setSwapControl(0);
        createWindow(1024, 768, settings);
    }
    
    ~Batch( ) {}
    
    // a redefinir pour utiliser les widgets.
    void processWindowResize( SDL_ResizeEvent& event )
    {
        m_ui.reshape(event.w, event.h);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseButtonEvent( SDL_MouseButtonEvent& event )
    {
        m_ui.processMouseButtonEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseMotionEvent( SDL_MouseMotionEvent& event )
    {
        m_ui.processMouseMotionEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processKeyboardEvent( SDL_KeyboardEvent& event )
    {
        m_ui.processKeyboardEvent(event);
    }
    
    int init( )
    {
        m_ui.init(windowWidth(), windowHeight());
        
        m_time_query= gk::createQuery(GL_TIME_ELAPSED);
        if(m_time_query == NULL || m_time_query->createGLResource() < 0)
            return -1;
        m_time_query->begin();  // pour la premiere image ...
        
        m_triangles_query= gk::createQuery(GL_PRIMITIVES_GENERATED);
        if(m_triangles_query == NULL || m_triangles_query->createGLResource() < 0)
            return -1;
        m_triangles_query->begin();
        
        if(m_mesh.read( "bigguy.obj" ) < 0)
            return -1;
        if(m_mesh_alt.read( "bigguy.obj" ) < 0)
            return -1;
        
        // construire un index buffer degenere
        {
            gk::ScopedMapIndexBuffer<unsigned int> map(m_mesh_alt.indices);
            
            int n= m_mesh_alt.mesh->indiceCount();
            const std::vector<int>& indices= m_mesh_alt.mesh->indices();
            for(int i= 0; i < n; i+= 3)
            {
                int a= indices[i];
                int b= indices[i +1];
                int c= indices[i +2];
                
                // triangle degenere
                map[i]= a;
                map[i]= a;
                map[i]= a;
            }
        }
        
        const gk::MeshMaterial *material= m_mesh.materials[m_mesh.submeshes[0].material_id];
        if(material->diffuse_texture.empty() == false)
        {
            //~ m_diffuse_texture= gk::GLTexture2DIO::read(gk::UNIT0, material->diffuse_texture);
            //~ if(m_diffuse_texture == NULL || m_diffuse_texture->createGLResource() < 0)
                //~ return -1;
            
            gk::Image *image= gk::ImageIO::read(material->diffuse_texture);
            if(image == NULL)
                return -1;
            m_diffuse_texture= gk::createTexture2D(gk::UNIT0, image);
            if(m_diffuse_texture == NULL || m_diffuse_texture->createGLResource() < 0)
                return -1;
            
            m_diffuse_texture_alt= gk::createTexture2D(gk::UNIT0, image);
            if(m_diffuse_texture_alt == NULL || m_diffuse_texture_alt->createGLResource() < 0)
                return -1;
        }
        
        gk::Effect *effect= gk::EffectIO::read( "batch.gkfx" );
        if(effect == NULL)
            return -1;
        
        m_program= gk::EffectShaderManager(effect).createShaderProgram("batch");
        if(m_program == NULL || m_program->createGLResource() < 0)
            return -1;
        
        m_program_alt= gk::EffectShaderManager(effect).createShaderProgram("batch");
        if(m_program_alt == NULL || m_program_alt->createGLResource() < 0)
            return -1;
        
        return 0;
    }
    
    int quit( )
    {
        return 0;
    }
    
    void present( )
    {
        static GLint64 last_frame= 0;
        GLint64 frame;
        glGetInteger64v(GL_TIMESTAMP, &frame);
        int frame_time= (frame - last_frame) / 1000;

        m_time_query->end();
        long long int gpu_time= m_time_query->result64();
        
        m_triangles_query->end();
        m_triangles_stat= m_triangles_query->result();
        
        char tmp[1024];
        
        m_ui.begin();
        m_ui.beginGroup(nv::GroupFlags_GrowDownFromLeft);
            sprintf(tmp, "% 6d draw calls, % 8d triangles, cpu % 4dms % 4dus / % 3dfps,  gpu % 4dms % 4dus", 
                m_draw_calls_stat, m_triangles_stat,
                frame_time / 1000, frame_time % 1000, 1000000 / frame_time,
                (int) (gpu_time / 1000000), (int) ((gpu_time / 1000) % 1000));
            m_ui.doLabel(nv::Rect(), tmp);
        
            float count= m_draw_count;
            if(m_ui.doHorizontalSlider(nv::Rect(0, 0, 200, 0), 10.f, 1000.f, &count))
                m_draw_count= count;
            
            sprintf(tmp, "%d draw calls/s, %d%% cpu / 60fps", 
                (int) ((float) m_draw_calls_stat / (float) frame_time * 1000000.f),
                frame_time * 100 / (1000000 / 60));
            m_ui.doLabel(nv::Rect(), tmp);
            
            const std::string& summary= gk::StatsCounterIO::manager().getSummaryString();
            m_ui.doLabel(nv::Rect(), summary.c_str());

            m_ui.doButton(nv::Rect(), "change shader", &m_change_shader);
            m_ui.doButton(nv::Rect(), "change buffer", &m_change_buffer);
            m_ui.doButton(nv::Rect(), "change texture", &m_change_texture);
            m_ui.doButton(nv::Rect(), "draw degenerate triangles", &m_draw_0);
        m_ui.endGroup();
        m_ui.end();
        
        //~ if(m_change_shader != 0)
            //~ // les buffers sont re-actives pour un changement de shader
            //~ m_change_buffer= 1;
        
        SDL_GL_SwapBuffers();
        
        glGetInteger64v(GL_TIMESTAMP, &last_frame);
        m_time_query->begin();
        m_triangles_query->begin();
        
        m_draw_calls_stat= 0;
        m_triangles_stat= 0;
    }
    
    int draw( )
    {
        assert(glGetError() == GL_NO_ERROR);
        
        if(key(SDLK_ESCAPE))
            Close();
        
        glViewport(0, 0, windowWidth(), windowHeight());
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // transformations
        gk::Transform model;
        gk::Transform view= gk::Translate( gk::Vector(0.f, 0.f, -50.f) );
        gk::Transform perspective= gk::Perspective(50.f, 1.f, 1.f, 1000.f);
        gk::Transform mvp= perspective * view * model;
        
        {
            gk::ScopedGLTimer wait_timer("draw", 10000);
            
            if(m_mesh.indices != NULL)
            {
                GLint position= 0;      // declare dans le shader, cf layout(location= xxx) in vec3 position;
                GLint normal= 1;
                GLint texcoord= 2;
                
                if(m_change_shader == 0)
                {
                    // configurer le shader utilise pour tous les affichages
                    //~ gk::ScopedGLTimer shader_timer("shader0");
                    
                    glUseProgram(m_program->name());
                    glUniform1i(m_program->sampler("diffuse_texture"), 0);
                    glUniformMatrix4fv(m_program->uniform("mvpMatrix"), 1, GL_TRUE, mvp.matrix());
                    //~ position= m_program->attribute("position").location();
                    //~ texcoord= m_program->attribute("texcoord").location();
                    //~ normal= m_program->attribute("normal").location();
                    assert(position ==  m_program->attribute("position").location());
                    assert(normal ==  m_program->attribute("normal").location());
                    assert(texcoord ==  m_program->attribute("texcoord").location());
                }
                
                if(m_change_buffer == 0)
                {
                    // configurer les buffers utilises pour tous les affichages
                    //~ gk::ScopedGLTimer buffer_timer("buffer0");
                    
                    assert(position != -1);
                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh.positions->name());
                    glVertexAttribPointer(position, 
                        m_mesh.positions_layout.size,  m_mesh.positions_layout.type, GL_FALSE, 
                        m_mesh.positions_layout.stride, m_mesh.positions_layout.getOffset());
                    glEnableVertexAttribArray(position);
                    
                    if(m_mesh.texcoords != NULL && texcoord != -1)
                    {
                        glBindBuffer(GL_ARRAY_BUFFER, m_mesh.texcoords->name());
                        glVertexAttribPointer(texcoord, 
                            m_mesh.texcoords_layout.size,  m_mesh.texcoords_layout.type, GL_FALSE, 
                            m_mesh.texcoords_layout.stride, m_mesh.texcoords_layout.getOffset());
                        glEnableVertexAttribArray(texcoord);
                    }
                    
                    if(m_mesh.normals != NULL && normal != -1)
                    {
                        glBindBuffer(GL_ARRAY_BUFFER, m_mesh.normals->name());
                        glVertexAttribPointer(normal, 
                            m_mesh.normals_layout.size,  m_mesh.normals_layout.type, GL_FALSE, 
                            m_mesh.normals_layout.stride, m_mesh.normals_layout.getOffset());
                        glEnableVertexAttribArray(normal);
                    }
                    
                    if(m_draw_0 != 0)
                        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_mesh_alt.indices->name());
                    else
                        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_mesh.indices->name());                    
                }
                
                if(m_change_texture == 0)
                {
                    // configurer les textures utilisees pour tous les affichages
                    glActiveTexture(GL_TEXTURE0);
                    glBindTexture(m_diffuse_texture->target(), m_diffuse_texture->name());
                }
                
                for(int draw= 0; draw < m_draw_count; draw++, m_draw_calls_stat++)
                {
                    //~ glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
                    
                    int n= m_mesh.submeshes.size();
                    for(int i= 0; i < n; i++)
                    {
                        if(m_change_shader != 0)
                        {
                            // reconfigurer le shader, si necessaire, pour chaque affichage
                            //~ gk::ScopedGLTimer buffer_timer("shader");
                            if((m_draw_calls_stat % 2) == 0)    // alterne entre 2 shaders (meme code, mais objets opengl differents)
                            {
                                glUseProgram(m_program->name());
                                
                                glUniformMatrix4fv(m_program->uniform("mvpMatrix"), 1, GL_TRUE, mvp.matrix());
                                glUniform1i(m_program->sampler("diffuse_texture"), 0);
                                //~ position= m_program->attribute("position").location();
                                //~ texcoord= m_program->attribute("texcoord").location();
                                //~ normal= m_program->attribute("normal").location();
                                assert(position ==  m_program->attribute("position").location());
                                assert(normal ==  m_program->attribute("normal").location());
                                assert(texcoord ==  m_program->attribute("texcoord").location());
                            }
                            else
                            {
                                glUseProgram(m_program_alt->name());
                                
                                glUniformMatrix4fv(m_program_alt->uniform("mvpMatrix"), 1, GL_TRUE, mvp.matrix());
                                glUniform1i(m_program_alt->sampler("diffuse_texture"), 0);
                                //~ position= m_program_alt->attribute("position").location();
                                //~ texcoord= m_program_alt->attribute("texcoord").location();
                                //~ normal= m_program_alt->attribute("normal").location();
                            }
                        }
                        
                        // reconfigurer les buffers pour chaque affichage
                        if(m_change_buffer != 0)
                        {
                            //~ gk::ScopedGLTimer buffer_timer("buffer");
                            
                            if(m_draw_0 != 0)
                                glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_mesh_alt.indices->name());
                            else
                                glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_mesh.indices->name());
                            
                            assert(position != -1);
                            if((m_draw_calls_stat % 2) == 0)                            
                                glBindBuffer(GL_ARRAY_BUFFER, m_mesh.positions->name());
                            else
                                glBindBuffer(GL_ARRAY_BUFFER, m_mesh_alt.positions->name());
                            glVertexAttribPointer(position, 
                                m_mesh.positions_layout.size,  m_mesh.positions_layout.type, GL_FALSE, 
                                m_mesh.positions_layout.stride, m_mesh.positions_layout.getOffset());
                            glEnableVertexAttribArray(position);
                            
                            if(m_mesh.texcoords != NULL && texcoord != -1)
                            {
                                if((m_draw_calls_stat % 2) == 0)                            
                                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh.texcoords->name());
                                else
                                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh_alt.texcoords->name());
                                glVertexAttribPointer(texcoord, 
                                    m_mesh.texcoords_layout.size,  m_mesh.texcoords_layout.type, GL_FALSE, 
                                    m_mesh.texcoords_layout.stride, m_mesh.texcoords_layout.getOffset());
                                glEnableVertexAttribArray(texcoord);
                            }
                            
                            if(m_mesh.normals != NULL && normal != -1)
                            {
                                if((m_draw_calls_stat % 2) == 0)                            
                                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh.normals->name());
                                else
                                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh_alt.normals->name());
                                glVertexAttribPointer(normal, 
                                    m_mesh.normals_layout.size,  m_mesh.normals_layout.type, GL_FALSE, 
                                    m_mesh.normals_layout.stride, m_mesh.normals_layout.getOffset());
                                glEnableVertexAttribArray(normal);
                            }
                        }
                    
                        // reconfigurer les textures pour chaque affichage
                        if(m_change_texture != 0)
                        {
                            if((m_draw_calls_stat % 2) == 0)
                            {
                                glActiveTexture(GL_TEXTURE0);
                                glBindTexture(m_diffuse_texture->target(), m_diffuse_texture->name());
                            }
                            else
                            {
                                glActiveTexture(GL_TEXTURE0);
                                glBindTexture(m_diffuse_texture_alt->target(), m_diffuse_texture_alt->name());
                            }
                        }

                        // draw !!
                        const gk::SubMesh& sub= m_mesh.submeshes[i];
                        glDrawElements(GL_TRIANGLES, sub.end - sub.begin, 
                            m_mesh.indices_layout.type, m_mesh.indices_layout.getOffset(sub.begin));
                        
                        // nettoyage
                        if(m_change_texture != 0)
                        {
                            if((m_draw_calls_stat % 2) == 0)
                            {
                                glActiveTexture(GL_TEXTURE0);
                                glBindTexture(m_diffuse_texture->target(), 0);
                            }
                            else
                            {
                                glActiveTexture(GL_TEXTURE0);
                                glBindTexture(m_diffuse_texture_alt->target(), 0);
                            }
                        }
                        
                        if(m_change_buffer != 0)
                        {
                            //~ glBindBuffer(GL_ARRAY_BUFFER, 0);
                            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
                            
                            if(position != -1)
                                glDisableVertexAttribArray(position);
                            if(texcoord != -1)
                                glDisableVertexAttribArray(texcoord);
                            if(normal != -1)
                                glDisableVertexAttribArray(normal);
                        }
                        
                        if(m_change_shader != 0)
                            glUseProgram(0);
                    }
                }
            }
        }       
        
        present();
        return 1;
    }
};


int main( int argc, char **argv )
{
    Batch app;
    app.run();
    
    return 0;
}
