
#include <cstring>

#include "GL/GLPlatform.h"
#include "IOFileSystem.h"
#include "ShaderManager.h"
#include "EffectIO.h"
#include "EffectShaderManager.h"
#include "App.h"


class GLSLCC : public gk::App
{
    int m_argc;
    char **m_argv;
    
    int shader_type( char *fname, GLenum& type )
    {
        char *ext;
        
        if(fname == NULL)
            return -1;
        
        ext= strrchr(fname, '.');
        if(ext == NULL)
            return -1;
        
        if(strchr(ext, 'v') != NULL || strchr(ext, 'V') != NULL)
        {
            type= GL_VERTEX_SHADER;
            return 0;
        }
        
        if(strchr(ext, 'f') != NULL || strchr(ext, 'F') != NULL)
        {
            type= GL_FRAGMENT_SHADER;
            return 0;
        }
        
    #if defined GK_OPENGL3 || defined GK_OPENGL4
        if(strchr(ext, 'g') != NULL || strchr(ext, 'G') != NULL)
        {
            type= GL_GEOMETRY_SHADER;
            return 0;
        }
    #endif
        
    #ifdef GK_OPENGL4
        if(strchr(ext, 'c') != NULL || strchr(ext, 'C') != NULL)
        {
            type= GL_TESS_CONTROL_SHADER;
            return 0;
        }
    #endif
        
    #ifdef GK_OPENGL4
        if(strchr(ext, 'e') != NULL || strchr(ext, 'E') != NULL)
        {
            type= GL_TESS_EVALUATION_SHADER;
            return 0;
        }
    #endif
        
        return -1;
    }
    
public:
    GLSLCC( int argc, char **argv, const int w= 1024, const int h= 768 )
        :
        gk::App(w, h),
        m_argc(argc),
        m_argv(argv)
    {}
    
    ~GLSLCC( ) {}
    
    int init( )
    {
        int code= 0;
        
        if(m_argc > 1 && strcmp(m_argv[1], "-c") == 0)
        {
            // compiler separement tous les shaders de la ligne de commande
            for(int i= 2; i < m_argc; i++)
            {
                GLenum type;
                if(shader_type(m_argv[i], type) < 0)
                {
                    printf("  %s: unknown shader type.\n", m_argv[i]);
                    return -1;
                }
                
                switch(type)
                {
                    case GL_VERTEX_SHADER:
                    case GL_FRAGMENT_SHADER:
            #if defined GK_OPENGL3 || defined GK_OPENGL4
                    case GL_GEOMETRY_SHADER:
            #endif
            #ifdef GK_OPENGL4
                case GL_TESS_CONTROL_SHADER:
                case GL_TESS_EVALUATION_SHADER:
            #endif
                        break;
                    
                    default:
                        // type de shader non reconnu
                        return -1;
                }
                
                gk::GLShaderObject *shader= gk::GLManager<gk::GLShaderObject>::manager().insert(
                        new gk::GLShaderObject(type));
                
                gk::TextFile *source= gk::TextFileIO::read(m_argv[i]);
                if(source == NULL)
                {
                    printf("error reading source '%s'.\n", m_argv[i]);
                    return -1;
                }
                
                shader->pushSource(source);
                code= shader->createGLResource();
            }
        }
        
        else if(m_argc > 1 && gk::IOFileSystem::isType(m_argv[1], ".gkfx") == true)
        {
            // charger l'effet et compiler le programe
            gk::Effect *effect= gk::EffectIO::read(m_argv[1]);
            if(effect == NULL)
                return -1;
            
            if(m_argc < 3)
            {
                printf("usage: %s %s program_key\n", m_argv[0], m_argv[1]);
                return -1;
            }
            
            gk::TextFile *program_key= effect->find(m_argv[2]);
            if(program_key == gk::TextFile::notFound())
            {
                printf("program key '%s' not found.\n", m_argv[2]);
                return -1;
            }
            
            const char *keys[]= { 
                "vertex", "fragment", 
                "geometry", 
                "control", "evaluation" 
            };
            GLenum types[]= { 
                GL_VERTEX_SHADER, GL_FRAGMENT_SHADER, 
                GL_GEOMETRY_SHADER, 
                GL_TESS_CONTROL_SHADER, GL_TESS_EVALUATION_SHADER
            };
            
            printf("shader program '%s'...\n", m_argv[2]);
            gk::GLShaderProgram *program= new gk::GLShaderProgram();
            
            for(int i= 0; i < 5; i++)
            {
                const gk::TextValue& section= program_key->find(keys[i]);
                if(section == gk::TextValue::notFound())
                    continue;
                
                gk::TextFile *source= effect->find(section.asName());
                if(source == gk::TextFile::notFound())
                {
                    printf("section '%s' not found...\n", keys[i]);
                    continue;
                }
                
                gk::GLShaderObject *shader= gk::GLManager<gk::GLShaderObject>::manager().insert( 
                    new gk::GLShaderObject(types[i]) );
                shader->pushSource(source);
                printf("  %s shader '%s'...\n", keys[i], section.asName().c_str());
                if(shader->createGLResource() < 0)
                {
                    code= -1;
                    continue;
                }
                
                program->attachShader(shader);
            }
            
            //~ printf("linking shader program...\n");
            printf("done.\n");
            code= program->createGLResource();
            program->releaseGLResource();
            delete program;
        }
        
        else if(m_argc > 1)
        {
            // compiler les shaders et les linker
            gk::GLShaderProgram *program= new gk::GLShaderProgram();

            // identifie le vertex shader et le fragment shader
            for(int i= 1; i < m_argc; i++)
            {
                GLenum type;
                if(shader_type(m_argv[i], type) < 0)
                {
                    printf("  %s: unknown shader type.\n", m_argv[i]);
                    return -1;
                }
                
                printf("  %s (type 0x%x)\n", m_argv[i], type);
                
                switch(type)
                {
                    case GL_VERTEX_SHADER:
                    case GL_FRAGMENT_SHADER:
            #if defined GK_OPENGL3 || defined GK_OPENGL4
                    case GL_GEOMETRY_SHADER:
            #endif
            #ifdef GK_OPENGL4
                case GL_TESS_CONTROL_SHADER:
                case GL_TESS_EVALUATION_SHADER:
            #endif
                        break;
                    
                    default:
                        // type de shader non reconnu
                        return -1;
                }
                
                gk::GLShaderObject *shader= gk::GLManager<gk::GLShaderObject>::manager().insert(
                        new gk::GLShaderObject(type));
                
                gk::TextFile *source= gk::TextFileIO::read(m_argv[i]);
                if(source == NULL)
                {
                    printf("error reading source '%s'.\n", m_argv[i]);
                    return -1;
                }
                
                shader->pushSource(source);
                program->attachShader(shader);
            }
            
            code= program->createGLResource();
            program->releaseGLResource();
            delete program;
        }
        
        return code;
    }
    
    int quit( )
    {
        return 0;
    }
    
    int draw( )
    {
        // rien a faire
        return 0;
    }
};


int main( int argc, char *argv[] )
{
    if(argc < 2)
    {
        printf("usage: %s [-c] shader.*v* [shader.*c*] [shader.*e*] [shader.*g*] shader.*f*\n", argv[0]);
        printf("usage: %s effect.gkfx program_section\n", argv[0]);
        return 0;
    }

    GLSLCC app(argc, argv);
    int code= app.run();
    if(code < 0)
        printf("failed.\n");
    else
        printf("done.\n");
    return (code < 0) ? 1 : 0;
}
