
#ifndef _RT_TRIANGLE_H
#define _RT_TRIANGLE_H

#include "Triangle.h"
#include "Geometry.h"

namespace gk {

//! representation d'un triangle pour l'intersecteur rayon / triangle.
struct RTTriangle
{
    Point a;
    Vector ab, ac;
    
    //! constructeur par defaut.
    RTTriangle( ) {}
    
    //! construit un triangle connaissant ses 3 sommets.
    RTTriangle( const Point& _a, const Point& _b, const Point& _c )
        :
        a(_a),  ab(_a, _b), ac(_a, _c)
    {}
    
    //! construit un triangle a partir d'un triangle 'geometrique'.
    RTTriangle( const Triangle& triangle)
        :
        a(triangle.a),  ab(triangle.a, triangle.b), ac(triangle.a, triangle.c)
    {}
    
    //! destructeur.
    ~RTTriangle( ) {}
    
    float getArea( ) const
    {
        return .5f * Cross(ab, ac).Length();
    }
    
    BBox getBBox( ) const
    {
        BBox bbox;
        
        bbox.Union(a);
        bbox.Union(a + ab);
        bbox.Union(a + ac);
        
        return bbox;
    }
    
    //! calcule la normale geometrique du triangle.
    void getTriangleNormal( Normal &n ) const
    {
        n= Normal( Normalize(Cross(ab, ac)) ) ;
    }

    Normal getTriangleNormal( ) const
    {
        return Normal( Normalize( Cross(ab, ac) ));
    }
    
    //! intersection avec un rayon.
    //! renvoie faux s'il n'y a pas d'intersection, une intersection peut exister mais peut ne pas se trouver dans [tmin tmax] du rayon.
    //! renvoie vrai + les coordonnees barycentriques du point d'intersection + sa position le long du rayon.
    //! convention barycentrique : t(u, v)= (1 - u - v) * a + u * b + v * c
    /*! "Fast, Minimum Storage Ray/Triangle Intersection"
        cf http://www.acm.org/jgt/papers/MollerTrumbore97/
    */
    bool Intersect( const Ray &ray, const float htmax, 
        float &rt, float &ru, float&rv ) const
    {
        /* begin calculating determinant - also used to calculate U parameter */
        const Vector pvec= Cross(ray.d, ac);
        
        /* if determinant is near zero, ray lies in plane of triangle */
        const float det= Dot(ab, pvec);
        if (det > -EPSILON && det < EPSILON)
            return false;
        
        const float inv_det= 1.0f / det;
        
        /* calculate distance from vert0 to ray origin */
        const Vector tvec(a, ray.o);
        
        /* calculate U parameter and test bounds */
        const float u= Dot(tvec, pvec) * inv_det;
        if(u < 0.0f || u > 1.0f)
            return false;
        
        /* prepare to test V parameter */
        const Vector qvec= Cross(tvec, ab);
        
        /* calculate V parameter and test bounds */
        const float v= Dot(ray.d, qvec) * inv_det;
        if(v < 0.0f || u + v > 1.0f)
            return false;
        
        /* calculate t, ray intersects triangle */
        rt= Dot(ac, qvec) * inv_det;
        ru= u;
        rv= v;
        
        // ne renvoie vrai que si l'intersection est valide (comprise entre tmin et tmax du rayon)
        return (rt < htmax && rt > ray.tmin);
    }
};

}

#endif
