
#ifndef _TP_FRAMEBUFFER_H
#define _TP_FRAMEBUFFER_H

#include "GL/GLPlatform.h"
#include "GLResource.h"
#include "GL/TPTexture.h"


namespace  gk {

//! identification des textures associees au framebuffer.
enum 
{
    COLOR0= 0,
    COLOR1= 1,
    COLOR2= 2,
    COLOR3= 3,
    COLOR4= 4,
    COLOR5= 5,
    COLOR6= 6,
    COLOR7= 7,
    LAST_COLOR= 8,
    DEPTH= 9,
    LAST= 10
};

//! creation des textures par le framebuffer : identification des textures a initialiser.
enum
{
    COLOR0_BIT= 1<<COLOR0,
    COLOR1_BIT= 1<<COLOR1,
    COLOR2_BIT= 1<<COLOR2,
    COLOR3_BIT= 1<<COLOR3,
    COLOR4_BIT= 1<<COLOR4,
    COLOR5_BIT= 1<<COLOR5,
    COLOR6_BIT= 1<<COLOR6,
    COLOR7_BIT= 1<<COLOR7,
    DEPTH_BIT= 1<<DEPTH
};

//! identification des drawbuffers.
enum
{
    DRAW0= 0,
    DRAW1= 1,
    DRAW2= 2,
    DRAW3= 3,
    DRAW4= 4,
    DRAW5= 5,
    DRAW6= 6,
    DRAW7= 7,
};

    
//! utilisation interne. representation d'un framebuffer.
class GLRendertarget : public GLResource
{
    // non copyable
    GLRendertarget( const GLRendertarget& );
    GLRendertarget& operator= ( const GLRendertarget& );

protected:
    std::vector<GLTexture *> m_textures;
    std::vector<GLenum> m_draw_buffers;
    int m_width;
    int m_height;
    int m_color_mask;
    int m_depth_mask;

    template< typename Texture >
    int attach_buffer( const unsigned int buffer, Texture *texture )
    {
        if(m_name == 0)
            return -1;
        if(buffer >= LAST)
            return -1;
        if(texture == NULL)
            return -1;

    #ifdef VERBOSE_DEBUG
        if(buffer < LAST_COLOR)
            printf("GLFramebuffer::attachTexture( ): buffer 0x%x, texture %d.\n", 
                GL_COLOR_ATTACHMENT0 + buffer, texture->name());
        else if(buffer == DEPTH)
            printf("GLFramebuffer::attachTexture( ): buffer 0x%x, texture %d\n.", 
                GL_DEPTH_ATTACHMENT, texture->name());
    #endif
        
        // verifie les dimensions de la texture
        if(m_width > 0 && texture->width() != m_width)
            return -1;
        m_width= texture->width();
        if(m_height > 0 && texture->height() != m_height)
            return -1;
        m_height= texture->height();
        
        // attache la texture
        m_textures[buffer]= static_cast<GLTexture *>(texture);
        if(buffer == GL_DEPTH_ATTACHMENT)
        {
            m_depth_mask= 1;
            return GL_DEPTH_ATTACHMENT;
        }

        // recompte le nombre de draw buffers couleur
        m_color_mask= 0;
        int draw_buffer= -1;
        m_draw_buffers.clear();
        for(int i= 0; i < LAST_COLOR; i++)
            if(m_textures[i]  != NULL)
            {
                if(i == buffer)
                    draw_buffer= GL_COLOR_ATTACHMENT0 + (unsigned int) m_draw_buffers.size();
                m_draw_buffers.push_back(GL_COLOR_ATTACHMENT0 + (unsigned int) m_draw_buffers.size());
                m_color_mask= 1;
            }
        
        // renvoyer l'indice du draw buffer associe a la texture
        return draw_buffer;
    }

public:
    //! constructeur par defaut.
    GLRendertarget( );

    //! associe une texture existante au framebuffer. buffer = gk::COLOR0, etc. ou gk::DEPTH. 
    //! toutes les textures doivent avoir les memes dimensions.\n
    int attachTexture( const GLenum target, const unsigned int buffer, GLTexture *texture, const int level= 0 );
    int attachTexture( const GLenum target, const unsigned int buffer, GLDepthTexture *texture, const int level= 0 );
    int attachTexture( const GLenum target, const unsigned int buffer, GLTexture2DArray *texture, const int layer, const int level= 0 );
    int attachTexture( const GLenum target, const unsigned int buffer, GLTextureCube *texture, const  GLenum face, const int level= 0 );

    //! constructeur : cree les textures et les associe au framebuffer. buffer_bits= gk::COLOR0_BIT | gk::DEPTH_BIT, par exemple.
    /*! exemple d'utilisation :
    \code
    gk::GLFramebuffer framebuffer(512, 512, gk::COLOR0_BIT | gk::DEPTH_BIT);    // cree un framebuffer + une texture couleur + une texture profondeur
    gk::GLTexture *color= framebuffer.texture(gk::COLOR0);  // pour recuperer la texture associee au framebuffer.
    \endcode
    */
    GLRendertarget( const GLenum target, const int w, const int h, const unsigned int buffer_bits, 
        const TextureFormat& color_format= TextureRGBA, const TextureFormat& depth_format= TextureDepth );
    
    //! destructeur.
    virtual ~GLRendertarget(  );

    //! creation de l'objet opengl.
    int createGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }
    
    //! destruction de l'objet opengl.
    int releaseGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }
    
    //! verifie la configuration du framebuffer.
    int validate( const GLenum target );
    
    //! renvoie la largeur du framebuffer. renvoie 0, si la largeur du framebuffer n'est pas definie.
    int width( ) const
    {
        return m_width;
    }
    
    //! renvoie la hauteur du framebuffer. renvoie 0, si la hauteur du framebuffer n'est pas definie.
    int height( ) const
    {
        return m_height;
    }
    
    //! renvoie la texture associee a buffer (cf gk::COLOR0, gk::COLOR1, gk::DEPTH, etc.)
    GLTexture *texture( const unsigned int buffer );
    
    //! renvoie le zbuffer
    GLDepthTexture *zbuffer( );
    
    //! renvoie les draw buffers opengl / textures attachees (cf. glDrawBuffers() et GL_COLOR_ATTACHMENT0, etc.).
    /*!
    \code
    const std::vector<GLenum>& buffers= framebuffer->drawBuffers();
    glDrawbuffers((GLsizei) buffers.size(), &buffers.front());
    \endcode
    */
    const std::vector<GLenum>& drawBuffers( )
    {
        if(m_draw_buffers.empty())
            // le frame buffer ne contient qu'un depth attachement.
            m_draw_buffers.push_back(GL_NONE);
        
        return m_draw_buffers;
    }
    
    //! renvoie vrai si des textures couleurs sont attachees.
    //! cf. glColorMask( ) pour bloquer l'ecriture dans le framebuffer, si necessaire.
    bool colorMask( ) const
    {
        return m_color_mask;
    }
    
    //! renvoie vrai si un zbuffer est attache.
    //! cf. glDepthMask( ) pour bloquer l'ecriture dans le framebuffer, si necessaire.
    bool depthMask( ) const
    {
        return m_depth_mask;
    }
};

//! representation d'un (draw) framebuffer.
class GLFramebuffer : public GLRendertarget
{
public:
    //! constructeur.
    GLFramebuffer(  )
        :
        GLRendertarget( )
    {}
    
    //! constructeur, cf GLRendertarget().
    GLFramebuffer( const int w, const int h, const unsigned int buffer_bits, 
        const TextureFormat& color_format= TextureRGBA, const TextureFormat& depth_format= TextureDepth )
        :
        GLRendertarget(GL_DRAW_FRAMEBUFFER, w, h, buffer_bits, color_format, depth_format)
    {}

    //! destructeur.
    ~GLFramebuffer(  ) {}

    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const unsigned int buffer, GLTexture *texture, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, level);
    }
    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const GLenum target, const unsigned int buffer, GLDepthTexture *texture, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, level);
    }
    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const GLenum target, const unsigned int buffer, GLTexture2DArray *texture, const int layer, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, layer, level);
    }
    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const GLenum target, const unsigned int buffer, GLTextureCube *texture, const  GLenum face, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, face, level);
    }

    //! valide la configuration du framebuffer.
    int validate( )
    {
        return GLRendertarget::validate(GL_DRAW_FRAMEBUFFER);
    }
};

//! representation d'un (read) framebuffer.
class GLReadFramebuffer : public GLRendertarget
{
public:
    //! constructeur.
    GLReadFramebuffer( )
        :
        GLRendertarget()
    {}
    
    //! constructeur. cf GLRendertarget::GLRendertarget( ).
    GLReadFramebuffer( const int w, const int h, const unsigned int buffer_bits, 
        const TextureFormat& color_format= TextureRGBA, const TextureFormat& depth_format= TextureDepth )
        :
        GLRendertarget(GL_READ_FRAMEBUFFER, w, h, buffer_bits, color_format, depth_format)
    {}

    //! destructeur.
    ~GLReadFramebuffer(  ) {}

    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const unsigned int buffer, GLTexture *texture, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, level);
    }
    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const GLenum target, const unsigned int buffer, GLDepthTexture *texture, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, level);
    }
    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const GLenum target, const unsigned int buffer, GLTexture2DArray *texture, const int layer, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, layer, level);
    }
    //! associe une texture au framebuffer. cf GLRendertarget::attachTexture().
    int attachTexture( const GLenum target, const unsigned int buffer, GLTextureCube *texture, const  GLenum face, const int level= 0 )
    {
        return GLRendertarget::attachTexture(GL_DRAW_FRAMEBUFFER, buffer, texture, face, level);
    }
    
    //! verifie la configuration du framebuffer.
    int validate( )
    {
        return GLRendertarget::validate(GL_READ_FRAMEBUFFER);
    }
};

}

#endif
