
#include <cstdio>

#include "GLManager.h"
#include "GL/TPFramebuffer.h"


namespace gk {

GLRendertarget::GLRendertarget( )
    :
    GLResource(),
    m_textures(LAST, NULL),
    m_draw_buffers(),
    m_width(0),
    m_height(0),
    m_color_mask(0),
    m_depth_mask(0)
{
    glGenFramebuffers(1, &m_name);
}

GLRendertarget::GLRendertarget( const GLenum target, const int w, const int h, const unsigned int buffer_bits,
    const TextureFormat& color_format, const TextureFormat& depth_format )
    :
    GLResource(),
    m_textures(LAST, NULL),
    m_draw_buffers(),
    m_width(w),
    m_height(h),
    m_color_mask(0),
    m_depth_mask(0)
{
    glGenFramebuffers(1, &m_name);
    glBindFramebuffer(target, m_name);
    assert(glGetError() == GL_NO_ERROR);
    
    // parcourir tous les bits : enumerer les textures couleurs a creer / initialiser.
    unsigned int count= 0;
    for(unsigned int bit= 0; bit < LAST_COLOR; bit++)
    {
        if((buffer_bits & (1<<bit)) == 0)
            continue;
        
        GLTexture2D *color= new GLTexture2D(gk::UNIT0, w, h, color_format);   // creation arbitraire sur texture unit 0
        if(color == NULL || color->createGLResource() < 0)
            return;
        
        glFramebufferTexture2D(target, 
            GL_COLOR_ATTACHMENT0 + count, color->target(), color->name(), 0);
        assert(glGetError() == GL_NO_ERROR);
        
        m_draw_buffers.push_back(GL_COLOR_ATTACHMENT0 + count);
        m_textures[bit]= GLManager<GLTexture2D>::manager().insert(color);
        m_color_mask= 1;
        count++;
    }
    
    // cree un zbuffer, si necessaire.
    if(buffer_bits & DEPTH_BIT)
    {
        GLDepthTexture *depth= new GLDepthTexture(gk::UNIT0, w, h, depth_format);     // creation arbitraire sur texture unit 0
        if(depth == NULL || depth->createGLResource() < 0)
            return;
        
        glFramebufferTexture2D(target, 
            GL_DEPTH_ATTACHMENT, depth->target(), depth->name(), 0);
        assert(glGetError() == GL_NO_ERROR);

        m_textures[DEPTH]= GLManager<GLDepthTexture>::manager().insert(depth);
        m_depth_mask= 1;
    }
    
    // verifie la configuration du frmaebuffer.
    GLenum status= (GLenum) glCheckFramebufferStatus(target);
    if(status != GL_FRAMEBUFFER_COMPLETE)
        printf("GLFramebuffer: error.\n");
    
    // desactive le nouveau framebuffer.
    glBindFramebuffer(target, 0);
}

GLRendertarget::~GLRendertarget(  )
{
    if(m_name != 0)
        glDeleteFramebuffers(1, &m_name);
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLTexture *texture, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;
    
    glFramebufferTexture2D(target, 
        draw_buffer, texture->target(), texture->name(), level);
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLDepthTexture *texture, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;

    glFramebufferTexture2D(target, 
        draw_buffer, texture->target(), texture->name(), level);
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLTexture2DArray *texture, const int layer, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;
    
    glFramebufferTextureLayer(target, 
        draw_buffer, texture->name(), level, layer);
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLTextureCube *texture, const GLenum face, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;
    
    glFramebufferTexture2D(target, 
        draw_buffer, face, texture->name(), level);
    assert(glGetError() == GL_NO_ERROR);
    return 0;    
}

GLTexture *GLRendertarget::texture( const unsigned int buffer )
{
    if(m_name == 0)
        return NULL;
    if(buffer >= LAST)
        return NULL;
    
    return m_textures[buffer];
}

GLDepthTexture *GLRendertarget::zbuffer( )
{
    if(m_name == 0)
        return NULL;
    return static_cast<GLDepthTexture *>(m_textures[DEPTH]);
}

int GLRendertarget::validate( const GLenum target )
{
    assert(glGetError() == GL_NO_ERROR);

    glBindFramebuffer(target, m_name);
    GLenum status= (GLenum) glCheckFramebufferStatus(target);
    if(status != GL_FRAMEBUFFER_COMPLETE)
    {
        printf("GLFramebuffer::validate( ) error: ");
        switch(status)
        {
            case GL_FRAMEBUFFER_UNDEFINED:
                printf("  framebuffer undefined");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
                printf("  framebuffer incomplete attachment");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
                printf("  framebuffer incomplete missing attachment");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER:
                printf("  framebuffer incomplete draw buffer");
                break;
            case GL_FRAMEBUFFER_UNSUPPORTED:
                printf("  framebuffer unsupported");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS:
                printf("  framebuffer incomplete layer targets");
                break;
            
            default:
                printf("  ??");
                break;
        }
        
        printf("\n");
        return -1;
    }
    
    return 0;
}

}
