
#ifndef _TP_BUFFER_H
#define _TP_BUFFER_H

#include <cstdio>
#include <cassert>

#include "GL/GLPlatform.h"
#include "GLResource.h"
#include "GL/TPProgramName.h"
#include "GL/TPAttributes.h"


namespace gk {

//! utilisation interne. representation d'un buffer openGL.
class GLBuffer : public GLResource
{
protected:
    GLenum m_usage;
    unsigned int m_length;
    unsigned int m_count;

    // non copyable
    GLBuffer( const GLBuffer& );
    GLBuffer& operator=( const GLBuffer& );
    
    //! constructeur d'un buffer, un tableau de vecteurs de 'count' elements.
    //! \param count nombre de vecteurs,
    //! \param length longueur totale du buffer en OCTETS,
    //! \param data pointeur sur les donnees a transferer dans le buffer, ou NULL pour initialiser un buffer vide,
    //! \param usage decrit le type d'utilisation du buffer.
    GLBuffer( const GLenum target, 
        const unsigned int count, const unsigned int length, const void *data, const GLenum usage= GL_STATIC_DRAW )
        :
        GLResource(),
        m_usage(usage),
        m_length(length),
        m_count(count)
    {
        glGenBuffers(1, &m_name);
        
        glBindBuffer(target, m_name);
        glBufferData(target, m_length, data, m_usage);
    }

    //! efface le contenu du buffer.
    int clear( const GLenum target )
    {
        if(m_name == 0)
            return -1;
        
        glBindBuffer(target, m_name);
        glBufferData(target, m_length, NULL, m_usage);
        return 0;
    }

    //! modifie le contenu d'une partie du buffer.
    //! \param offset position du premier octet a modifier,
    //! \param length nombre d'octets a modifier.
    int update( const GLenum target, 
        const unsigned long int offset, const unsigned long int length, const void *data )
    {
        if(m_name == 0)
            return -1;
        if(offset + length > m_length)
            return -1;
        
        glBindBuffer(target, m_name);
        glBufferSubData(target, offset, length, data);
        return 0;
    }
    
    //! mappe le contenu du buffer en memoire host.
    //! cf. glMapBufferRange pour les flags 'access'.
    void *map( const GLenum target, 
        const unsigned long int offset, const unsigned int length, const GLbitfield access )
    {
        if(m_name == 0)
            return NULL;
        if(offset + length > m_length)
        {
            printf("GLBuffer::map( ): offset + length > buffer length\n");
            return NULL;
        }
        
        glBindBuffer(target, m_name);
        return glMapBufferRange(target, offset, length, access);
    }
    
    int unmap( const GLenum target )
    {
        if(m_name == 0)
            return -1;
        
        glUnmapBuffer(target);
        return 0;
    }
    
    int flush( const GLenum target, 
        const unsigned long int offset, const unsigned int length )
    {
        if(m_name == 0)
            return -1;
        if(offset + length > m_length)
        {
            printf("GLBuffer::flush( ): offset + length > buffer length\n");
            return -1;
        }
        
        glFlushMappedBufferRange(target, offset, length);
        return 0;
    }
    
public:
    //! destructeur.
    virtual ~GLBuffer( )
    {
        if(m_name != 0)
            glDeleteBuffers(1, &m_name);
    }

    //! creation de la ressource openGL.
    int createGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }

    //! destruction de la ressource openGL.
    int releaseGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }

    //! renvoie le nombre de vecteurs alloues (valeur passee au contructeur, cf GLBuffer()).
    unsigned int count( ) const
    {
        if(m_name == 0)
            return 0;
        return m_count;
    }

    //! renvoie le nombre d'octets alloues.
    unsigned long int length( ) const
    {
        return m_length;
    }
};

//! representation d'un buffer d'attribut.
class GLAttributeBuffer : public GLBuffer
{
public:
    //! constructeur.
    GLAttributeBuffer( const unsigned int count, const unsigned int length, const void *data, const GLenum usage= GL_STATIC_DRAW )
        :
        GLBuffer(GL_ARRAY_BUFFER, count, length, data, usage)
    {}
    
    //! destructeur.
    ~GLAttributeBuffer( ) {}
    
    //! efface le contenu du buffer.
    int clear( )
    {
        return GLBuffer::clear(GL_ARRAY_BUFFER);
    }
    
    //! remplace une partie de contenu du buffer.
    int update( const unsigned long int offset, const unsigned long int length, const void *data )
    {
        return GLBuffer::update(GL_ARRAY_BUFFER, offset, length, data );
    }
    
    //! mappe une partie du buffer en memoire centrale.
    //! cf. glMapBufferRange pour les flags 'access'.
    void *map( const unsigned long int offset, const unsigned long int length, const GLbitfield access )
    {
        return GLBuffer::map(GL_ARRAY_BUFFER, offset, length, access );
    }
    
    //! termine le mapping du contenu du buffer en memoire centrale, les donnees modifiees sont transferees sur le gpu.
    int unmap( )
    {
        return GLBuffer::unmap(GL_ARRAY_BUFFER);
    }
    
    //! termine une partie du mapping du contenu du buffer en memoire centrale, les donnees modifiiees sont transferees sur le gpu.
    int flush( const unsigned long int offset, const unsigned long int length )
    {
        return GLBuffer::flush(GL_ARRAY_BUFFER, offset, length);
    }
};

template < typename T >
class ScopedMapAttributeBuffer
{
    GLAttributeBuffer *m_buffer;
    T *m_begin;
    T *m_end;
    
    ScopedMapAttributeBuffer( );
    ScopedMapAttributeBuffer( const ScopedMapAttributeBuffer& );
    ScopedMapAttributeBuffer& operator=( const ScopedMapAttributeBuffer& );
    
public:
    //! cf. glMapBufferRange pour les flags 'access'.
    ScopedMapAttributeBuffer( GLAttributeBuffer *buffer, 
        GLbitfield access= GL_MAP_WRITE_BIT | GL_MAP_INVALIDATE_BUFFER_BIT, 
        const unsigned long int offset= 0, const unsigned int _length= 0 )
        :
        m_buffer(buffer),
        m_begin(NULL),
        m_end(NULL)
    {
        if(m_buffer == NULL)
            return;
        
        int length= (_length == 0) ? m_buffer->length() : _length;
        m_begin= (T *) m_buffer->map(offset, length, access);
        m_end= m_begin + length / sizeof(T);
    }
    
    T& operator[] ( const unsigned int id )
    {
        assert(m_begin != NULL);
        assert(m_begin + id < m_end);
        return m_begin[id];
    }
    
    ~ScopedMapAttributeBuffer( )
    {
        if(m_buffer == NULL)
            return;
        m_buffer->unmap();
    }
};


//! representation d'un buffer d'indexation.
class GLIndexBuffer : public GLBuffer
{
public:
    GLIndexBuffer( const unsigned int count, const unsigned int length, const void *data, const GLenum usage= GL_STATIC_DRAW )
        :
        GLBuffer(GL_ELEMENT_ARRAY_BUFFER, count, length, data, usage )
    {}
    
    ~GLIndexBuffer( ) {}
    
    int clear( )
    {
        return GLBuffer::clear(GL_ELEMENT_ARRAY_BUFFER);
    }
    
    int update( const unsigned long int offset, const unsigned long int length, const void *data )
    {
        return GLBuffer::update(GL_ELEMENT_ARRAY_BUFFER, offset, length, data );
    }
    
    //! cf. glMapBufferRange pour les flags 'access'.
    void *map( const unsigned long int offset, const unsigned long int length, const GLbitfield access )
    {
        return GLBuffer::map(GL_ELEMENT_ARRAY_BUFFER, offset, length, access );
    }
    
    int unmap( )
    {
        return GLBuffer::unmap(GL_ELEMENT_ARRAY_BUFFER);
    }
    
    int flush( const unsigned long int offset, const unsigned long int length )
    {
        return GLBuffer::flush(GL_ELEMENT_ARRAY_BUFFER, offset, length);
    }
};


template < typename T >
class ScopedMapIndexBuffer
{
    GLIndexBuffer *m_buffer;
    T *m_begin;
    T *m_end;
    
    ScopedMapIndexBuffer( );
    ScopedMapIndexBuffer( const ScopedMapIndexBuffer& );
    ScopedMapIndexBuffer& operator=( const ScopedMapIndexBuffer& );
    
public:
    //! cf. glMapBufferRange pour les flags 'access'.
    ScopedMapIndexBuffer( GLIndexBuffer *buffer, 
        GLbitfield access= GL_MAP_WRITE_BIT | GL_MAP_INVALIDATE_BUFFER_BIT, 
        const unsigned long int offset= 0, const unsigned int _length= 0 )
        :
        m_buffer(buffer),
        m_begin(NULL),
        m_end(NULL)
    {
        if(m_buffer == NULL)
            return;
        
        int length= (_length == 0) ? m_buffer->length() : _length;
        m_begin= (T *) m_buffer->map(offset, length, access);
        m_end= m_begin + length / sizeof(T);
    }
    
    T& operator[] ( const unsigned int id )
    {
        assert(m_begin != NULL);
        assert(m_begin + id < m_end);
        return m_begin[id];
    }
    
    ~ScopedMapIndexBuffer( )
    {
        if(m_buffer == NULL)
            return;
        m_buffer->unmap();
    }
};

struct BufferLayout
{
    int size;
    GLenum type;
    unsigned long int stride;
    unsigned long int offset;
    int divisor;
    bool normalize_flag;
    bool integer_flag;
    
    enum
    {
        NORMALIZE_BIT= 1,
        INTEGER_BIT= 2
    };
    
    BufferLayout( )
        :
        size(0),
        type(GL_NONE),
        stride(0),
        offset(0),
        divisor(0),
        normalize_flag(false),
        integer_flag(false)
    {}
    
    BufferLayout( const int _size, const GLenum _type, const unsigned long int _stride, const unsigned long int _offset= 0, 
        const int _divisor= 0, const unsigned int flags= 0 )
        :
        size(_size),
        type(_type),
        stride(_stride),
        offset(_offset),
        divisor(_divisor),
        normalize_flag((flags & NORMALIZE_BIT)),
        integer_flag((flags & INTEGER_BIT))
    {
        assert(stride != 0);
    }
    
    const GLvoid *getOffset( const unsigned int begin= 0 )
    {
        return (const GLvoid *) (offset + begin * stride);
    }
    
    bool operator== ( const BufferLayout& b )
    {
        if(stride != b.stride)
            return false;
        if(offset != b.offset)
            return false;
        if(size != b.size || type != b.type)
            return false;
        if(divisor != b.divisor)
            return false;
        
        return true;
    }
    
    bool operator!= ( const BufferLayout& b )
    {
        return !operator==(b);
    }
};

}

#endif
