
#include <cassert>
#include <cstdio>

#include "GL/GLPlatform.h"

#include "App.h"
#include "ProfilerClock.h"

namespace gk {

App::App( )
    :
    m_key_state(NULL), m_key_map(NULL),
    m_width(0), m_height(0),
    m_stop(1)
{}

App::App( const int w, const int h, const AppSettings& flags )
    :
    m_key_state(NULL), m_key_map(NULL),
    m_width(0), m_height(0),
    m_stop(0)
{
    if(createWindow(w, h, flags) < 0)
        Close();
}

App::~App( )
{
    delete [] m_key_state;
    delete [] m_key_map;
}

void AppSettings::apply( ) const
{
    SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, depth_size);
    SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, double_buffer);
    SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, swap_control);
    if(multi_samples > 0)
    {
        SDL_GL_SetAttribute(SDL_GL_MULTISAMPLEBUFFERS, 1);
        SDL_GL_SetAttribute(SDL_GL_MULTISAMPLESAMPLES, multi_samples);
    }
    
    // opengl context, sdl > 1.3
    //~ SDL_GL_CONTEXT_MAJOR_VERSION,
    //~ SDL_GL_CONTEXT_MINOR_VERSION

    // opengl core / compatibility profile, sdl patch ?
    // opengl debug, sdl patch ?
}



int App::resizeWindow( const int w, const int h, const AppSettings& settings )
{
    //~ SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
    //~ SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
    //~ SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, 1);
    //~ SDL_GL_SetAttribute(SDL_GL_MULTISAMPLEBUFFERS, 1);
    //~ SDL_GL_SetAttribute(SDL_GL_MULTISAMPLESAMPLES, 4);
    settings.apply();
    
    // redimensionnement la fenetre
    int width= (w != 0) ? w : m_width;
    int height= (h != 0) ? h : m_height;
    unsigned int mode_flags= SDL_OPENGL | settings.flags;
    //~ if(settings.flags == 0)
        //~ // resizable et fullscreen sont exclusifs, et la fenetre par defaut est resizable
        //~ mode_flags|= SDL_RESIZABLE;
        // ne pas forcer resizable...
    
    if((settings.flags & SDL_FULLSCREEN) != 0)
    {
        SDL_Rect **modes= SDL_ListModes(NULL, SDL_OPENGL | SDL_FULLSCREEN);
        if(modes != (SDL_Rect **) -1)
        {
            width= modes[0]->w;
            height= modes[0]->h;
            printf("App::resizeWindow( ): no frame / fullscreen %d x %d\n", width, height);
        }
    }
    
    if(SDL_GetVideoInfo() == NULL 
    || SDL_SetVideoMode(width, height, 0, mode_flags) < 0)
    {
        printf("SDL_SetVideoMode() failed:\n%s\n", SDL_GetError());
        return -1;
    }
    
    // conserve la nouvelle taille
    m_stop= 0;
    m_width= width;
    m_height= height;
    return 0;
}

int App::createWindow( const int w, const int h, const AppSettings& flags )
{
    // initialise SDL
    if(SDL_Init(SDL_INIT_EVERYTHING) < 0 
    || SDL_GetVideoInfo() == NULL )
    {
        printf("SDL_Init() failed:\n%s\n", SDL_GetError());
        Close();
        return -1;
    }
    
    // enregistre le destructeur de sdl
    atexit(SDL_Quit);
    
    // fixe le titre de la fenetre
    SDL_WM_SetCaption("gKit", "");
    
    // clavier unicode
    SDL_EnableUNICODE(1);
    
    // + contournement d'un bug de sdl windows sur claviers non qwerty
    int keys_n= 0;
    SDL_GetKeyState(&keys_n);
    //~ SDL_GetKeyboardState(&keys_n);  // sdl 1.3
    
    m_key_state= new unsigned char[keys_n];
    m_key_map= new unsigned int[keys_n];
    for(int i= 0; i < keys_n; i++)
    {
        m_key_state[i]= 0;
        m_key_map[i]= i;
    }
    
    // creer la fenetre et le contexte openGL
    if(resizeWindow(w, h, flags) < 0)
    {
        Close();
        return -1;
    }
    
    {
        printf("openGL version: '%s'\nGLSL version: '%s'\n",
            glGetString(GL_VERSION),
            glGetString(GL_SHADING_LANGUAGE_VERSION));
    }
    
    {
        int swap= 0;
        SDL_GL_GetAttribute(SDL_GL_SWAP_CONTROL, &swap);
        printf("swap control: %s\n", swap ? "on" : "OFF");
    }
    
    {
        int direct_rendering= 0;
        SDL_GL_GetAttribute(SDL_GL_ACCELERATED_VISUAL, &direct_rendering);
        printf("direct rendering: %s\n", direct_rendering ? "on" : "OFF");
    }
    
    {
        int multisample= 0;
        SDL_GL_GetAttribute(SDL_GL_MULTISAMPLEBUFFERS, &multisample);
        if(multisample == 0)
            printf("multisample: OFF\n");
        
        else
        {
            int samples= 0;
            SDL_GL_GetAttribute(SDL_GL_MULTISAMPLESAMPLES, &samples);
            printf("multisample: %d samples\n", samples);
        }
    }
    
    {
        GLint n;
        glGetIntegerv(GL_MAX_TEXTURE_IMAGE_UNITS, &n);
        printf("texture units: %d\n", n);
    }
    
    {
        GLint n;
        glGetIntegerv(GL_MAX_VERTEX_ATTRIBS, &n);
        printf("vertex attributes: %d\n", n);
    }
    
    // initialise les extensions openGL 
    GLenum err;
    err= glewInit();
    if(err != GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        Close();
        return -1;
    }

    if(flags.major_version != 0)
    {
        const char *version_string= (const char *) glGetString(GL_VERSION);
        int major, minor;
        if(sscanf(version_string, "%d.%d", &major, &minor) != 2)
        {
            printf("openGL version: parse error %s\n", version_string);
            major= flags.major_version;
            minor= flags.minor_version;
        }
        
        int flags_version= flags.major_version * 100 + flags.minor_version * 10;
        int version= major * 100 + minor * 10;
        if(flags_version > version)
        {
            printf("requested openGL version %d, not supported: %d\n", flags_version, version);
            Close();
            return -1;
        }
        
        printf("matching openGL version %d.%d, supported: %s\n", 
            flags.major_version, flags.minor_version, version_string);
    }
    
    if(!glewIsSupported("GL_EXT_texture_filter_anisotropic"))
    {
        printf("EXT texture filter anisotropic: extension missing.\n");
        //~ return -1;
    }
    else
    {
        GLint max_aniso;
        glGetIntegerv(GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &max_aniso);
        printf("EXT texture filter anisotropic: max %d\n", max_aniso);
    }
    
    // fixe l'etat openGL par defaut 
    glClearColor(0.0, 0.0, 0.0, 1.0);
    glClearDepth(1.f);
    
    glDisable(GL_DITHER);
    if(flags.multi_samples == 0)
        glDisable(GL_MULTISAMPLE);
    
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    
    glPolygonMode(GL_FRONT, GL_FILL);
    glPolygonMode(GL_BACK, GL_FILL);    
    
    return 0;
}

bool App::isClosed( )
{
    return (m_stop == 1);
}

void App::Close( )
{
    m_stop= 1;
}

unsigned char *App::getKeys( )
{
    return m_key_state;
}

unsigned char& App::key( const int key )
{
    return m_key_state[key];

}

bool App::processEvents( )
{
    SDL_Event event;

    while(SDL_PollEvent(&event))
    {
        switch(event.type)
        {
            case SDL_VIDEORESIZE:
                resizeWindow(event.resize.w, event.resize.h);
                
                // prevenir l'application
                processWindowResize(event.resize);
                break;
                
            case SDL_QUIT:
                m_stop= 1;
                break;
                
            case SDL_KEYUP:
                // gestion clavier unicode
                // desactive la touche
                m_key_state[m_key_map[event.key.keysym.sym]]= 0;
                
                // prevenir l'application
                processKeyboardEvent(event.key);
                break;
                
            case SDL_KEYDOWN:
                // gestion clavier unicode
                m_key_state[event.key.keysym.sym]= 1;
                if(event.key.keysym.unicode < 128 && event.key.keysym.unicode > 0 )
                {
                    // conserve l'association entre le code de la touche et l'unicode 
                    m_key_map[event.key.keysym.sym]= (unsigned int) event.key.keysym.unicode;
                    // annule l'etat associe au code la touche
                    m_key_state[event.key.keysym.sym]= 0;
                    // active l'etat associe a l'unicode
                    m_key_state[event.key.keysym.unicode]= 1;
                }
                
                // prevenir l'application
                processKeyboardEvent(event.key);
                break;
            
            case SDL_MOUSEMOTION:
                // prevenir l'application
                processMouseMotionEvent(event.motion);
                break;
            
            case SDL_MOUSEBUTTONUP:
            case SDL_MOUSEBUTTONDOWN:
                // prevenir l'application
                processMouseButtonEvent(event.button);
                break;
        }
    }
    
    return (m_stop == 0);
}


int App::run( )
{
    if(isClosed())
    {
        printf("App::run( ): no window.\n");
        return -1;
    }
    
    // termine l'initialisation des classes derivees, chargement de donnees, etc.
    if(init() < 0)
    {
    #ifdef VERBOSE
        printf("App::init( ): failed.\n");
    #endif
        return -1;
    }

    ProfilerClock::Ticks start= ProfilerClock::getTicks();
    ProfilerClock::Ticks last_frame= start;
    while(!isClosed())
    {
        // traitement des evenements : clavier, souris, fenetre, etc.
        processEvents();
        
        // mise a jour de la scene
        ProfilerClock::Ticks frame= ProfilerClock::getDelay(start);
        ProfilerClock::Ticks delta= ProfilerClock::delay(frame, last_frame);
        
        if(update( frame / 1000, delta / 1000 ) == 0)
            break;
        
        // affiche la scene
        if(draw() == 0)
            break;
        
        last_frame= frame;
    }
    
    // destruction des ressources chargees par les classes derivees.
    if(quit() < 0)
        return -1;

    return 0;
}

}
