/*
    render.c
    
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "vec.h"
#include "model.h"
#include "dpoint.h"
#include "render.h"


MODEL *render_get_model(RENDER *render)
{
    assert(render!=NULL);
    assert(render->model!=NULL);
    
    return render->model;
}

LIGHT *render_get_light_ptr(RENDER *render, int light_id)
{
    assert(render!=NULL);
    assert(light_id>=0 && light_id<render->lights_n);
    
    return &render->lights[light_id];
}

float *light_get_norm_ptr(LIGHT *light)
{
    
    assert(light!=NULL);
    
    return model_face_get_norm_ptr(light->model, 
        model_get_face_ptr(light->model, light->face_id));
}

float render_get_light_emission(RENDER *render, int light_id, DPOINT *p, DPOINT *q)
{
    /* calcule l'energie emise par le point q (situe sur une source de lumiere) vers le point p 
     */
    
    LIGHT *light;
    VEC w;
    float costheta;
    
    light= render_get_light_ptr(render, light_id);
    vec3_sub(w, p->p, q->p);
    vec3_norm(w, w);
    costheta= vec3_dot(w, light_get_norm_ptr(light));
    if(costheta < 0.f)
        return 0.f;
    else
        return light->emission * costheta;
}

float render_get_Le(RENDER *render, DPOINT *p)
{
    /* renvoie l'energie emise par p
     */
    
    float *le;
    
    le= dpoint_get_material_ptr(p)->e;
    return (le[0] + le[1] + le[2]) / 3.f;
}

static void render_add_light(RENDER *render, int face_id)
{
    LIGHT *light;
    MODEL *model;
    FACE *face;
    float *Le;
    
    array_add((void **) &render->lights,
        &render->lights_size,
        render->lights_n,
        sizeof(LIGHT), RENDER_LIGHT_MORE);
    
    light= &render->lights[render->lights_n++];

    model= render_get_model(render);
    light->model= model;
    light->face_id= face_id;
    
    face= model_get_face_ptr(model, face_id);
    light->area= model_face_get_area(model, face);
    
    Le= model_face_get_material_ptr(model, face)->e;
    light->emission= (Le[0] + Le[1] + Le[2]) / 3.f;
    
    render->lights_total_area+= light->area;
    render->lights_total_emission+= light->emission * light->area;
}

int render_material_is_light(RENDER *render, int material_id)
{
    return (material_id==render->light_material_id);
}

static void render_init_lights(RENDER *render)
{
    MODEL *model;
    FACE *face;
    int i, n;
    
    model= render_get_model(render);

    render->light_material_id= material_find(model, render->light_material_name);
    assert(render->light_material_id >= 0);

    render->lights_total_emission= 0.f;
    render->lights_total_area= 0.f;
    
    n= model_get_faces_n(model);
    for(i= 0; i < n; i++)
    {
        face= model_get_face_ptr(model, i);
        if(face_get_material_id(face)==render->light_material_id)
            render_add_light(render, i);
    }
    
    printf("render_init_lights(): %d  area %f  emission %f\n", render->lights_n,
        render->lights_total_area, render->lights_total_emission);
}


static void render_add_face(RENDER *render, FACE *face)
{
    MODEL *model;
    float *area;
 
    array_add((void **) &render->faces_area,
        &render->faces_area_size,
        render->faces_area_n,
        sizeof(float), RENDER_FACE_MORE);
    
    area= &render->faces_area[render->faces_area_n++];
    
    model= render_get_model(render);
    if(render_material_is_light(render, face_get_material_id(face))==0)
        *area= model_face_get_area(model, face);
    else
        // ne pas comptabiliser les sources de lumiere
        *area= 0.f;
    
    render->faces_total_area+= *area;
}

static void render_init_area(RENDER *render)
{
    /* calcule l'aire totale du modele (sources de lumiere exclues)
     */
    
    MODEL *model;
    FACE *face;
    int i, n;
    
    model= render_get_model(render);
    n= model_get_faces_n(model);
    render->faces_total_area= 0.f;
    for(i= 0; i < n; i++)
    {
        face= model_get_face_ptr(model, i);
        render_add_face(render, face);
    }
    
    printf("render_init_area(): %f\n", render->faces_total_area);
}

float render_face_get_area(RENDER *render, int face_id)
{
    assert(render!=NULL);
    assert(face_id>=0 && face_id<render->faces_area_n);
    
    return render->faces_area[face_id];
}

void render_set_default_material(RENDER *render)
{
    int i, j;
    int n;

    MATERIAL materials[]=
    {
        {
            .name=    "diffuse50SG",
            .e=       {0.f, 0.f, 0.f},
            .m=       {1.f, 1.f, 1.f},
            .m_sp=    {1.f, 1.f, 1.f},
            .k_du=     .5f,
            .k_dd=     0.f,
            .dd_shiny= 1.f,
            .k_sp=     0.f
        },

        {
            .name=    "mirrorSG",
            .e=       {0.f, 0.f, 0.f},
            .m=       {1.f, 1.f, 1.f},
            .m_sp=    {1.f, 1.f, 1.f},
            .k_du=     0.f,
            .k_dd=     0.f,
            .dd_shiny= 1.f,
            .k_sp=     1.f
        },

        {
            .name=    "diffuseLuminaire1SG",
            .e=       {1000., 1000., 1000.},
            .m=       {1., 1., 1.},
            .m_sp=    {1., 1., 1.},
            .k_du=     1.,
            .k_dd=     0.,
            .dd_shiny= 1.,
            .k_sp=     0.
        }
    };

    const int materials_n= sizeof(materials) / sizeof(MATERIAL);

    n= 0;
    for(i= 0; i < render->model->mat_n; i++)
    {
        for(j= 0; j < materials_n; j++)
        {
            if(strcmp(materials[j].name, render->model->mat[i].name)==0)
            {
                // fix: beurk !!
                free(render->model->mat[i].name);
                memcpy(&render->model->mat[i], &materials[j], sizeof(MATERIAL));
                render->model->mat[i].name= strdup(materials[j].name);
                n++;
                break;
            }
        }
    }

    printf("render_set_default_material(): %d\n", n);
}

CAMERA *render_get_camera(RENDER *render)
{
    assert(render!=NULL);
    
    return &render->camera;
}

RENDER *render_new(MODEL *model, char *light_material_name)
{
    RENDER *render;
    
    render= (RENDER *) malloc(sizeof(RENDER));
    assert(render!=NULL);
    
    render->model= model;
    
    render->light_material_name= strdup(light_material_name);
    render->lights= NULL;
    render->lights_size= 0;
    render->lights_n= 0;
    
    render->faces_area= NULL;
    render->faces_area_size= 0;
    render->faces_area_n= 0;
    
    render->rays_n= 0;
    render->rays_camera_n= 0;
    render->rays_light_n= 0;
    
    render_set_default_material(render);
    render_init_lights(render);
    render_init_area(render);
    
    printf("render_init(): done.\n");
    return render;
}

void render_free(RENDER *render)
{
    if(render==NULL)
        return;
    
    if(render->light_material_name!=NULL)
        free(render->light_material_name);
    if(render->lights!=NULL)
        free(render->lights);
    if(render->faces_area!=NULL)
        free(render->faces_area);
    
    free(render);
}
