/*
    main.c
    
    exemple d'utilisation de rendertoy
    
    questions ??
    mailto:jciehl@bat710.univ-lyon1.fr
    
    decembre 2006
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <math.h>

#include <SDL/SDL.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "vec.h"
#include "model.h"
#include "maya_obj.h"
#include "off.h"
#include "img.h"
#include "hdr.h"
#include "camera.h"
#include "ray.h"
#include "dpoint.h"
#include "render.h"

void shade(void *vparam);


int stop= 0;
const SDL_VideoInfo *screen_info= NULL;
SDL_Surface *screen= NULL;

void *display_param= NULL;
void (*display_scene)(void *)= NULL;

// position et orientation du point de vue
float camera_x, camera_y, camera_z;
float camera_angle, camera_angle2;
float camera_vitesse;

int lumiere;

// timer pour l'animation
int timer;
int timer_rate;

// conversion degres vers radians
float radians(float d)
{
    return d * 0.0174532925f;
}

void camera_pref_fname(char *pref, char *scene_fn)
{
    char *ext;

    // construit le nom du fichier de preference pour le modele
    strcpy(pref, scene_fn);
    ext= strrchr(pref, '.');
    if(ext==NULL)
        ext= pref;
    
    strcpy(ext, ".mob");
}

void camera_coller(void)
{
    char pref_fn[1024];
    MODEL *model;
    FILE *in;
    float x, y, z, v, a, b;
    
    model= render_get_model((RENDER *) display_param);
    camera_pref_fname(pref_fn, model->filename);
    in= fopen(pref_fn, "rt");
    if(in==NULL)
        return;
    
    if(fscanf(in, "x %f y %f z %f v %f a %f b %f ", 
        &x, &y, &z, &v, 
        &a, &b)==6)
    {
        camera_x= x;
        camera_y= y;
        camera_z= z;
        camera_vitesse= v;
        camera_angle= a;
        camera_angle2= b;
    }
    
    fclose(in);
}

void camera_copier(void)
{
    char pref_fn[1024];
    MODEL *model;
    FILE *out;
    
    model= render_get_model((RENDER *) display_param);
    camera_pref_fname(pref_fn, model->filename);
    out= fopen(pref_fn, "wt");
    if(out==NULL)
        return;
    
    fprintf(out, "x %f y %f z %f\nv %f\na %f b %f\n", 
        camera_x, camera_y, camera_z, camera_vitesse, 
        camera_angle, camera_angle2);
    
    fclose(out);
}

// affichage
void display(void)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);		// selectionne la matrice de la scene
    glLoadIdentity();				// reinitialise les transformations

    if(lumiere!=0)
    {
        GLfloat mat_specular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
        GLfloat mat_shininess[] = { 20.0f };
        GLfloat light_position[] = { 0.0f, 1.0f, 1.0f, 0.0f };
        
        glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
        glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
        glLightfv(GL_LIGHT0, GL_POSITION, light_position);
        
        glEnable(GL_LIGHTING);
        glEnable(GL_LIGHT0);
        glPolygonMode(GL_FRONT, GL_FILL);
        glPolygonMode(GL_BACK, GL_FILL);
    }
    else
    {
        glDisable(GL_LIGHTING);
        glPolygonMode(GL_FRONT, GL_LINE);
        glPolygonMode(GL_BACK, GL_LINE);
    }
    
    // oriente la scene par rapport a la camera qui est restee en 0,0,0
    glRotatef(360.f - camera_angle2, 1.f, 0.f, 0.f);
    glRotatef(360.f - camera_angle, 0.f, 1.f, 0.f);
    glTranslatef(-camera_x, -camera_y, -camera_z);

    // desssine la scene
    display_scene(display_param);
    
    SDL_GL_SwapBuffers();
}

// callback de glut : appellee regulierement
void anime(void)
{
    // attente
    SDL_Delay(1000. / timer_rate);
    
    timer++;
}

static void keyboard(SDL_Event *event)
{
   SDLMod mod;
    
    mod= SDL_GetModState();
    
    switch(event->key.keysym.sym)
    {
        case SDLK_q:
        case SDLK_ESCAPE:
            stop= 1;
        break;

        case SDLK_c:
            camera_copier();
        break;
        
        case SDLK_v:
            camera_coller();
        break;

        case SDLK_SPACE:
            lumiere= (lumiere +1) %2;
        break;
        
        case SDLK_r:
        case SDLK_s:
            shade(display_param);
        break;
        
        // navigation
        case SDLK_UP:
            if(mod&KMOD_CTRL || mod&KMOD_ALT)
            {
                // rotation de la camera sur X
                camera_angle2+= 1.;
                if(camera_angle2 >= 360.)
                        camera_angle2= 0.;
            }
            else
            {
                // deplacement de la camera dans la direction de la visee
                camera_z-= cos(radians(camera_angle)) * camera_vitesse;
                camera_x-= sin(radians(camera_angle)) * camera_vitesse;
            }
        break;

        case SDLK_DOWN:
            if(mod&KMOD_CTRL || mod&KMOD_ALT)
            {
                // rotation de la camera sur Y+
                camera_angle2-= 1.;
                if(camera_angle2 < 0.)
                    camera_angle2= 359.;
            }
            else
            {
                // deplacement de la camera dans la direction opposee de la visee
                camera_z+= cos(radians(camera_angle)) * camera_vitesse;
                camera_x+= sin(radians(camera_angle)) * camera_vitesse;
            }
        break;

        case SDLK_LEFT:
            if(mod&KMOD_CTRL || mod&KMOD_ALT)
            {
                // rotation de la camera sur Y-
                camera_angle+= 1.;
                if(camera_angle >= 360.)
                    camera_angle= 0.;
            }
            else
            {
                // deplacement perpendiculaire a la direction de visee
                camera_z-= cos(radians(camera_angle + 90.)) * camera_vitesse;
                camera_x-= sin(radians(camera_angle + 90.)) * camera_vitesse;
            }
        break;

        case SDLK_RIGHT:
            if(mod&KMOD_CTRL || mod&KMOD_ALT)
            {
                // rotation de la camera sur Y+
                camera_angle-= 1.;
                if(camera_angle < 0.)
                    camera_angle= 359.;
            }
            else
            {
                // deplacement perpendiculaire a la direction de visee
                camera_z-= cos(radians(camera_angle - 90.)) * camera_vitesse;
                camera_x-= sin(radians(camera_angle - 90.)) * camera_vitesse;
            }
        break;

        case SDLK_PAGEUP:
            camera_y+= camera_vitesse;
        break;
        
        case SDLK_PAGEDOWN:
            camera_y-= camera_vitesse;
        break;
                
        default:
            break;
    }
}

static int resize(int width, int height)
{
    screen_info= SDL_GetVideoInfo();
    if(screen_info==NULL)
    {
        printf("\n -- failed: '%s'\n", SDL_GetError());
        SDL_Quit();
        return -1;
    }
    
    SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
    SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
    
    screen= SDL_SetVideoMode(width, height, 
        screen_info->vfmt->BitsPerPixel, 
        SDL_RESIZABLE | SDL_OPENGL | SDL_HWSURFACE);
    if(screen==NULL)
    {
        printf("\n -- failed: '%s'\n", SDL_GetError());
        SDL_Quit();
        return -1;
    }
    
    glMatrixMode(GL_PROJECTION);	// selectionne la matrice de visualisation
    glLoadIdentity();				// reinitialise les transformations

    gluPerspective(50., 1., 1., 10000.);
  
    return 0;
}

static int initsdl(int width, int height)
{
    if(SDL_Init(SDL_INIT_VIDEO) < 0)
    { 
        printf("\n -- failed : '%s'\n", SDL_GetError());
        return -1;
    }

    SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, 20);
    
    atexit(SDL_Quit);
    resize(width, height);
    return 0;
}


// mise en place des parametres d'openGL
void initgl(void)
{
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.);

    glShadeModel (GL_SMOOTH);
    
    glEnable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
    // glCullFace(GL_BACK);

    camera_x= 0.f;
    camera_y= 0.f;
    camera_z= 50.f;
    camera_angle= 0.f;
    camera_vitesse= 10.f;

    timer= 0;
    timer_rate= 50;	// frequence d'affichage
    
    lumiere= 0;
}

int view_init(int largeur, int hauteur)
{
    if(initsdl(largeur, hauteur) < 0)
    {
        printf("\n -- failed: '%s'\n", SDL_GetError());
        return -1;
    }
    SDL_WM_SetCaption("RenderToy Viewer", "");
    
    initgl();
    
    return 0;
}

static void process_events(void)
{
    SDL_Event event;

    while(SDL_PollEvent(&event))
    {
        switch(event.type)
        {
            // case SDL_KEYUP:
            case SDL_KEYDOWN:
                keyboard(&event);
                break;
            
            case SDL_VIDEORESIZE:
                resize(event.resize.w, event.resize.h);
                break;
            
            case SDL_QUIT:
                stop= 1;
                break;
        }
    }
}

void render_display_debug(void *vparam)
{
    RENDER *render= (RENDER *) vparam;
    MODEL *model;
    FACE *face;
    int n;
    int i;

    glLineWidth(1.);
    model= render_get_model(render);
    n= model_get_faces_n(model);
    for(i= 0; i < n; i++)
    {
        face= model_get_face_ptr(model, i);
        
        if(render_material_is_light(render, face_get_material_id(face)))
            // sources de lumieres en jaune
            glColor3f(.5f, .5f, 0.f);
        else
            // objets en blanc
            glColor3f(.5f, .5f, .5f);

        if(face_get_vertex_n(face)==3)
        {
            glBegin(GL_TRIANGLES);
                glNormal3fv(model_face_get_norm_ptr(model, face));
                
                glVertex3fv(model_face_get_vertex_ptr(model, face, 0));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 1));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 2));
            glEnd();
        }
        else if(face_get_vertex_n(face)==4)
        {
            glBegin(GL_QUADS);
                glNormal3fv(model_face_get_norm_ptr(model, face));
                
                glVertex3fv(model_face_get_vertex_ptr(model, face, 0));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 1));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 2));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 3));
            glEnd();
        }
    }
}


void view(void (*displayfunc)(void *), void *param)
{
    display_param= param;
    display_scene= displayfunc;
    
   while(!stop)
    {
        process_events();
        anime();
        display();
    }
}


/* ecrire un pixel dans une image */
void img_set_pix(IMG *img, int x, int y, float r, float v, float b)
{
    unsigned int pix;
    assert(img!=NULL);
    assert(x>=0 && x < img->largeur);
    assert(y>=0 && y < img->hauteur);
    
    pix= y * img->largeur * img->channels + x * img->channels;
    img->dataf[pix]= r;
    img->dataf[pix +1]= v;
    img->dataf[pix +2]= b;
}

void shade(void *vparam)
{
    IMG *img;
    RENDER *render= (RENDER *) vparam;
    MODEL *model;
    CAMERA *camera;
    RAY ray;
    DPOINT o, p, q;
    VEC in, out;
    float qa;
    float Le;
    float brdf;
    float L, Ln;
    int viewport[4];
    int light_id;
    int x, y;

    const int N= 1000;
    int i;
    
    /* cree une camera correspondant a la vue opengl active */
    model= render_get_model(render);
    camera= render_get_camera(render);
    camera_set_projection_opengl(camera);
     
    /* recupere le plan image :
        viewport[0]= xmin, [1]= ymin, [2]= largeur, [3]= hauteur */
    camera_get_viewport(camera, viewport);

     /* cree l'image resultat */
    img= new_img_datafloat(viewport[2], viewport[3]);
    
    /* parcours l'image et genere un rayon / pixel */
    for(y= 0; y < viewport[3]; y++)
    {
        for(x= 0; x < viewport[2]; x++)
        {
            Ln= 0.f;
            for(i= 0; i < N; i++)
            {
                camera_sample_pixel(camera, 
                    (float) x + (float) viewport[0] /* + (float) drand48() */,
                    (float) y + (float) viewport[1] /* + (float) drand48() */,
                    model, &ray);
                
                if(ray_intersect(&ray, &p) < 0)
                {
                    /* pas d'intersection : ecrire un pixel noir dans l'image
                        le repere de la camera et de l'image sont inverses sur Y !!
                     */
                    L= 0.f;
                    continue;
                }

                Ln+= render_get_Le(render, &p);
                
                /* intersection : calculer l'energie reflechie par p vers la camera
                */
                
                /* choisir un point q sur une source de lumiere */
                render_sample_light_Le(render, &light_id, &q, &qa);
                
                /* determiner l'energie emise par q vers p */
                Le= render_get_light_emission(render, light_id, &p, &q);
                if(Le <= 0.f)
                    continue;
                
                /* reflechir l'energie */
                ray_get_origin(&ray, &o);
                brdf= dpoint_brdf_3points(&o, &p, &q);
                if(brdf <= 0.f)
                    continue;
                
                if(dpoint_visibility_ray(&p, &q) < 0)
                {
                    /* construire l'estimateur : f(x) / p(x), avec f(x)= Le_q * brdf_p */
                    L= Le * brdf * dpoint_get_G(&p, &q);
                    L= L / qa;
                }
                else
                {
                    /* p et q ne sont pas visibles ... */
                    L= 0.f;
                }
                
                /* accumuler l'echantillon */
                Ln+= L;
            }
            
            /* enregistrer le resultat dans l'image */
            Ln/= (float) N;
            img_set_pix(img, x, viewport[3] -1 - y, 
                Ln, Ln, Ln);
        }
        
        printf("%03d\r", viewport[3] -1 - y); fflush(stdout);
    }
    
    /* enregistre l'image resultat au format hdr */
    printf("write 'render.hdr'.\n");
    hdr_write(img, "render.hdr");
    free_img(img);
}

int main(int argc, char **argv)
{
    MODEL *model;
    RENDER *render;
    int width, height;
    
    model= NULL;
    if(argc > 1)
    {
        /* charge le modele .obj ou .off
         */
        if((model_is_obj(argv[1])==0 
            && model_load_obj(&model, argv[1]) < 0)
        || (model_is_off(argv[1])==0 
            && model_load_off(&model, argv[1]) < 0))
        {
            printf("\n  -- unable to load model '%s'\n", argv[1]);
            return 1;
        }
        
        /* recupere la taille de l'image a calculer */
        if(argc <= 2 || sscanf(argv[2], "%d", &width)!=1)
        {
            width= 256;
            printf("render(): using default width: %d\n", width);
        }
            
        if(argc <= 3 || sscanf(argv[3], "%d", &height)!=1)
        {
            height= 256;
            printf("render(): using default height: %d\n", height);
        }
    }
    else
    {
        printf("usage: %s model.obj [width height]\n", argv[0]);
        printf("usage: %s model.off [width height]\n", argv[0]);
        return 0;
    }
    
    if(model==NULL)
    {
        printf("\n -- unable to load model\n");
        return 1;
    }
    
    /* prepare le modele pour le calcul : calcule les normales des faces */
    model_set_faces_norm(model);
    
    /* cree le contexte de calcul
     */
    render= render_new(model, "diffuseLuminaire1SG");
    
    /* visualisation interactive du modele */
    view_init(width, height);
    view((void *) render_display_debug, (void *) render);
    /* la fonction shade sera appellee directement depuis view */
    
    /* fin
     */
    render_free(render);
    model_free(model);
    
    printf("\ndone.\n");
    return 0;
}
