#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <math.h>

#include "vec.h"
#include "model.h"
#include "ray.h"
#include "dpoint.h"


// construit un repere local 
void dpoint_set_frame(DPOINT *point, VEC norm)
{
    VEC t;
    int  m;

    assert(point!=NULL);

    vec3_norm(point->w, norm);

    // construire un vecteur non colineaire avec la normale
    vec3_copy(t, point->w);
    m= 0;
    if(fabsf(t[1]) < fabsf(t[m]))
        m= 1;
    if(fabsf(t[2]) < fabsf(t[m]))
        m= 2;

    t[m]= 1.f;

    // construire un repere direct ortho
    vec3_cross(point->u, t, point->w);
    vec3_norm(point->u, point->u);

    vec3_cross(point->v, point->w, point->u);
    vec3_norm(point->v, point->v); // ?? 
}

// passage repere local vers global
void dpoint_transform_vec(DPOINT *point, VEC local, VEC global)
{
    assert(point!=NULL);
    
    global[0]= point->u[0]*local[0] + point->v[0]*local[1] + point->w[0]*local[2];
    global[1]= point->u[1]*local[0] + point->v[1]*local[1] + point->w[1]*local[2];
    global[2]= point->u[2]*local[0] + point->v[2]*local[1] + point->w[2]*local[2];
}


int dpoint_init_ray(MODEL *model, DPOINT *point, RAY *ray)
{
    FACE *face;
    
    assert(model!=NULL);
    assert(point!=NULL);
    assert(ray!=NULL);
    
    if(ray->hit < 0)
    {
        dpoint_init(model, point);
        return -1;
    }

    // calcule la position du point d'intersection
    vec3_add_mul_const(point->p, ray->origin, ray->t, ray->direction);
    
    // determine un repere local 
    face= model_get_face_ptr(model, ray->hit);
    dpoint_set_frame(point, model_face_get_norm_ptr(model, face));
    
    // recupe le materiau 
    point->model= model;
    point->face_id= ray->hit;
    return 0;
}

void dpoint_init_face_point(MODEL *model, DPOINT *point, int face_id, VEC p)
{
    FACE *face;
    assert(model!=NULL);
    assert(point!=NULL);
    
    vec3_copy(point->p, p);
    face= model_get_face_ptr(model, face_id);
    dpoint_set_frame(point, model_face_get_norm_ptr(model, face));
    point->model= model;
    point->face_id= face_id;
}

void dpoint_init(MODEL *model, DPOINT *point)
{
    assert(model!=NULL);
    assert(point!=NULL);
    
    vec3_zero(point->p);
    vec3_zero(point->u);
    vec3_zero(point->v);
    vec3_zero(point->w);
    point->face_id= -1;
    point->model= model;
}

MATERIAL *dpoint_get_material_ptr(DPOINT *point)
{
    FACE *face;
    
    face= model_get_face_ptr(point->model, point->face_id);
    return model_face_get_material_ptr(point->model, face);
}

int dpoint_get_material_id(DPOINT *point)
{
    FACE *face;
    
    face= model_get_face_ptr(point->model, point->face_id);
    return face_get_material_id(face);
}

static float material_albedo(MATERIAL *mat)
{
    return .8f;
}

static float material_brdf(MATERIAL *mat, VEC in, VEC N, VEC out)
{
    return material_albedo(mat) / M_PI;

#if 0
    VEC out_m;
    float d;

    vec3_mirror(out_m, N, in);
    d= vec3_dot(out_m, out);
    
    if(d < EPSILON)
        return mat->k_du;
    
    else if(mat->dd_shiny==1.f)
        return mat->k_du + mat->k_dd*d;
    else	
        return mat->k_du + mat->k_dd*pow(d, mat->dd_shiny);
#endif
}

float dpoint_brdf(DPOINT *point, VEC in, VEC out)
{
    FACE *face;
    MATERIAL *material;
    
    face= model_get_face_ptr(point->model, point->face_id);
    material= model_face_get_material_ptr(point->model, face);
    if(vec3_dot(point->w, out) < 0.f)
        return 0.f;
    else
        return material_brdf(material, in, point->w, out);
}

/* calcule la brdf au point p : la lumiere parcours q -> p -> o
 */
float dpoint_brdf_3points(DPOINT *o, DPOINT *p, DPOINT *q)
{
    VEC in, out;
    
    vec3_sub(out, o->p, p->p);
    vec3_norm(out, out);
    vec3_sub(in, q->p, p->p);
    vec3_norm(in, in);
    return dpoint_brdf(p, in, out);
}

float dpoint_get_dWdA(DPOINT *p, DPOINT *q)
{
    /* calcule le changement de mesure entre p et q 
     */

    VEC w;
    float costhetaq, r2;
    
    vec3_sub(w, p->p, q->p);
    r2= vec3_length2(w);
    
    vec3_norm(w, w);
    costhetaq= vec3_dot(q->w, w);
    
    return costhetaq / r2;
}

float dpoint_get_G(DPOINT *p, DPOINT *q)
{
    /* calcule le facteur geometrique G entre 2 points visibles
    */
    
    VEC w;
    float costheta;
    
    vec3_sub(w, q->p, p->p);
    vec3_norm(w, w);
    costheta= vec3_dot(p->w, w);
    
    return costheta * dpoint_get_dWdA(p, q);
}

int dpoint_trace_ray(DPOINT *point, VEC w, DPOINT *q)
{
    /* determine le point q visible dans la direction w
     */

    RAY ray;
    
    // recherche le point visible dans la direction choisie
    ray_init_direction(point->model, &ray, point->p, w);
    return ray_intersect(&ray, q);
}

int dpoint_visibility_ray(DPOINT *p, DPOINT *q)
{
    RAY ray;
    VEC pp, qq;
    
    /* decaler les points le long de la normale de leur face
        pour eviter les problemes de precision numerique
     */
    vec3_add_mul_const(pp, p->p, .001f, p->w);
    vec3_add_mul_const(qq, q->p, .001f, q->w);
    
    /* cree le rayon*/
    ray_init_occlusion(p->model, &ray, pp, qq);
    
    return ray_occlusion(&ray);
}


/* mesure sur les directions */

static void sample_direction_local(DPOINT *point, VEC w, float *pw)
{
    /* choisit une direction uniformement
       pdf(w)= 1 / (2*pi)
       cf. "global illumination compendium", dutre, eq 34, pp 19
     */

    float r, theta;
    float u1, u2;

    u1= (float) drand48();
    u2= (float) drand48();
    
    r= sqrtf(1.f - u2*u2);
    theta= 2.f* M_PI * u1;
    w[0]= r * cosf(theta);
    w[1]= r * sinf(theta);
    w[2]= u2;
    // w[2]= sqrtf(1.f - w[0]*w[0] - w[1]*w[1]);
    *pw= 1.f / (2.f * M_PI);
}

void dpoint_sample_direction(DPOINT *point, VEC w, float *pw)
{
    /* choisit une direction uniformement
     */
    
    VEC lw;
    
    sample_direction_local(point, lw, pw);
    dpoint_transform_vec(point, lw, w);
}

float dpoint_weight_direction(DPOINT *point, VEC w)
{
    return 1.f / (2.f * M_PI);
}

static void sample_cos_local(DPOINT *point, VEC w, float *pw)
{
    /* choisit une direction selon une densite proportionnelle au cosinus avec la normale
       pdf(w)= dot(n, w) / pi
       cf. "global illumination compendium", dutre, eq 35, pp 20
     */

    float r, theta;
    float u1, u2;

    u1= (float) drand48();
    u2= (float) drand48();
    
    r= sqrtf(1.f - u2);
    theta= 2.f* M_PI * u1;
    w[0]= r * cosf(theta);
    w[1]= r * sinf(theta);
    // w[2]= sqrtf(1.f - w[0]*w[0] - w[1]*w[1]);
    w[2]= sqrtf(u2);
    *pw= w[2] / M_PI;
}

void dpoint_sample_direction_cos(DPOINT *point, VEC w, float *pw)
{
    /* choisit une direction selon une densite proportionnelle au cosinus avec la normale
     */

    VEC lw;
    
    sample_cos_local(point, lw, pw);
    dpoint_transform_vec(point, lw, w);
}

float dpoint_weight_direction_cos(DPOINT *point, VEC w)
{
    /* calcule la probabilte de generer w selon une densite proportionnelle au cosinus
     */
    
    return vec3_dot(w, point->w) / vec3_length(w) / M_PI;
}

void dpoint_sample_direction_brdf(DPOINT *point, VEC w, float *pw)
{
    /* choisit une direction selon une densite proportionnelle a la brdf du point
     */
    
    // TODO
    dpoint_sample_direction_cos(point, w, pw);
}

float dpoint_weight_direction_brdf(DPOINT *point, VEC w)
{
    return dpoint_weight_direction_cos(point, w);
}

/* mesure sur les aires */
static int sample_direction_dWdA(DPOINT *point, VEC w, float pw, DPOINT *q, float *pa)
{
    if(dpoint_trace_ray(point, w, q) < 0)
        // pas de point visible dans la direction w
        return -1;
    
    // calcule la probabilite sur les aires
    *pa= pw * dpoint_get_dWdA(point, q);
    return 0;
}

int dpoint_sample(DPOINT *point, DPOINT *q, float *pa)
{
    /* choisit une direction uniformement
     */

    VEC w;
    float pw;
    
    dpoint_sample_direction(point, w, &pw);
    *pa= 0.f;
    return sample_direction_dWdA(point, w, pw, q, pa);
}

float dpoint_weight(DPOINT *point, DPOINT *q)
{
    VEC w;
    
    vec3_sub(w, q->p, point->p);
    vec3_norm(w, w);
    return dpoint_weight_direction(point, w) * dpoint_get_dWdA(point, q);
}

int dpoint_sample_cos(DPOINT *point, DPOINT *q, float *pa)
{
    /* choisit une direction selon une densite proportionnelle au cosinus avec la normale
     */
    
    VEC w;
    float pw;
    
    dpoint_sample_direction_cos(point, w, &pw);
    *pa= 0.f;
    return sample_direction_dWdA(point, w, pw, q, pa);
}

float dpoint_weight_cos(DPOINT *point, DPOINT *q)
{
    VEC w;
    
    vec3_sub(w, q->p, point->p);
    vec3_norm(w, w);
    return dpoint_weight_direction_cos(point, w) * dpoint_get_dWdA(point, q);   
}

int dpoint_sample_brdf(DPOINT *point, DPOINT *q, float *pa)
{
    /* choisit une direction selon une densite proportionnelle a la brdf du point
     */
    
    VEC w;
    float pw;
    
    dpoint_sample_direction_brdf(point, w, &pw);
    *pa= 0.f;
    return sample_direction_dWdA(point, w, pw, q, pa);
}

float dpoint_weight_brdf(DPOINT *point, DPOINT *q)
{
    VEC w;
    
    vec3_sub(w, q->p, point->p);
    vec3_norm(w, w);
    return dpoint_weight_direction_brdf(point, w) * dpoint_get_dWdA(point, q);   
}

