#include <stdlib.h>
#include <assert.h>
#include <math.h>

#include "vec.h"
#include "model.h"
#include "dpoint.h"
#include "ray.h"

// http://www.acm.org/jgt/papers/MollerTrumbore97/
static inline int intersect(VERTEX vert0, VERTEX vert1, VERTEX vert2, 
    VERTEX origine, VEC direction, float *t)
{
    VEC edge1, edge2, tvec, pvec, qvec;
    float det, inv_det;
    float u, v;

    /* find vectors for two edges sharing vert0 */
    vec3_sub(edge1, vert1, vert0);
    vec3_sub(edge2, vert2, vert0);

    /* begin calculating determinant - also used to calculate U parameter */
    vec3_cross(pvec, direction, edge2);

    /* if determinant is near zero, ray lies in plane of triangle */
    det= vec3_dot(edge1, pvec);

    if (det > -EPSILON && det < EPSILON)
        return 0;

    inv_det= 1.0f / det;

    /* calculate distance from vert0 to ray origin */
    vec3_sub(tvec, origine, vert0);

    /* calculate U parameter and test bounds */
    u= vec3_dot(tvec, pvec) * inv_det;
    if(u < 0.0f || u > 1.0f)
        return 0;

    /* prepare to test V parameter */
    vec3_cross(qvec, tvec, edge1);

    /* calculate V parameter and test bounds */
    v= vec3_dot(direction, qvec) * inv_det;
    if(v < 0.0f || u + v > 1.0f)
        return 0;

    /* calculate t, ray intersects triangle */
    *t= vec3_dot(edge2, qvec) * inv_det;
    return 1;
}


static inline int quad_intersect(MODEL *model, int face_id, 
    VERTEX origine, VEC direction, 
    float *t)
{
    FACE *face;
    float *v0;
    float *v1;
    float *v2;
    float *v3;

    face= model_get_face_ptr(model, face_id);
    v0= model_face_get_vertex_ptr(model, face, 0);
    v1= model_face_get_vertex_ptr(model, face, 1);
    v2= model_face_get_vertex_ptr(model, face, 2);
    v3= model_face_get_vertex_ptr(model, face, 3);

    return intersect(v0, v1, v2, origine, direction, t)
        || intersect(v0, v2, v3, origine, direction, t);
}

static inline int triangle_intersect(MODEL *model, int face_id, 
    VERTEX origine, VEC direction, float *t)
{
    FACE *face;
    float *v0;
    float *v1;
    float *v2;

    face= model_get_face_ptr(model, face_id);
    v0= model_face_get_vertex_ptr(model, face, 0);
    v1= model_face_get_vertex_ptr(model, face, 1);
    v2= model_face_get_vertex_ptr(model, face, 2);

    return intersect(v0, v1, v2, origine, direction, t);
}


static inline int model_intersect(MODEL *model, 
    VERTEX origine, VEC direction, 
    float *t)
{
    float tt, tmin;
    int f;
    int i;

    assert(model!=NULL);

    f= -1;
    tmin= *t;
    
    for(i= 0; i < model->faces_n; i++)
    {
        if((model->faces[i].n==3 
            && triangle_intersect(model, i, origine, direction, &tt)==1)
        || (model->faces[i].n==4 
            && quad_intersect(model, i, origine, direction, &tt)==1))
        {
            if(tt > EPSILON && tt < tmin)
            {
                f= i;
                tmin= tt;
            }
        }
    }

    if(f!=-1)
        *t= tmin;

    return f;
}

static inline int model_occlusion(MODEL *model, 
    VERTEX origine, VEC direction)
{
    float t;
    int i;
    
    assert(model!=NULL);
    
    for(i= 0; i < model->faces_n; i++)
    {
        if((model->faces[i].n==3 
            && triangle_intersect(model, i, origine, direction, &t)==1)
        || (model->faces[i].n==4 
            && quad_intersect(model, i, origine, direction, &t)==1))
        {
            if(t > 0.f && t < 1.f)
                return i;
        }
    }

    return -1;
}


int ray_intersect(RAY *ray, DPOINT *point)
{
    float t;
    int hit;
    
    assert(ray!=NULL);

    t= ray->tmax;
    hit= model_intersect(ray->model, ray->origin, ray->direction, &t);
    ray->hit= hit;
    if(hit>=0)
        ray->t= t;
    
    dpoint_init_ray(ray->model, point, ray);
    return hit;
}

int ray_occlusion(RAY *ray)
{
    assert(ray!=NULL);
    
    return model_occlusion(ray->model, ray->origin, ray->direction);
}

void ray_get_origin(RAY *ray, DPOINT *point)
{
    vec3_copy(point->p, ray->origin);
    dpoint_set_frame(point, ray->direction);
    point->face_id= -1;
    point->model= ray->model;
}


void ray_zero(MODEL *model, RAY *ray)
{
    assert(ray!=NULL);
    
    vec3_zero(ray->origin);
    vec3_zero(ray->direction);
    ray->tmin= 0.f;
    ray->tmax= 0.f;
    ray->t= 0.f;
    ray->hit= -1;
    ray->model= model;
}
    
void ray_init_direction(MODEL *model, RAY *ray, VEC origin, VEC direction)
{
    assert(ray!=NULL);
    
    vec3_copy(ray->origin, origin);
    vec3_copy(ray->direction, direction);
    ray->tmin= 0.f;
    ray->tmax= HUGE_VAL;
    ray->t= 0.f;
    ray->hit= -1;
    ray->model= model;
}

void ray_init_occlusion(MODEL *model, RAY *ray, VEC origin, VEC destination)
{
    assert(ray!=NULL);
    
    vec3_copy(ray->origin, origin);
    vec3_sub(ray->direction, destination, origin);
    ray->tmin= 0.f;
    ray->tmax= 1.f;
    ray->t= 0.f;
    ray->hit= -1;
    ray->model= model;
}
