/*
	base sur :
	"Terrain Generation Using Fluid Simulation"
	by Francis "DeathWish" Woodhouse     
	http://www.dwish.net
	http://www.gamedev.net/reference/articles/article2001.asp

	mailto:jciehl@bat710.univ-lyon1.fr
	avril 2004
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <math.h>

#include "img.h"
#include "tga.h"

int main(int argc, char **argv)
{
	IMG *out;
	IMG *in;
	float *hfield[3];
	float *tmp;

	float hmin= -50.f;
	float hmax= 50.f;

	const float c= 200.f;
	const float d= 10.f;
	const float mu= 20.f;
	const float t= 0.033f;
	const int imax= 200;

	const int largeur= 512;
	const int hauteur= 512;

	int hsize;
	int x, y;
	int i, k;

	//
	srand(time(NULL));

	out= new_img_data24(largeur, hauteur);

	hsize= largeur*hauteur;
	for(i= 0; i < 3; i++)
	{
		hfield[i]= (float *) malloc(sizeof(float) * hsize);
		assert(hfield[i]!=NULL);

		for(k= 0; k < hsize; k++)
			hfield[i][k]= 0.;
	}

	// hfield[0] iteration k+1
	// hfield[1]           k
	// hfield[2]           k-1

	// genere du bruit: iteration k et k-1
	for(y= 1; y < hauteur -1 ; y++)
	{
		for(x= 1; x < largeur -1; x++)
		{
			hfield[1][y*largeur + x]= (hmax - hmin) * (float) rand() / (float) RAND_MAX + hmin;
			hfield[2][y*largeur + x]= (hmax - hmin) * (float) rand() / (float) RAND_MAX + hmin;
		}
	}

	if(argc > 1)
	{
		// charge un terrain de reference pour l'iteration k-1
		in= lire_tga(argv[1]);

		assert(in!=NULL);

		for(y= 1; y < hauteur -1 ; y++)
			for(x= 1; x < largeur -1; x++)
				hfield[2][y*largeur + x]= (hmax - hmin) * (float) in->data[3*(y*in->largeur +x)] / 255.f + hmin;

		free_img(in);
		in= NULL;
	}

	// simulation
	for(i= 0; i < imax; i++)
	{
		printf("%03d/%03d\r", i+1, imax);
		fflush(stdout);

		for(y= 1; y < hauteur -1 ; y++)
		{
			for(x= 1; x < largeur -1; x++)
			{
				hfield[0][y*largeur + x]= 
					(4.f - 8.f* c*c * t*t / (d*d)) / (mu*t + 2.f) 
					* hfield[1][y*largeur + x]
					
					+ (mu*t - 2.f) / (mu*t + 2.f) 
					* hfield[2][y*largeur + x]
					
					+ (2.f* c*c * t*t / (d*d)) / (mu*t + 2.f) 
					* (hfield[1][y*largeur + x+1]
					+ hfield[1][y*largeur + x-1]
					+ hfield[1][(y+1)*largeur + x]
					+ hfield[1][(y-1)*largeur + x]);
			}
		}

		// permute les buffers
		tmp= hfield[2];
		hfield[2]= hfield[1];
		hfield[1]= hfield[0];
		hfield[0]= tmp;
	}


	// normalisation
	hmin= HUGE_VAL;
	hmax= -HUGE_VAL;

	for(i= 0; i < hsize; i++)
	{
		if(hfield[1][i] < hmin)
			hmin= hfield[1][i];

		if(hfield[1][i] > hmax)
			hmax= hfield[1][i];
	}

	// sauvegarde
	for(i= 0; i < hsize; i++)
	{
		out->data[3*i]=    color_clamp(255.f * (hfield[1][i] - hmin) 
			/ (hmax - hmin));
		out->data[3*i +1]= color_clamp(255.f * (hfield[1][i] - hmin)
			/ (hmax - hmin));
		out->data[3*i +2]= color_clamp(255.f * (hfield[1][i] - hmin)
			/ (hmax - hmin));
	}

	printf("\nheight.tga\n");
	ecrire_tga("height.tga", out);

	free_img(out);
	for(i= 0; i < 3; i++)
		free(hfield[i]);

	return 0;
}

