/*
	mesh.c
	
	manipulation de maillages de triangles
	mailto: jciehl@bat710.univ-lyon1.fr
	fevrier 2006
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "mesh.h"

/* construction du maillage d'apres le modele triangule
 */

static inline int find_edge(MESH *mesh, ELIST *edges, int v)
{
	int i;
	int e;
	
	for(i= 0; i < edges->n; i++)
	{
		e= edges->data[i];
		if(mesh->edges[e].v[0]==v || mesh->edges[e].v[1]==v)
			return e;
	}
	
	return -1;
}


static void build(MESH *mesh)
{
	MODEL *model;
	FACE *face;
	EFACE *eface;
	int i, j;
	int a0, a1;
	int e;
	int v0, v1;
	
	model= mesh->model;

	/* passe 1 : creer les faces et les listes de faces par sommet
	 */	
	mesh->faces_n= model->faces_n;
	array_add((void **) &mesh->faces, &mesh->faces_size, 
		mesh->faces_n,
		sizeof(EFACE), MODEL_FACE_MORE);

	mesh->vfaces_n= model->v_n;
	array_add((void **) &mesh->vfaces, &mesh->vfaces_size, 
		mesh->vfaces_n,
		sizeof(FLIST), MODEL_VERTEX_MORE);

	for(i= 0; i < mesh->vfaces_n; i++)
	{
		mesh->vfaces[i].data= NULL;
		mesh->vfaces[i].size= 0;
		mesh->vfaces[i].n= 0;
	}
	
	for(i= 0; i < mesh->faces_n; i++)
	{
		face= &model->faces[i];

		eface= &mesh->faces[i];
		eface->mat= face->mat;
		
		for(j= 0; j < 3; j++)
		{
			a0= face->attr + j;
			v0= model->attr[a0][gl_vertex];
			
			// inserer la face dans la liste de faces du sommet
			array_add((void **) &mesh->vfaces[v0].data, &mesh->vfaces[v0].size, 
				mesh->vfaces[v0].n,
				sizeof(int), MESH_FLIST_MORE);
			mesh->vfaces[v0].data[mesh->vfaces[v0].n]= i;
			mesh->vfaces[v0].n++;
		}
	}

	/* passe 2 : construction des aretes et des listes d'aretes par sommet
	 */
	mesh->vedges_n= model->v_n;
	array_add((void **) &mesh->vedges, &mesh->vedges_size, 
		mesh->vedges_n,
		sizeof(ELIST), MODEL_VERTEX_MORE);

	for(i= 0; i < mesh->vedges_n; i++)
	{
		mesh->vedges[i].data= NULL;
		mesh->vedges[i].size= 0;
		mesh->vedges[i].n= 0;
	}

	for(i= 0; i < mesh->faces_n; i++)
	{
		face= &model->faces[i];
		eface= &mesh->faces[i];
		
		for(j= 0; j < 3; j++)
		{
			a0= face->attr + j;
			v0= model->attr[a0][gl_vertex];
			a1= face->attr + (j + 1) % 3;
			v1= model->attr[a1][gl_vertex];

			e= find_edge(mesh, &mesh->vedges[v0], v1);
			if(e < 0)
				e= find_edge(mesh, &mesh->vedges[v1], v0);
			if(e < 0)
			{
				array_add((void **) &mesh->edges, &mesh->edges_size, 
					mesh->edges_n,
					sizeof(EDGE), MESH_EDGE_MORE);
				mesh->edges[mesh->edges_n].v[0]= v0;
				mesh->edges[mesh->edges_n].attr[0]= a0;
				mesh->edges[mesh->edges_n].v[1]= v1;
				mesh->edges[mesh->edges_n].attr[1]= a1;
				
				e= mesh->edges_n;
				mesh->edges_n++;
			}
			
			eface->edges[j]= e;
			if(mesh->edges[e].v[0]==v0)
				eface->flips[j]= 0;
			else
				eface->flips[j]= 1;
			
			array_add((void **) &mesh->vedges[v0].data, &mesh->vedges[v0].size, 
				mesh->vedges[v0].n,
				sizeof(ELIST), MESH_ELIST_MORE);
			mesh->vedges[v0].data[mesh->vedges[v0].n]= e;
			mesh->vedges[v0].n++;
			
			array_add((void **) &mesh->vedges[v1].data, &mesh->vedges[v1].size, 
				mesh->vedges[v1].n,
				sizeof(ELIST), MESH_ELIST_MORE);
			mesh->vedges[v1].data[mesh->vedges[v1].n]= e;
			mesh->vedges[v1].n++;
		}
	}
	
	mesh_print_stat(mesh);
}


MESH *mesh_new(MODEL *model)
{
	MESH *mesh;
	
	mesh= (MESH *) malloc(sizeof(MESH));
	assert(mesh!=NULL);
	
	if(model_is_triangles(model)==0)
		model_set_triangles(&model);
	
	mesh->model= model;
	
	mesh->faces= NULL;
	mesh->faces_size= 0;
	mesh->faces_n= 0;
	
	mesh->edges= NULL;
	mesh->edges_size= 0;
	mesh->edges_n= 0;
	
	mesh->vfaces= NULL;
	mesh->vfaces_size= 0;
	mesh->vfaces_n= 0;

	mesh->vedges= NULL;
	mesh->vedges_size= 0;
	mesh->vedges_n= 0;

	build(mesh);
	
	return mesh;
}

void mesh_free(MESH *mesh)
{
	if(mesh==NULL)
		return;
	
	if(mesh->faces!=NULL)
		free(mesh->faces);
	if(mesh->edges!=NULL);
		free(mesh->edges);
	if(mesh->vfaces!=NULL)
		free(mesh->vfaces);
	if(mesh->vedges!=NULL)
		free(mesh->vedges);
	
	free(mesh);
}

void mesh_print_stat(MESH *mesh)
{
	printf("mesh '%s' : %d (%d) faces, %d (%d) edges, %d (%d) vertex\n", 
		mesh->model->filename,
		mesh->faces_n, mesh->faces_size,
		mesh->edges_n, mesh->edges_size,
		mesh->model->v_n, mesh->model->v_size);
	printf("  vertex face list %d (%d)\n",
		mesh->vfaces_n, mesh->vfaces_size);
	printf("  vertex edge list %d (%d)\n",
		mesh->vedges_n, mesh->vedges_size);
}

MODEL *mesh_build_model(MESH *mesh)
{
	MODEL *model;
	MODEL *build;
	EFACE *eface;
	FACE *face;
	int *vertex, *tex, *norm, *mat;
	int a0;
	int i, j;
	
	model= mesh->model;
	build= model_new(model->filename);

	vertex= (int *) malloc(sizeof(int) * model->v_n);
	assert(vertex!=NULL);
	for(i= 0; i < model->v_n; i++)
		vertex[i]= -1;
	
	tex= (int *) malloc(sizeof(int) * model->tex_n);
	assert(tex!=NULL);
	for(i= 0; i < model->tex_n; i++)
		tex[i]= -1;
	
	norm= (int *) malloc(sizeof(int) * model->norm_n);
	assert(norm!=NULL);
	for(i= 0; i < model->norm_n; i++)
		norm[i]= -1;

	mat= (int *) malloc(sizeof(int) * model->mat_n);
	assert(mat!=NULL);
	for(i= 0; i < model->mat_n; i++)
		mat[i]= -1;
	
	//
	for(i= 0; i < mesh->faces_n; i++)
	{
		eface= &mesh->faces[i];
		
		array_add((void **) &build->faces, &build->faces_size,
			build->faces_n,
			sizeof(FACE), MODEL_FACE_MORE);
		
		face= &build->faces[build->faces_n];
		face->attr= build->attr_n;
		face->norm= -1;

		if(mat[eface->mat]==-1)
		{
			mat[eface->mat]= build->mat_n;

			array_add((void **) &build->mat, &build->mat_size, 
				build->mat_n,
				sizeof(MATERIAL), MODEL_MATERIAL_MORE);
			mat_copy(&build->mat[build->mat_n], &model->mat[eface->mat]);
			build->mat_n++;
		}

		face->mat= mat[eface->mat];
		
		for(j= 0; j < 3; j++)
		{
			if(eface->flips[j]==0)
				a0= mesh->edges[eface->edges[j]].attr[0];
			else
				a0= mesh->edges[eface->edges[j]].attr[1];
			
			if(vertex[model->attr[a0][gl_vertex]]==-1)
			{
				vertex[model->attr[a0][gl_vertex]]= build->v_n;
				
				array_add((void **) &build->v, &build->v_size,
					build->v_n,
					sizeof(VERTEX), MODEL_VERTEX_MORE);
				vertex_copy(&build->v[build->v_n], &model->v[model->attr[a0][gl_vertex]]);
				build->v_n++;
			}
			
			if(model->attr[a0][gl_tex] >= 0 
			&& tex[model->attr[a0][gl_tex]]==-1)
			{
				tex[model->attr[a0][gl_tex]]= build->tex_n;
				
				array_add((void **) &build->tex, &build->tex_size,
					build->tex_n, 
					sizeof(VERTEXT), MODEL_VERTEX_MORE);
				tex_copy(&build->tex[build->tex_n], &model->tex[model->attr[a0][gl_tex]]);
				build->tex_n++;
			}
			
			if(model->attr[a0][gl_norm] >= 0
			&& norm[model->attr[a0][gl_norm]]==-1)
			{
				norm[model->attr[a0][gl_norm]]= build->norm_n;
				
				array_add((void **) &build->norm, &build->norm_size,
					build->norm_n,
					sizeof(VERTEXN), MODEL_VERTEX_MORE);
				norm_copy(&build->norm[build->norm_n], &model->norm[model->attr[a0][gl_norm]]);
				build->norm_n++;
			}
			
			//
			array_add((void **) &build->attr, &build->attr_size,
				build->attr_n,
				sizeof(ATTR), MODEL_ATTR_MORE);
			build->attr[build->attr_n][gl_vertex]= vertex[model->attr[a0][gl_vertex]];
			if(model->attr[a0][gl_tex] >= 0)
				build->attr[build->attr_n][gl_tex]= tex[model->attr[a0][gl_tex]];
			else
				build->attr[build->attr_n][gl_tex]= -1;
			if(model->attr[a0][gl_norm] >= 0)
				build->attr[build->attr_n][gl_norm]= norm[model->attr[a0][gl_norm]];
			else
				build->attr[build->attr_n][gl_norm]= -1;
			
			build->attr_n++;
		}
		
		face->n= 3;
		build->faces_n++;
	}

	free(vertex);
	free(tex);
	free(norm);
	free(mat);
	
	return build;
}

