/*
	scenes maya obj 3, ascii, triangulees
	
	mailto:jciehl@bat710.univ-lyon1.fr

	fevrier 2004 : premiere version
	fevrier 2006 : 
		stockage des attributs compatible avec vertex_buffer_objects
		model_save_obj
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include <GL/glut.h>

#include "model.h"
#include "model_clean.h"
#include "maya_obj.h"
#include "vec.h"

// lire un mot dans le fichier modele
int get_token(FILE *in, char *tmp, int tmp_size)
{
	char c;
	int i= 0;

	// sauter les blancs
	c= fgetc(in);
	while(c!=EOF && (c==' ' || c=='\t'))
		c= fgetc(in);

	// lire tous les caracteres alphanumeriques
	while(c!=EOF && ISTOKEN(c))
	{
		tmp[i++]= c;
		
		if(i>=tmp_size)
		{
			printf("  -- line length > %d\n", tmp_size);
			return EOF;
		}
		
		c= fgetc(in);
	}

	// separateurs
	if(c=='#')
	{
		if(i==0)
			tmp[i++]= c;
		else
			ungetc(c, in);
	}
	else if(c=='/')
		tmp[i++]= c;
	
	// terminer la chaine
	tmp[i]= 0;

	// indiquer la fin de la ligne ou du fichier
	if(i==0)
		return c;
	
	// forcer une fin de ligne avant la fin du fichier
	if(c==EOF)
		ungetc('\n', in);
		
	else if(c=='\n')
		ungetc(c, in);

	return 0;
}


int obj_load_attr(char *tmp, int *attr, int *attr_n)
{
	if(strcmp(tmp, "/")==0)
		return 0;
		
	if(sscanf(tmp, "%d", attr)==0)
	{
		printf("  -- missing value\n");
		return -1;
	}
	
	if(*attr < 0)
		*attr= *attr_n + *attr;
	else
		*attr-= 1;
		
	if(*attr < 0 || *attr >= *attr_n)
		return -1;
	
	return 0;
}

// lire le fichier modele
int model_load_obj(MODEL **pmodel, char *filename)
{
	char tmp[1024];

	int *attribs_n[3];
		
	int groups[16];
	int groups_n;
	int g;
	int mat;
	
	MODEL *model;
	GROUP *group;
	FACE *face;
		
	FILE *in;
	int code= 0;

	int i, j;
	
	in= fopen(filename, "rt");
	if(in==NULL)
		return -1;
	
	model= model_new(filename);
	
	// attributs en cours de lecture
	attribs_n[gl_vertex]= &model->v_n;
	attribs_n[gl_tex]= &model->tex_n;
	attribs_n[gl_norm]= &model->norm_n;
	groups_n= 0;
	mat= -1;
	
	while(code==0 && get_token(in, tmp, sizeof(tmp))!=EOF)
	{
		// commentaire
		if(strcmp(tmp, "#")==0)
		{
			// lire jusqu'a la fin de la ligne
			while(get_token(in, tmp, sizeof(tmp))!='\n')
				{;}
		}
		
		// groupe ident
		else if(strcmp(tmp, "g")==0 || strcmp(tmp, "s")==0)
		{
			groups_n= 0;
			
			while(get_token(in, tmp, sizeof(tmp))!='\n')
			{
			#if 0
				g= group_find(model, tmp);
				if(g==-1)
				{	
					array_add((void **) &model->groups, &model->groups_size, 
						model->groups_n, 
						sizeof(GROUP), MODEL_GROUP_MORE);
					g= model->groups_n;
					
					model->groups[model->groups_n].name= strdup(tmp);
					assert(model->groups!=NULL);
					model->groups[model->groups_n].faces= NULL;
					model->groups[model->groups_n].faces_size= 0;
					model->groups[model->groups_n].faces_n= 0;
					model->groups_n++;
				}
				
				groups[0]= g;
				groups_n= 1;
				
				assert(groups_n < 16);
			#endif
			}
		}
		
		// matiere ident
		else if(strcmp(tmp, "usemtl")==0)
		{
			get_token(in, tmp, sizeof(tmp));

			mat= material_find(model, tmp);
			if(mat==-1)
			{
				array_add((void **) &model->mat, &model->mat_size, 
					model->mat_n, 
					sizeof(MATERIAL), MODEL_MATERIAL_MORE);
				mat= model->mat_n;
				
				model->mat[model->mat_n].name= strdup(tmp);
				
				model->mat_n++;
			}
		}
		
		// vertex x y z [w]
		else if(strcmp(tmp, "v")==0)
		{
			array_add((void **) &model->v, &model->v_size, 
				model->v_n, 
				sizeof(VERTEX), MODEL_VERTEX_MORE);
			
			for(i= 0; get_token(in, tmp, sizeof(tmp))!='\n'; i++)
				if(i<4 && sscanf(tmp, "%f", &model->v[model->v_n][i])==0)
				{
					printf(" -- missing value\n");
					code= -1;
					break;
				}

			for(; i<4; i++)
				model->v[model->v_n][i]= 0.f;
			
			model->v_n++;
		}
		
		// vertex param 
		else if(strcmp(tmp, "vp")==0)
		{
			array_add((void **) &model->param, &model->param_size, 
				model->param_n, 
				sizeof(VERTEXP), MODEL_VERTEX_MORE);
			
			for(i= 0; get_token(in, tmp, sizeof(tmp))!='\n'; i++)
				if(i < 2 && sscanf(tmp, "%f", &model->param[model->param_n][i])==0)
				{
					printf(" -- missing value\n");
					code= -1;
					break;
				}
			
			for(; i < 2; i++)
				model->param[model->param_n][i]= 0.f;
			
			model->param_n++;
		}
		
		// vertex normal
		else if(strcmp(tmp, "vn")==0)
		{
			array_add((void **) &model->norm, &model->norm_size, 
				model->norm_n, 
				sizeof(VERTEXN), MODEL_VERTEX_MORE);
			
			for(i= 0; get_token(in, tmp, sizeof(tmp))!='\n'; i++)
				if(i < 3 && sscanf(tmp, "%f", &model->norm[model->norm_n][i])==0)
				{
					printf(" -- missing value\n");
					code= -1;
					break;
				}
		
			for(; i < 3; i++)
				model->norm[model->norm_n][i]= 0.f;
			
			model->norm_n++;
		}

		// vertex texture
		else if(strcmp(tmp, "vt")==0)
		{
			array_add((void **) &model->tex, &model->tex_size, 
				model->tex_n, 
				sizeof(VERTEXT), MODEL_VERTEX_MORE);
			
			for(i= 0; get_token(in, tmp, sizeof(tmp))!='\n'; i++)
				if(i < 2 && sscanf(tmp, "%f", &model->tex[model->tex_n][i])==0)
				{
					printf(" -- missing value\n");
					code= -1;
					break;
				}

			for(; i < 2; i++)
				model->tex[model->tex_n][i]= 0.f;
			
			model->tex_n++;
		}
		
		// face
		else if(strcmp(tmp, "f")==0)
		{
			array_add((void **) &model->faces, &model->faces_size, 
				model->faces_n, 
				sizeof(FACE), MODEL_FACE_MORE);
			
			face= &model->faces[model->faces_n];
			face->attr= model->attr_n;
			face->n= 0;
			face->norm= -1;
			face->mat= mat;
			assert(mat!=-1);

			for(i= 0; get_token(in, tmp, sizeof(tmp))!='\n'; i++)
			{
				array_add((void **) &model->attr, &model->attr_size, 
					model->attr_n, 
					sizeof(ATTR), MODEL_ATTR_MORE);
				for(j= 0; j < 3; j++)
					model->attr[model->attr_n][j]= -1;
				
				// lecture des parametres des sommets
				if(obj_load_attr(tmp, &model->attr[model->attr_n][gl_vertex], attribs_n[gl_vertex]) < 0)
					break;
				if(tmp[strlen(tmp) -1]!='/' || get_token(in, tmp, sizeof(tmp))=='\n')
					goto next_face;
				
				if(obj_load_attr(tmp, &model->attr[model->attr_n][gl_tex], attribs_n[gl_tex]) < 0)
					break;
				if(tmp[strlen(tmp) -1]!='/' || get_token(in, tmp, sizeof(tmp))=='\n')
					goto next_face;
				
				if(obj_load_attr(tmp, &model->attr[model->attr_n][gl_norm], attribs_n[gl_norm]) < 0)
					break;

			next_face:
				face->n++;
				model->attr_n++;
			}
			
			// ajouter la face aux groupes courants
			for(g= 0; g < groups_n; g++)
			{
				group= &model->groups[groups[g]];
				array_add((void **) &group->faces, &group->faces_size, 
					group->faces_n, 
					sizeof(int), GROUP_FACE_MORE);
				group->faces[group->faces_n++]= model->faces_n;
			}
			
			model->faces_n++;
		}
	}
	
	fclose(in);

	//	
	model_print_stat(model);
	
	*pmodel= model;
	return code;
}

// enregistrer le fichier modele
int model_save_obj(MODEL *model, char *fname)
{
	FILE *out;
	int mat;
	int i, j;
	
	if(model==NULL)
		return 0;
	
	out= fopen(fname, "wt");
	if(out==NULL)
	{
		printf("\n  -- unable to create file '%s'", fname);
		return -1;
	}
	
	//
	fprintf(out, "g default\n");
	
	//
	for(i= 0; i < model->v_n; i++)
		fprintf(out, "v %f %f %f\n", 
			model->v[i][0], model->v[i][1], model->v[i][2]);
	
	for(i= 0; i < model->tex_n; i++)
		fprintf(out, "vt %f %f\n", 
			model->tex[i][0], model->tex[i][1]);
	
	for(i= 0; i < model->norm_n; i++)
		fprintf(out, "vn %f %f %f\n", 
			model->norm[i][0], model->norm[i][1], model->norm[i][2]);
	
	mat= -1;
	for(i= 0; i < model->faces_n; i++)
	{
		if(model->faces[i].attr < 0)
			continue;

		if(mat!=model->faces[i].mat)
		{
			mat= model->faces[i].mat;
			fprintf(out, "usemtl %s\n", model->mat[model->faces[i].mat].name);
		}
		
		fprintf(out, "f ");
		for(j= 0; j < model->faces[i].n; j++)
		{
			if(model->attr[model->faces[i].attr + j][gl_vertex] < 0)
				goto error;
			fprintf(out, "%d", model->attr[model->faces[i].attr + j][gl_vertex] +1);
			
			if(model->attr[model->faces[i].attr + j][gl_tex] >= 0)
				fprintf(out, "/%d", model->attr[model->faces[i].attr + j][gl_tex] +1);
			else if(model->attr[model->faces[i].attr + j][gl_norm] >= 0)
				fprintf(out, "/");

			if(model->attr[model->faces[i].attr + j][gl_norm] >= 0)
				fprintf(out, "/%d", model->attr[model->faces[i].attr + j][gl_norm] +1);
			
			fprintf(out, " ");
		}
		
		fprintf(out, "\n");
	}
	
	fclose(out);
	return 0;
	
error:
	fclose(out);
	printf("\n  -- error in model '%s'", fname);
	return -1;
}
