/*
	materiaux elementaires
	
	
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>

#include "struct.h"
#include "matiere.h"
#include "vec.h"

MATIERE *matiere(float du, float dd, float dd_n, float sp, float n, float a)
{
	MATIERE *m= (MATIERE *) malloc(sizeof(MATIERE));
	assert(m!=NULL);
	
	m->k_du= du;
	m->k_dd= dd;
	m->dd_n= dd_n;
	m->k_sp= sp;
	
	m->n= n;
	m->a= a;
	
	return m;
}


MATIERE *matiere_diffus(float du)
{
	return matiere(du, 0., 1., 0., 1., 1.);
}

MATIERE *matiere_diffus_directionnel(float du, float dd, float dd_n)
{
	return matiere(du, dd, dd_n, 0., 1., 1.);
}

MATIERE *matiere_speculaire(float dd, float dd_n, float sp)
{
	return matiere(0., dd, dd_n, sp, 1., 1.);
}

MATIERE *matiere_miroir(float sp)
{
	return matiere(0., 0., 0., sp, 1., 1.);
}

MATIERE *matiere_transparent(float n, float a)
{
	return matiere(0., 0., 0., 0., n, a);
}

// valeur de la brdf associee
float matiere_brdf(MATIERE *matiere, float direction[3], float normale[3], float v[3])
{
	float v_refl[3];
	float r= 0.;
	
	assert(matiere!=NULL);
	
	if(matiere->k_du > EPSILON)
		r+= vec3_dot(normale, v) * matiere->k_du;
	
	if(matiere->k_dd > EPSILON || matiere->k_sp > EPSILON)
	{
		miroir_rayon(matiere, direction, normale, v_refl);

		if(matiere->k_dd > EPSILON)
			r+= pow(vec3_dot(v_refl, v), matiere->dd_n) * matiere->k_dd;
			
		if(matiere->k_sp > EPSILON)
			r+= pow(vec3_dot(v_refl, v), 512.) * matiere->k_sp;
	}

	return r;
}

// construit un repere local 
void repere_local(float u[3], float v[3], float w[3], float normale[3])
{
	float t[3];
	float mf, tf;
	int m;
	int i;
	
	vec3_copy(w, normale);	
	vec3_copy(t, normale);
	
	m= 0;
	mf= fabs(t[0]);
	for(i= 1; i<3; i++)
	{
		tf= fabs(t[i]); 
		if(tf < mf)
		{
			m= i;
			mf= tf;
		}
	}
	
	t[m]= 1.;

	vec3_cross(u, t, w);
	vec3_norm(u, u);

	vec3_cross(v, w, u);
	vec3_norm(v, v);
}


// passage repere local vers global
void repere_local_vers_global(float u[3], float v[3], float w[3], float local[3], float global[3])
{
	global[0]= u[0]*local[0] + v[0]*local[1] + w[0]*local[2];
	global[1]= u[1]*local[0] + v[1]*local[1] + w[1]*local[2];
	global[2]= u[2]*local[0] + v[2]*local[1] + w[2]*local[2];
}

// genere une direction dans l'hemisphere positif avec une densite proportionelle au cosinus 
// pdf(v)= dot(n, v) / pi
// cf. "global illumination compendium", dutre, eq 35, pp 20
// cf. lrt, pharr
void matiere_rayon(MATIERE *matiere, float normale[3], float v_refl[3], float *poids)
{
	float d[3];
	float u[3], v[3], w[3];
	float r, theta;
	float u1, u2;

	// repere local 
	u1= (float) rand() / (float) RAND_MAX;
	u2= (float) rand() / (float) RAND_MAX;
	
	r= sqrt(u2);
	theta= 2.*M_PI * u1;
	d[0]= r * cos(theta);
	d[1]= r * sin(theta);
	d[2]= 1. - d[0]*d[0] - d[1]*d[1];
	if(d[2] < EPSILON)
		d[2]= 0.;
	else
		d[2]= sqrt(d[2]);

	// repere global
	repere_local(u, v, w, normale);
	repere_local_vers_global(u, v, w, d, v_refl);

	*poids= vec3_dot(w, v_refl) / M_PI;
}


void miroir_rayon(MATIERE *matiere, float direction[3], float normale[3], float v[3])
{
	float d;
	
	d= -2. * vec3_dot(direction, normale);
	vec3_add_mul_const(v, direction, d, normale);
	vec3_norm(v, v);
}

