/*
	outils format bmp 256 couleurs + 24 bits

	jciehl
	octobre 2003

	fix: la taille des lignes *est* un multiple de 4
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "bmp.h"

void debug_bmp_entete(BMP_HEADER *bmp)
{
	printf("header %dB\n", sizeof(BMP_HEADER));
	printf("type %hu '%c%c'\n", bmp->type, LO(bmp->type), HI(bmp->type));
	printf("size %u\n", bmp->size);
	printf("offset %u\n", bmp->offset);
}

int lire_bmp_entete(FILE *in, BMP_HEADER *bmp)
{
	unsigned char *header;
	size_t n;

	header= (unsigned char  *) malloc(14*sizeof(char));
	assert(header);
	n= fread(header, 14, 1, in);
	if(n==1)
	{
		bmp->type= LOHI(header[0], header[1]);
		bmp->size= LOHI_int(header[2], header[3], header[4], header[5]);
		// header[6..9] : reserved
		bmp->offset= LOHI_int(header[10], header[11], header[12], header[13]);

		free(header);

		if(LO(bmp->type)!='B' || HI(bmp->type)!='M')
		{
			printf("-- format BMP non reconnu (header BM)\n");
			return 0;
		}

		// debug_bmp_entete(bmp);
		return 1;
	}

	free(header);
	return 0;
}

void debug_bmp_info(BMP_INFO *bmp)
{
	printf("info %dB\n", sizeof(BMP_INFO));
	printf("size %u\n", bmp->size);
	printf("width %d, height %d\n", bmp->width, bmp->height);
	printf("planes %hu\n", bmp->planes);
	printf("bits %hu\n", bmp->bits);
	printf("compression %u\n", bmp->compression);
	printf("imagesize %u\n", bmp->imagesize);
	printf("xres %d, yres %d\n", bmp->xresolution, bmp->yresolution);
	printf("ncolors %u\n", bmp->ncolours);
	printf("importantcolors %u\n", bmp->importantcolours);
}

int lire_bmp_info(FILE *in, BMP_INFO *bmp)
{
	unsigned char *info;
	size_t n;

	info= (unsigned char *) malloc(sizeof(char[40]));
	assert(info);
	n= fread(info, 40, 1, in);
	if(n==1)
	{
		bmp->size=  LOHI_int(info[0], info[1], info[2], info[3]);
		bmp->width= LOHI_int(info[4], info[5], info[6], info[7]);
		bmp->height= LOHI_int(info[8], info[9], info[10], info[11]);
		bmp->planes= LOHI(info[12], info[13]);
		bmp->bits= LOHI(info[14], info[15]);
		bmp->compression= LOHI_int(info[16], info[17], info[18], info[19]);
		bmp->imagesize= LOHI_int(info[20], info[21], info[22], info[23]);
		bmp->xresolution= LOHI_int(info[24], info[25], info[26], info[27]);
		bmp->yresolution= LOHI_int(info[28], info[29], info[30], info[31]);
		bmp->ncolours= LOHI_int(info[32], info[33], info[34], info[35]);
		bmp->importantcolours= LOHI_int(info[36], info[37], info[38], info[39]);

		free(info);
		// debug_bmp_info(bmp);
		return 1;
	}

	free(info);
	return 0;
}

// determine le multiple de 4 superieur ou egal a n
int aligne32b(int n)
{
	if((n % 4)==0)
		return n;
	else
		return (n/4 +1) * 4;
}


size_t lire_aligne32b_256(FILE *in, IMG *img)
{
	size_t n, size;
	unsigned char *tmp;
	unsigned char *data;
	int largeur;
	int y;

	// determine l'alignement (multiple de 32 bits superieur)
	if(img->palette_n==256)
	{
		img->alignement= aligne32b(img->largeur * sizeof(unsigned char));
		largeur= img->largeur * sizeof(unsigned char);
	}
	else
		return 0;
	
	tmp= (unsigned char *) malloc(img->alignement);
	assert(tmp);

	printf("aligne32b: %d pixels   alignement %d octets largeur %d octets\n", img->largeur, img->alignement, largeur);
	
	size= 0;
	data= img->data;
	for(y= 0; y < img->hauteur; y++)
	{	
		n= fread(tmp, sizeof(unsigned char), img->alignement, in);
		size+= n;
		if(n!= (size_t) img->alignement)
			break;
		
		memcpy(data , tmp, largeur);
		data+= largeur;
	}

	free(tmp);
	
	return y*img->largeur;
}


size_t lire_aligne32b_rgb(FILE *in, IMG *img)
{
	size_t n, size;
	unsigned char *tmp;
	unsigned char *data;
	int largeur;
	int i, y;
	unsigned char r, g, b;

	// determine l'alignement (multiple de 32 bits superieur)
	if(img->palette_n==0)
	{
		img->alignement= aligne32b(img->largeur * sizeof(unsigned char[3]));
		largeur= img->largeur * sizeof(unsigned char[3]);
	}
	else
		return 0;
	
	tmp= (unsigned char *) malloc(img->alignement);
	assert(tmp);

	size= 0;
	data= img->data;
	for(y= 0; y < img->hauteur; y++)
	{	
		n= fread(tmp, sizeof(unsigned char), img->alignement, in);
		size+= n;
		if(n!= (size_t) img->alignement)
			break;
		
		/* place les triplets rgb dans le bon ordre */
		for(i= 0; i<largeur; i+= 3)
		{
			data[i]=   tmp[i+2];
			data[i+1]= tmp[i+1];
			data[i+2]= tmp[i];
		}
		
		data+= largeur;
	}

	free(tmp);
	
	return y*img->largeur;
}


IMG *lire_bmp(char *fname)
{
	BMP_HEADER bmpheader;
	BMP_INFO bmpinfo;
	FILE *in;
	IMG *img;
	size_t n;
	int size;
	int i;
	int nc;
	unsigned char r, v, b;

	in= fopen(fname, "rb");
	if(in==NULL)
	{
		printf("-- erreur de lecture %s\n", fname);
		return NULL;
	}

	img= NULL;
	if(lire_bmp_entete(in, &bmpheader)==1)
	{
		if(lire_bmp_info(in, &bmpinfo)==1)
		{
			if(bmpinfo.bits==8)
			{
				img= new_img_palette256(bmpinfo.width, bmpinfo.height);
				
				if(bmpinfo.ncolours>0 && bmpinfo.ncolours <= (unsigned int) img->palette_n)
					nc= bmpinfo.ncolours;
				else
					nc= 256;

				n= fread(img->palette, sizeof(unsigned char[4]), nc, in);
				if(n!= (size_t) nc)
				{
					printf("-- erreur de lecture %s\n", fname);

					free_img(img);
					return NULL;
				}

				// reordonne les couleurs de la palette : r, g, b
				size= nc*4;
				for(i= 0; i<size; i+= 4)
				{
					r= img->palette[i+2];
					v= img->palette[i+1];
					b= img->palette[i];

					img->palette[i]= r;
					img->palette[i+1]= v;
					img->palette[i+2]= b;
				}
			
				// efface les couleurs non utilisees	
				size= img->palette_n*4;
				for(; i<size; i++)
					img->palette[i]= 0;

				// lecture
				// printf("image %dx%d\n", bmpinfo.width, bmpinfo.height);
				
				n= lire_aligne32b_256(in, img);
				if(n!= (size_t) bmpinfo.width*bmpinfo.height)
				{
					printf("\n-- lecture incomplete (%d/%d pixels): %s\n", n, img->largeur*img->hauteur, fname);

					size= sizeof(unsigned char) * img->largeur*img->hauteur;
					for(i= (int) n * sizeof(unsigned char); i<size; i++)
						img->data[i]= 0;
				}
			}

			else if(bmpinfo.bits==24)
			{
				img= new_img_data24(bmpinfo.width, bmpinfo.height);

				// lecture
				// printf("image %dx%d\n", bmpinfo.width, bmpinfo.height);

				n= lire_aligne32b_rgb(in, img);
				if(n!= (size_t) bmpinfo.width*bmpinfo.height)
				{
					printf("\n-- lecture incomplete (%d/%d pixels): %s\n", n, img->largeur*img->hauteur, fname);

					size= sizeof(unsigned char [3]) * img->largeur*img->hauteur;
					for(i= (int) n * sizeof(unsigned char[3]); i<size; i++)
						img->data[i]= 0;
				}
			}
			else
				printf("-- format BMP non reconnu (info.bits)\n");
		}
	}

	return img;
}

int ecrire_bmp_entete(FILE *out, BMP_HEADER *bmp)
{
	unsigned char *header;
	size_t n;

	// debug_bmp_entete(bmp);
	
	header= (unsigned char *) malloc(sizeof(unsigned char[14]));
	assert(header);

	header[0]= LO(bmp->type);
	header[1]= HI(bmp->type);
	header[2]= int_LOLO(bmp->size);
	header[3]= int_LOHI(bmp->size);
	header[4]= int_HILO(bmp->size);
	header[5]= int_HIHI(bmp->size);
	header[6]= 0;
	header[7]= 0;
	header[8]= 0;
	header[9]= 0;
	header[10]= int_LOLO(bmp->offset);
	header[11]= int_LOHI(bmp->offset);
	header[12]= int_HILO(bmp->offset);
	header[13]= int_HIHI(bmp->offset);
		
	n= fwrite(header, sizeof(unsigned char[14]), 1, out);

	free(header);

	return (n == (size_t) 1);
}	

int ecrire_bmp_info(FILE *out, BMP_INFO *bmp)
{
	unsigned char *info;
	size_t n;
	
	// debug_bmp_info(bmp);
	
	info= (unsigned char *) malloc(sizeof(unsigned char[40]));
	assert(info);

	info[0]= int_LOLO(bmp->size);
	info[1]= int_LOHI(bmp->size);
	info[2]= int_HILO(bmp->size);
	info[3]= int_HIHI(bmp->size);
	info[4]= int_LOLO(bmp->width);
	info[5]= int_LOHI(bmp->width);
	info[6]= int_HILO(bmp->width);
	info[7]= int_HIHI(bmp->width);
	info[8]= int_LOLO(bmp->height);
	info[9]= int_LOHI(bmp->height);
	info[10]= int_HILO(bmp->height);
	info[11]= int_HIHI(bmp->height);
	info[12]= LO(bmp->planes);
	info[13]= HI(bmp->planes);
	info[14]= LO(bmp->bits);
	info[15]= HI(bmp->bits);
	info[16]= int_LOLO(bmp->compression);
	info[17]= int_LOHI(bmp->compression);
	info[18]= int_HILO(bmp->compression);
	info[19]= int_HIHI(bmp->compression);
	info[20]= int_LOLO(bmp->imagesize);
	info[21]= int_LOHI(bmp->imagesize);
	info[22]= int_HILO(bmp->imagesize);
	info[23]= int_HIHI(bmp->imagesize);
	info[24]= int_LOLO(bmp->xresolution);
	info[25]= int_LOHI(bmp->xresolution);
	info[26]= int_HILO(bmp->xresolution);
	info[27]= int_HIHI(bmp->xresolution);
	info[28]= int_LOLO(bmp->yresolution);
	info[29]= int_LOHI(bmp->yresolution);
	info[30]= int_HILO(bmp->yresolution);
	info[31]= int_HIHI(bmp->yresolution);
	info[32]= int_LOLO(bmp->ncolours);
	info[33]= int_LOHI(bmp->ncolours);
	info[34]= int_HILO(bmp->ncolours);
	info[35]= int_HIHI(bmp->ncolours);
	info[36]= int_LOLO(bmp->importantcolours);
	info[37]= int_LOHI(bmp->importantcolours);
	info[38]= int_HILO(bmp->importantcolours);
	info[39]= int_HIHI(bmp->importantcolours);
	
	n= fwrite(info, sizeof(unsigned char[40]), 1, out);

	free(info);

	return (n == (size_t) 1);
}


size_t ecrire_aligne32b_256(FILE *in, IMG *img)
{
	size_t n, size;
	unsigned char *tmp;
	unsigned char *data;
	int largeur;
	int y;

	// determine l'alignement (multiple de 32 bits superieur)
	if(img->palette_n==256)
	{
		img->alignement= aligne32b(img->largeur * sizeof(unsigned char));
		largeur= img->largeur * sizeof(unsigned char);
	}
	else
		return 0;
	
	tmp= (unsigned char *) malloc(img->alignement);
	assert(tmp);

	memset(tmp, 0, img->alignement);
	
	size= 0;
	data= img->data;
	for(y= 0; y < img->hauteur; y++)
	{	
		memcpy(tmp, data, largeur);

		n= fwrite(tmp, sizeof(unsigned char), img->alignement, in);
		size+= n;
		if(n!= (size_t) img->alignement)
			break;
		
		data+= largeur;
	}

	free(tmp);
	
	return y*img->largeur;
}


size_t ecrire_aligne32b_rgb(FILE *in, IMG *img)
{
	size_t n, size;
	unsigned char *tmp;
	unsigned char *data;
	int largeur;
	int i, y;
	unsigned char r, g, b;

	// determine l'alignement (multiple de 32 bits superieur)
	if(img->palette_n==0)
	{
		img->alignement= aligne32b(img->largeur * sizeof(unsigned char[3]));
		largeur= img->largeur * sizeof(unsigned char[3]);
	}
	else
		return 0;
	
	tmp= (unsigned char *) malloc(img->alignement);
	assert(tmp);

	memset(tmp, 0, img->alignement);

	size= 0;
	data= img->data;
	for(y= 0; y < img->hauteur; y++)
	{	
		/* place les triplets rgb dans le bon ordre */
		for(i= 0; i<largeur; i+= 3)
		{
			tmp[i]=   data[i+2];
			tmp[i+1]= data[i+1];
			tmp[i+2]= data[i];
		}

		n= fwrite(tmp, sizeof(unsigned char), img->alignement, in);
		size+= n;
		if(n!= (size_t) img->alignement)
			break;
		
		data+= largeur;
	}

	free(tmp);
	
	return y*img->largeur;
}


int ecrire_bmp(char *fname, IMG *img)
{
	BMP_HEADER bmphead;
	BMP_INFO bmpinfo;
	size_t n;
	unsigned char *tmp;
	FILE *out;
	int largeur;
	int code;
	int y, i;
	unsigned char r, v, b;

	bmphead.type= LOHI('B', 'M');
	bmphead.reserved1= 0;
	bmphead.reserved2= 0;

	bmpinfo.size= 40;
	bmpinfo.width= img->largeur;
	bmpinfo.height= img->hauteur;
	bmpinfo.planes= 1;

	if(img->palette_n==256)
	{
		bmphead.offset= 14 + 40 + 256*sizeof(unsigned char[4]);

		img->alignement= aligne32b(img->largeur * sizeof(unsigned char));
		bmpinfo.imagesize= img->alignement*img->hauteur;
		bmphead.size= bmphead.offset + bmpinfo.imagesize;
		bmpinfo.bits= 8;
	}
	else
	{
		bmphead.offset= 14 + 40;
		img->alignement= aligne32b(img->largeur * sizeof(unsigned char[3]));
		bmpinfo.imagesize= img->alignement*img->hauteur;
		bmphead.size= bmphead.offset + bmpinfo.imagesize;
		bmpinfo.bits= 24;
	}
		
	bmpinfo.compression= 0;	// no compression
	bmpinfo.xresolution= 10000;
	bmpinfo.yresolution= 10000;
	bmpinfo.ncolours= 0;
	bmpinfo.importantcolours= 0; // all colours
		
	out= fopen(fname, "wb");
	if(out==NULL)
	{
		printf("\n-- erreur ecriture: %s\n", fname);
		return 0;
	}

	code= ecrire_bmp_entete(out, &bmphead);
	if(code!=1)
	{
		printf("\n-- erreur ecriture: %s\n", fname);
		fclose(out);
		return 0;
	}
	
	code= ecrire_bmp_info(out, &bmpinfo);
	if(code!=1)
	{
		printf("\n-- erreur ecriture: %s\n", fname);
		fclose(out);
		return 0;
	}

	// ecrire la palette, si necessaire
	if(img->palette_n==256)
	{
		tmp= (unsigned char *) malloc(sizeof(unsigned char[4]) * img->palette_n);
		assert(tmp);
		
		for(i= 0; i<img->palette_n; i++)
		{
			r= img->palette[4*i];
			v= img->palette[4*i +1];
			b= img->palette[4*i +2];
			
			tmp[4*i]= b;
			tmp[4*i +1]= v;
			tmp[4*i +2]= r;
			tmp[4*i +3]= 0;
		}	

		n= fwrite(tmp, sizeof(unsigned char[4]), img->palette_n, out);
		free(tmp);
		
		if(n != (size_t) img->palette_n)
		{
			printf("\n-- erreur ecriture: %s\n", fname);
			fclose(out);
			return 0;
		}

		// ecrire les donnees
		n= ecrire_aligne32b_256(out, img);
		if(n != (size_t) img->largeur*img->hauteur)
		{
			printf("\n-- erreur ecriture: %s\n", fname);
			fclose(out);
			return 0;
		}
	}
	else if(img->palette_n==0)
	{
		n= ecrire_aligne32b_rgb(out, img);
		if(n != (size_t) img->largeur*img->hauteur)
		{
			printf("\n-- erreur ecriture: %s\n", fname);
			fclose(out);
			return 0;
		}
	}
	
	fclose(out);
	
	return 1;
}

