#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2006 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr

import casauth
import cgi
import time
import questions
import pages
import random
import student
import utilities
import statistics
import configuration
import real_names
import acls
import types

html = None

###############################################################################
# Information filling 
###############################################################################

def fill_questions_html(answerables):

    if answerables == []:
        return "<P CLASS='no_more_question'></P>"

    if len(answerables) < 50:

        focusable_found = False
        for q, info in answerables:
            if info.find("resigned ") == -1:
                focusable_found = True
                break
            
        s = ""
        focus_done = False
        for q, info in answerables:
            focus = ''
            if not focus_done:
                if not focusable_found or info.find("resigned ") == -1:
                    focus = "ID=1 "
                    
            s += "<A %sHREF='%s' CLASS='%s'>%s</A><BR>\n" % (
                focus,
                q.url(),
                info,
                cgi.escape(q.name),
                )

            if focus != '':
                focus_done = True
    
        s += "<script type='text/javascript'>document.getElementById(1).focus();</script>"
    else:
        s = """<form>
        <script>
        function f(x)
        {
        window.location = '?action=question_see&question=' + x ;
        }
        </script>
        """
        w = ""
        for q, info in answerables:
            world = q.name.split(":",1)[0]
            if w != world:
                w = world
                if s.find("<select") != -1:
                    s += "</select><br>"
                s += "<select OnChange='f(value);'>"
                s += "<option selected='1'>%s</option>" % world
            s += "<option>%s</option>\n" % q.name
        s += "</select></form>"

    return s

def client_ip(server):
    try:
        # In cas of proxy
        ip = server.headers["X-Forwarded-For"]
        try:
            # Take the first IP
            return ip.split(",")[0]
        except IndexError:
            return ip
    except KeyError:
        return server.client_address[0]

def the_service(server):
    try:
        host = server.headers['x-forwarded-server']
    except:
        host = server.headers["host"]

    return "http://%s/%s" % (host, configuration.prefix)


style = {
    -2: "<img align='top' src='../../very_bad.png'>",
    -1: "<img align='top' src='../../bad.png'>",
    1: "<img align='top' src='../../good.png'>",
    2: "<img align='top' src='../../very_good.png'>",
    }

###############################################################################
# 
###############################################################################

class State:
    def __init__(self, server, ticket, student_name):
        self.student = student.student(student_name)
        self.student.acls.state = self
        self.time_creation = time.time()
        self.question = None
        self.page = pages.page_home
        self.sort_column = 0
        self.ticket = ticket
        self.number = 0
        self.shuffle_seed = random.random()
        self.student_stat = None
        statistics.forget_stats()
        self.display_history = []
        self.client_ip = None
        self.client_browser = None
        self.update(server)

    # The user call the service with a different name (via an apache proxy)
    # Must be called on each page loading to have no problems.
    def update(self, server):
        login = False
        
        new_client_ip = client_ip(server)
        if self.client_ip != new_client_ip:
            login = True
        self.client_ip = new_client_ip

        new_client_browser = server.headers["User-Agent"]
        if self.client_browser != new_client_browser:
            login = True
        self.client_browser = new_client_browser
        
        self.url_base = the_service(server)

        if login:
            self.student.login(self.client_ip + " " + self.client_browser)
        
    def ticket_valid(self, server):
        if time.time() - self.time_creation > configuration.timeout:
            return False
        if self.client_ip != client_ip(server):
            return False
        if self.client_browser != server.headers["User-Agent"]:
            return False
        return True

    def initialize_times(self):
        data = self.data
        data["start_date"] = ""
        data["end_date"] = ""
        data["time_left"] = ""
        
        if not self.student.acls['gui_times']:
            return
        
        if self.start < configuration.dates[0]:
            data["start_date"] = "<br><span class='start_date'>%s</span>" % \
                                 utilities.user_date(configuration.dates[0])
        if self.start < configuration.dates[1]:
            data["end_date"] = "<br><span class='end_date'>%s</span>" % \
                               utilities.user_date(configuration.dates[1])
        else:
            data["end_date"] = ""
        if self.start > configuration.dates[0] \
               and self.start < configuration.dates[1]:
            t = int(configuration.dates[1] - self.start)
            data["time_left"] = "<br><span class='time_left'>%s</span>" % \
                                utilities.duration(t)

    def initialize_data(self):
        self.data = {}
        data = self.data
        data["title"        ] = self.student.name
        data["good_answer"  ] = ""
        data["bad_answer"   ] = ""
        data["question"     ] = ""
        data["student_stat" ] = ""
        data["comment"      ] = ""
        data["comments"     ] = ""
        data["answers"      ] = ""
        data['guest_list'   ] = ""
        data["entry"        ] = ""
        data["bad_answers"  ] = ""
        data["indices"      ] = ""
        data["required"     ] = ""
        data["question_stat"] = ""
        data["submenu"]       = ""
        data["session"]       = configuration.questions

        data["student_name"] = real_names.real_names.get(self.student.name
                                                         , self.student.name)

        self.data["url_base"] = "%s%s/%d/" % (
            self.url_base, cgi.urllib.quote(self.ticket), self.number+1)

    def a_stat(self, warning, value, item):
        value = str(value)
        s = "<a class='%s tips'>" % item
            
        if warning != 0 and self.student.acls["gui_smiley"]:
            s +=  style[warning] \
                 + value + '<span class="c%d"></span>' % warning
        else:
            s += value

        self.data[item] = s + "</a><br>"

    def finalize_stats(self):
        data = self.data

        self.a_stat(self.student.warning_nr_good_answers,
                    self.student.number_of_good_answers(),
                    "nr_good_answers")
        self.a_stat(self.student.warning_nr_bad_answers,
                    self.student.number_of_bad_answers(),
                    "nr_bad_answers")
        self.a_stat(self.student.warning_nr_given_indices,
                    self.student.number_of_given_indices(),
                    "nr_given_indices")
        self.a_stat(-abs(self.student.warning_time_after),
                    utilities.time_format(int(self.student.time_searching())),
                    "student_time")
        self.a_stat(0,
                    self.student.number_of_given_questions(),
                    "nr_given_questions")
        self.a_stat(0,
                    len(questions.questions),
                    "nr_questions")        
        if self.student.acls["gui_classement"]:
            self.a_stat(0,
                        self.student.classement(),
                        "classement")     
        else:
            data["classement"] = ""

    def finalize_answerables(self):
        answerables = self.student.answerables_typed()
        if self.student.acls["question_shuffle"]:
            random.seed(self.shuffle_seed)
            random.shuffle(answerables)
        if self.student.acls['question_see']:
            self.data["questions"] = fill_questions_html(answerables)
        else:
            self.data["questions"] = '?!@#'

        if self.student.acls["question_next"] \
               and self.student.acls["question_answerable_any"]:
            a = [ i[0] for i in answerables ]
            try:
                next = answerables[a.index(self.question)+1]
            except AttributeError:
                next = None
            except ValueError:
                next = None
            except IndexError: # Last one
                next = None

            if next:
                next = "<p><A HREF='%s' CLASS='next'></A></p>\n"% next[0].url()
                self.data["questions"] = next +  self.data["questions"]

    def finalize_action_menu(self):
        self.compute_stopped()
        self.data["action_url"] = self.student.acls.action_menu(
            self.system_stopped)

    def finalize_data(self):
        self.finalize_stats()
        self.finalize_answerables()
        self.finalize_action_menu()
        self.initialize_times()

        if not self.data.has_key("h1"):
            self.data["h1"] = self.data["title"]
        if self.stopped and not self.student.acls["question_answerable_anytime"]:
            s = ""
            if self.start < configuration.dates[0]:
                s = "<br><span class='time_start'></span>"
                t = configuration.dates[0]
            elif self.start > configuration.dates[1]:
                s = "<br><span class='time_end'></span>"
                t = configuration.dates[1]
            self.data["h1"] += s + time.strftime("<br>%H:%M %d/%m/%Y",
                                                 time.localtime(t))

    def analyse_form(self, form):
        self.form = form
        if form["number"] == None:
            form["number"] = str(self.number - 1)
        if form["number"] != self.number:
            form["action"] = ''
            try:
                h = self.display_history[int(form["number"])]
                self.page = h[0]
                print self.page
                self.question = h[1]
                self.sort_column = h[2]
            except IndexError:
                pass

        if form.get("sort_column", ""):
            self.sort_column = int(form["sort_column"])

        if form.get("student", ""):
            try:
                self.student_stat = student.students[form["student"]]
                if self.student_stat == self.student:
                    self.student_stat = None
            except KeyError:
                self.student_stat = None

        if form.get("question", ""):
            self.question = questions.questions[form["question"]]

    def perform_action(self):
        action = self.form.get("action","")
        if action and self.student.acls[action]:
            the_action = acls.all_acls[action]
            if the_action.action != None:
                # If action returns something : its a page
                p = the_action.action(self)
                if p != None:
                    return p
            if the_action.page != None:
                self.page = the_action.page

    def compute_stopped(self):
        self.system_stopped=self.stopped= self.start < configuration.dates[0] \
                              or self.start > configuration.dates[1]
        if self.student.acls["question_answerable_anytime"]:
            self.stopped = False

    def execute(self, form):
        self.start = time.time()

        self.compute_stopped()
        self.analyse_form(form)
        self.initialize_data()
        p = self.perform_action()
        statistics.update_stats() # Update statistics
        if p == None:
            p = self.page(self)

        if not isinstance(p, types.TupleType) \
           or p[0] == 'image/svg+xml':
            self.number += 1
            self.display_history.append( (self.page, self.question,
                                          self.sort_column))

        if isinstance(p, types.TupleType): # Page + Mime type
            return p
            
        self.data["informations"] = p
        self.data["time"] = time.time() - self.start
        self.finalize_data()
              
        return 'text/html', html % self.data

states = {}

    

def get_state(server, ticket):

    global html
    if html == None:
        f = open(configuration.root + "/HTML/question.html", "r")
        html = f.read()
        f.close()

    service = cgi.urllib.quote(the_service(server))

    try:            
        if ticket != "" and (ticket.startswith("guest") or states[ticket].ticket_valid(server)):
            state = states[ticket]
            state.update(server)
            return state
        else:
            casauth.redirect(server, service)
            return None
    except KeyError:
        if ticket and ticket.startswith('guest'):
            student_name = ticket
        else:
            try:
                student_name = casauth.get_name(ticket, service)
            except IOError:
                return get_state(server, "")
        # Search if it is a new ticket for an existing student
        for s in states.values():
            if s.student.filename == student_name:
                states[ticket] = s
                s.update(server)
                # s.student.login(s.client_ip + " " + s.client_browser)
                return s

        states[ticket] = State(server, ticket, student_name)
        return states[ticket]

