#!/usr/bin/env python

#    QUENLIG: Questionnaire en ligne (Online interactive tutorial)
#    Copyright (C) 2005-2006 Thierry EXCOFFIER, Universite Claude Bernard
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Contact: Thierry.EXCOFFIER@bat710.univ-lyon1.fr

import os
import sys
import statistics
import time
import configuration
import questions
import acls
import server

def mkdir(name):
    if not os.path.exists(name):
        os.mkdir(name)

def write(filename, text):
    f = open(filename, 'w')
    f.write(text)
    f.close()

def read(filename):
    try:
        f = open(filename, 'r')
    except IOError:
        return ''
    c = f.read()
    f.close()
    return c

class Session:
    def __init__(self, name):
        self.name = name
        self.cache = {}
        self.nr_requests = 0
        self.percentage_time_for_stat = None
        self.dir = 'Students/' + name + '/'
    def set_questions_directory(self, dirname):
        write(self.dir + 'questions', dirname)
    def set_port(self, port):
        write(self.dir + 'port', port)
    def set_begin_date(self, date):
        write(self.dir + 'begin_date', date)
    def set_end_date(self, date):
        write(self.dir + 'end_date', date)

    def init(self):
        configuration.port = int(read(self.dir + 'port'))
        configuration.dates = [
            time.mktime(
            time.strptime(read(self.dir + 'begin_date'), "%H:%M %d/%m/%Y")),
            time.mktime(
            time.strptime(read(self.dir + 'end_date'),  "%H:%M %d/%m/%Y"))
            ]
        if self.percentage_time_for_stat:
            configuration.statistics_cpu_allocation = \
            self.percentage_time_for_stat
        configuration.questions = read(self.dir + 'questions')
        
        questions.modules = {}
        for i in  os.listdir(configuration.questions):
            if i == '__init__.py':
                continue
            if i.endswith('.py'):
                i = i[:-3]
                name = configuration.questions.replace('/', '.') + '.' + i
                # Get module object
                module = __import__(name)
                for directory in name.split('.')[1:]:
                    module = module.__dict__[directory]
                questions.modules[i] = module

        questions.sort_questions()

        os.chdir(self.dir)
        acls.initialize()
        configuration.acls_default = acls.Acls(
            'acls.default', default=configuration.acl_default_see)
        configuration.acls_student = acls.Acls(
            'acls.student', default=configuration.student_role,
            role=configuration.acls_default)
        configuration.acls_teacher = acls.Acls(
            'acls.teacher', default=configuration.teacher_role,
            role=configuration.acls_default)
        configuration.acls_author = acls.Acls(
            'acls.author', default=configuration.author_role,
            role=configuration.acls_default)
        configuration.acls_admin = acls.Acls(
            'acls.admin', default=configuration.admin_role,
            role=configuration.acls_default)
        os.chdir(configuration.root)

    def start(self):
        self.init()
        write(self.dir + 'pid', str(os.getpid()))
        os.chdir(self.dir)
        server.run(self.nr_requests, self.cache)
        os.chdir(configuration.root)

    def stop(self):
        pid = read(self.dir + 'pid')
        if pid != '':
            try:
                os.kill( int(pid), 15)
            except OSError:
                pass
            os.remove(self.dir + 'pid')

    def plot(self):
        import pages

        self.init()
        os.chdir(self.dir)        
        statistics.update_stats()
        statistics.graph()
        statistics.graph_dot()
        statistics.graph2_dot()
        pages.page_plot(None)
        sys.exit(0)
        

# Analyse command line options

args = sys.argv[1:]
args.reverse()

try:
    session = Session(args.pop())
except IndexError:
    sys.stderr.write("""
Parameters for standard use :
 - REQUIRED: The name of an existing session in 'Students'
 - A LIST of actions executed in the order provided by the command line:
     'create QuestionDirectory Port# "%H:%M %d/%m/%Y" "%H:%M %d/%m/%Y"'
         creation of the session (with start and end dates)
     'start'
         start the server
     'stop'
         stop the server
     'plot'
         creation of all the plots related to this session
     'admin login_name'
         Give to the login_name the administrator role
     'nocache'
         Non-persistent option : does not cache HTML, CSS, PS, ... files
     'nr-requests-served #requests'
         Non-persistent option : the server stops after
     'percentage-time-for-stat'
         Non-persistent option : % of time used by the server to compute stats
""")
    sys.exit(1)

mkdir('Students')
mkdir(session.dir)
mkdir(session.dir + 'Logs' )
mkdir(session.dir + 'HTML' )
configuration.root = os.getcwd()

while args:
    action = args.pop()
    if action == 'start':
        session.start()
    elif action == 'stop':
        session.stop()
    elif action == 'nocache':
        session.cache = None
    elif action == 'plot':
        session.plot()
    elif action == 'nr-requests-served':
        session.nr_requests = int(args.pop())
    elif action == 'percentage-time-for-stat':
        session.percentage_time_for_stat = int(args.pop())
    elif action == 'create':
        session.set_questions_directory(args.pop())
        session.set_port(args.pop())
        session.set_begin_date(args.pop())
        session.set_end_date(args.pop())
    elif action == 'admin':
        user = session.dir + 'Logs/' + args.pop()
        mkdir(user)
        user += '/acl'
        write(user, read(user) + '\nrole_admin_i_have\n')
    else:
        sys.stderr.write("""Unknown action : %s\n""" % action)
        sys.exit(2)

