/*    NNFS: Non-Networked File System.
 *    Copyright (C) 1995-1998  Thierry EXCOFFIER (exco@ligim.univ-lyon1.fr)
 *
 *    This program is g_Free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the G_Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the G_Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "g_nnfsrc.h"
#include "g_help.h"
#include "g_malloc.h"
#include "g_debug.h"
#include "g_secure.h"
#include "g_unix.h"

#include <stdio.h>
#if HAVE_ERRNO_H
#include <errno.h>
#endif
#if HAVE_TIME_H
#include <time.h>
#endif

#include <stdlib.h>

/*
 * DO NOT CHANGE ORDER or ADD/REMOVE in first table without changing order
 * in second part of the function.
 */

#define S(N) t[i++].offset_in_g_config = (char*)&conf.N - (char*)&conf

g_Nnfsrc_Var *g_Nnfsrc_Var_Table()
{
  g_Config conf ;
  int i ;
  static g_Nnfsrc_Var t[] =
  {
    { "CASE"             , g_Var_String ,-1, 0,
      "Switch case in which current program go."},
    { "COMMENT"          , g_Var_Big_String ,0, 0,
      "A comment for this host dependant configuration"},
    /*
    { "TMP"              , g_Var_String ,0, 0,
      "Name of a file used for copying on medium. It is useful if the copy command can not work with standard input or output.\nThe filename is generated by a shell script to avoid overwriting by another program."},
    */
    { "READ_MEDIUM"      , g_Var_String ,0, 0,
      "The command that read the file on the medium and output it on standard output.\nThe '%' is replaced by the file name. For example, 'exco.AAA', 'exco.AAB', ...", "program" },
    { "WRITE_MEDIUM"     , g_Var_String ,0, 0,
      "The command that read its standard input and store it on the medium.\nThe '%' is replaced by the file name. For example, 'exco.AAA', 'exco.AAB', ...",
    "program" },
    { "COMPRESS_FILTER"  , g_Var_String ,1, 0,
    "The command compressing its standard input and writing the compressed result on its standard output",
    "program"},
    { "UNCOMPRESS_FILTER", g_Var_String ,1, 0,
    "The command uncompressing its standard input and writing the uncompressed result on its standard output",
    "program"},
    { "OUTPUT_FILTER"    , g_Var_Filter ,3, 0,
      "The filter used to avoid copy some files.\nFiles too big for the medium.\nFiles not useful. For example the '.o' if you synchronize hosts with uncompatible architecture.\nFiles dangerous to synchronize, for example the files used by a program while NNFS run. To be more precise, the mailbox summaries of 'Netscape' if you do not stop 'Netscape' while running NNFS."},
    { "EJECT"            , g_Var_String ,0, 0,
      "The command ejecting the medium. If there is no such command, use the 'true' command to report 'no error'.",
    "program"},
    { "MEDIUM_SIZE"      , g_Var_Integer,1, 0,
      "The number of bytes you are sure you can store on the medium. "
      "This value is limited to 2147483647 (2Gb) so if the medium "
      "is bigger multiple files will be stored on it."
    },
    { "HISTORY_DIR"      , g_Var_String ,2, 0,
      "Where to historize modified/deleted files. "
      "Do 'man strftime' for more information. "
      "The name is relative to $HOME/.nnfs/\n"
      "  %Y : Year\n"
      "  %m : Month\n"
      "  %d : Month day\n"
      "  %H : Hour\n"
      "  %M : Minute\n"
      "  %S : Second\n"
      ,
    "program"},
    { "MEDIUM_NAME"      , g_Var_String ,2, 0,
      "Name of the file on the medium (if the medium is not a cartridge). This name is postfixed by AAA, AAB, AAC, ... for numbering. If you have the same username on all the hosts, the default value works. If it is not the case, set the name to a constant, for example 'nnfs'",
    "program"},
    { "HISTORIZE"        , g_Var_Boolean,1, 0,
      "Let you choose which file you want to historize: modified/deleted/conflicting.", NULL,
      {
	{"no",
	 "deleted files and files with conflicting update"
	},
	{"yes",
	 "deleted files and modified files (safest option)."
	},
	{""}
      }
    },
    { "ASK_MEDIUM"       , g_Var_Boolean,2, 0,
      "By default, NNFS asks you to insert medium.\nIt is useful to not set this option if your medium is virtual (when you use FTP or 'remsh') or if you use an autochanger. A drawback is that in case of writing error, NNFS abort instead of asking a new medium.", NULL,
      {
	{"no",
	 "NNFS will not ask you to insert medium.\nBeware, if there is an error, NNFS will abort."
	},
	{"yes",
	 "NNFS asks you to insert medium"
	},
      {""}
      }
    },
    { "CONFIRMATION"     , g_Var_Choice,1, 1,
      "Indicate to NNFS in which case a question "
      "is asked to you before starting the updating of your files.", NULL,
      {
	{"never",
	 "NNFS never asks confirmation before updating."
	},
	{"always",
	 "NNFS always asks confirmation before updating."
	},
	{"yes",
	 "NNFS asks confirmation in case of conflict or file to delete"
	},
	{""}
      }
    },
    { "AUTO_START"     , g_Var_Choice,1, 0,
      "Indicate to NNFS X11 interface if synchronisation start "
      "without user intervention.", NULL,
      {
	{"no",
	 "NNFS asks confirmation before starting the update."
	},
	{"yes",
	 "NNFS start immediatly update."
	},
	{""}
      }
    },
    { "SYNC_READONLY_DIR"     , g_Var_Choice,1, 0,
      "Indicates if the content of read only directory should be synchronized."
      "\n"
      "If the content is synchronized, the directory mode should be "
      "temporary modified in order to update its content.", NULL,
      {
	{"no",
	 "NNFS does not update read-only directory content."
	},
	{"yes",
	 "NNFS updates read-only directory content."
	},
	{""}
      }
    },
    { "MODE_CHANGE_SENSITIVE"     , g_Var_Choice,2, 1,
      "Indicate to NNFS if the file mode is used to see if "
      "the file is modified or not.", NULL,
      {
	{"no",
	 "If only the file mode change,\nthe file is not synchronized. "
	 "If the file is synchronized,\nits mode will be synchronized."
	},
	{"yes",
	 "If the file mode change, the file is synchronized."
	},
	{""}
      }
    },
    { "TMP_FILE"         , g_Var_Choice ,2, 1,
      "If there is an error when writing on the medium, NNFS abort the update. Except if this option is set to 'memory' in which case if there is an error, NNFS will attempt to rewrite the medium. Beware, if the medium size is 100Mb you will need 100Mb of swap...", NULL,
      {
	{"none",
	 "NO TEMPORARY STORAGE.\nNNFS Abort in case of medium writing error"
	},
	{"memory",
	 "MEMORY AS TEMPORARY STORAGE\nNNFS will retry in case of medium writing error.\nBeware, if your medium contains 100Mb\nyou will need 100Mb of swap..."
	},
	{""}
      }
    },
    { "COPY_HARD_LINK"         , g_Var_Boolean ,2, 0,
      "If set to 'copy', the hard linked files on an host are copied as several files not hard linked on the other hosts. It is dangerous to use because if one of the not hard linked file is modified the hard link will be destroyed on the original host.\nIf set to 'not copy', the hard linked files are not copied at all, it is safest.", NULL,
      {
	{"no",
	 "Hard linked files are not copied on other hosts"
	},
	{"yes",
	 "Hard linked files are copied as\nnon hard linked files on the other hosts."
	},
	{""}
      }
    },
    { "CROSS_MOUNT_POINT"         , g_Var_Boolean ,2, 0,
      "If mount point are crossed, all the files are synchronized even they are on several file systems", NULL,
      {
	{"no",
	 "Do not cross mount point"
	},
	{"memory",
	 "Cross mount point"
	},
	{""}
      }
    },
    { NULL }
  } ;

  if ( t->offset_in_g_config )
    return(t) ;

  i = 0 ;
  S(switchcase) ;
  S(comment) ;
  S(read_medium) ;
  S(write_medium) ;
  S(compress_filter) ;
  S(uncompress_filter) ;
  S(file_filter) ;
  S(eject) ;
  S(medium_size) ;
  S(history_format) ;
  S(medium_name) ;
  S(historize) ;
  S(ask_medium) ;
  S(confirmation) ;
  S(auto_start) ;
  S(sync_readonly_dir) ;
  S(mode_change_sensitive) ;
  S(tmp_file) ;
  S(copy_hard_link) ;
  S(cross_mount_point) ;

  return(t) ;
}

/*
 *
 */

char *g_Read_Content(FILE *f)
{
  int c ;
  char tmp[G_LINE_CHUNK] ;
  char *w ;

  c = getc(f) ;
  if ( c != '"' )
    {
      g_Printf("Not a \" after =, it is a `%c'\n", c) ;
      exit(26) ;
    }

  w = tmp ;
  for(;!feof(f);)
    {
      c = getc(f) ;
      switch(c)
	{
	case '"':
	  *w = '\0' ;
	  return( g_Save_String(tmp) ) ;
	case '\\':
	  c = getc(f) ;
	  if ( c!='"' && c!='`' )
	    *w++ = '\\' ;
	  *w++ = c ;
	  break ;
	default:
	  *w++ = c ;
	}
    }
  return(g_Save_String("String not terminated in nnfsrc")) ;
}

void g_Write_Content(FILE *f, const char *c)
{
  putc('"', f) ;
  while(*c)
    {
      switch(*c)
	{
	case '"':
	  fputs("\\\"", f) ;
	  break ;
	case '`':
	  fputs("\\`", f) ;
	  break ;
	default:
	  putc(*c, f) ;
	  break ;
	}
      c++ ;
    }
  putc('"', f) ;
}

/*
 *
 */

char *g_Case_Name(const char *hostname)
{
  FILE *h ;
  char named[G_LINE_CHUNK] ;

  /*  
  if ( hostname && hostname[0] && strchr(hostname,'.')==NULL )
    {
      strcpy(named, hostname) ;
      h = g_popen("echo .`domainname 2>/dev/null`.`uname`","r");
      fgets(named+strlen(named), sizeof(named), h) ;
      g_pclose(h) ;
      named[strlen(named)-1] = '\0' ;      
    }
  else
    {
      h = g_popen("echo `hostname`.`domainname 2>/dev/null`.`uname`","r");
      fgets(named, sizeof(named), h) ;
      g_pclose(h) ;
      named[strlen(named)-1] = '\0' ;
    }
  */
  strcpy(named, hostname) ;
  h = g_popen("echo .`uname`","r");
  if ( fgets(named+strlen(named), sizeof(named), h) == NULL )
    PERROR("fgets") ;
  g_pclose(h) ;
  named[strlen(named)-1] = '\0' ;      

  return( g_Save_String(named) ) ;
}


void g_Nnfsrc_Insert_Case(FILE *g, const char *hostname, const g_Nnfsrc_Var *vt)
{
  time_t t ;
  char *tmp ;
  int i ;
  /*
   * New case to write
   */
  tmp = g_Case_Name(hostname) ;
  time(&t) ;
  fprintf(g, "\n\"%s\") # %s", tmp, ctime(&t)) ;
  fprintf(g, "CASE=\"%s\"\n", tmp) ;
  g_Free(tmp) ;
  
  fprintf(g, "COMMENT=") ;
  for(i=0;;i++)
    if( strcmp(vt[i].name, "COMMENT") == 0 )
      {
	g_Write_Content(g, vt[i].content) ;
	break ;
      }
  fprintf(g, "\nREAD_MEDIUM=") ;
  for(i=0;;i++)
    if( strcmp(vt[i].name, "READ_MEDIUM") == 0 )
      {
	g_Write_Content(g, vt[i].content) ;
	break ;
      }
  fprintf(g, "\nWRITE_MEDIUM=") ;
  for(i=0;;i++)
    if( strcmp(vt[i].name, "WRITE_MEDIUM") == 0 )
      {
	g_Write_Content(g, vt[i].content) ;
	break ;
      }
  fprintf(g, "\nEJECT=") ;
  for(i=0;;i++)
    if( strcmp(vt[i].name, "EJECT") == 0 )
      {
	g_Write_Content(g, vt[i].content) ;
	break ;
      }
  fprintf(g, "\n;;\n\n") ;
}


/*
 * This function get variable value of any case in the nnfsrc
 * or save the variable values if "g" is non NULL
 */

void g_Nnfsrc_Action(char *name,
		     g_Nnfsrc_Var *vt,
		     char *in_case,
		     g_Boolean *in_case_found,
		     char ***cases, int *nb_cases,
		     FILE *g,
		     const char *hostname
		     )
{
  FILE *f ;
  int i ;
  int c ;
  char named[G_LINE_CHUNK] ;
  char *current_case ;
  char *tmp ;

  f = g_fopen(name, "r") ;

  current_case = NULL ;
  *in_case_found = g_False ;
  *nb_cases = 0 ;
  G_MALLOC(*cases,1) ;

  if ( g == NULL )
    for(i=0;vt[i].name;i++)
      {
	g_Free(vt[i].content) ;
	vt[i].content = g_Save_String("") ;
      }
  
  do
    {
      /*
       * Jump start of line
       */
    start_of_line:
      do
	{
	  c = getc(f) ;
	  if ( c>=0 && g )
	    putc(c, g) ;
	}
      while( c==' ' || c=='\t' || c=='\n' ) ;
      /*
       * Read until '=' or ')'
       */
      i = 0 ;
      named[i++] = c ;
      for(;!feof(f);)
	{
	  switch( named[i-1] )
	    {
	    case '=':
	      named[i-1] = '\0' ;
	      tmp = g_Read_Content(f) ;

	      if ( current_case )
		if ( strcmp(current_case, in_case) )
		  {
		    if ( g )
		      g_Write_Content(g, tmp) ;
		    goto jump_over ;
		  }
	      for(i=0;vt[i].name;i++)
		if ( strcmp(named, vt[i].name) == 0 )
		  break ;
	      if ( vt[i].name==NULL )
		{
		  g_Printf("NNFS: Variable name mispelled? (%s)\n",
			   named) ;
		  if ( g )
		    g_Write_Content(g, tmp) ;
		  goto jump_over ;
		}
	      if ( g )
		{
		  if ( current_case )
		    g_Write_Content(g, tmp) ;
		  else
		    g_Write_Content(g, vt[i].content) ;
		}

	      g_Free(vt[i].content) ;
	      vt[i].content = tmp ;
	      goto jump_over ;
	      

	    case ')':
	      named[i-1] = '\0' ;
	      current_case = g_Save_String(named) ;
	      if ( strcmp(in_case, current_case) == 0 )
		{
		  if ( g )
		    {
		      /* Do not store this old value */
		      fseek(g, -i-2, SEEK_CUR) ;
		      if ( fgets(named, sizeof(named), f) == NULL )
			PERROR("fgets") ;
		      for(;;)
			{
			  if ( fgets(named, sizeof(named), f) == NULL )
			    return ;
			  for(i=0; named[i]; i++)
			    if ( named[i]!=' '
				 && named[i]!='\t'
				 && named[i]!='\n' )
			      break ;
			  if ( named[i] == ';' && named[i+1] == ';' )
			    break ;
			}
		      do
			i = getc(f) ;
		      while( i==' ' || i=='\n' || i=='\t' ) ;
		      ungetc(i, f) ;
		      goto jump_over ;
		    }
		  else
		    *in_case_found = g_True ;
		}
	      i = 0 ;
	      break ;

	    case ';':
	      c = getc(f) ;
	      if ( c>=0 && g )
		putc(c, g) ;
	      if ( c==';' )
		{
		  if ( current_case )
		    {
		      (*nb_cases)++ ;
		      G_REALLOC(*cases, *nb_cases) ;
		      (*cases)[*nb_cases-1] = current_case ;
		      current_case = NULL ;
		    }
		  goto jump_over ;
		}
	      if ( c=='\n' )
		goto start_of_line ;
	      goto jump_over ;
	    case ' ':
	      named[i] = '\0' ;
	      if ( g
		   /* && *in_case_found == g_False */
		   && strncmp(named, "case", 4) == 0
		   )
		{
		  if ( fgets(named, sizeof(named), f) == NULL )
		    PERROR("fgets") ;
		  fprintf(g, "%s", named) ;
		  g_Nnfsrc_Insert_Case(g, hostname, vt) ;
		  goto start_of_line ;
		}
	    case '(':
	    case '\t':
	      goto jump_over ;
	    case '\n':
	      goto start_of_line ;
	    case '"':
	      i-- ;
	      do
		{
		  c = getc(f) ;
		  named[i++] = c ;
		  if ( c>0 && g )
		    putc(named[i-1], g) ;
		}
	      while(named[i-1]!='"' && !feof(f)) ;
	      if ( feof(f) )
		{
		  g_Printf("Unterminated '\"' in %s\n", name) ;
		  return ;
		}
	      i-- ;
	      break ;
	    }
	  c = getc(f) ;
	  named[i++] = c ;
	  if ( c>0 && g )
	    putc(named[i-1], g) ;
	}
      /*
       *
       */
    jump_over:
      while( (c=getc(f)) != '\n' )
	{
	  if ( c>=0 && g )
	    putc(c, g) ;
	  if ( feof(f) )
	    break ;
	}
      if ( g && c == '\n' )
	putc('\n', g) ;

    }
  while(!feof(f)) ;
}

/*
 * Compile configuration
 */

void G_FUNCTION(g_Compile_Config, (g_Nnfsrc_Var *v, g_Config *config))


char *name ;
char *(*choices)[2] ;
void *ptr ;
int i, j ;
char *t ;

for( i=0 ; v[i].name ; i++) 
{
  ptr = (char*)config + v[i].offset_in_g_config ;
  name = v[i].name ;
  t = v[i].content ;
  
  switch(v[i].type)
    {
    case g_Var_String:
    case g_Var_Big_String:
    case g_Var_Filter:
      *(char**)ptr = g_Save_String(t) ;
      break ;
    case g_Var_Choice:
      choices = v[i].choices ;
      *(int*)ptr = v[i].default_choice ;
      for(j=0;j<choices[j][0][0];j++)
	if ( strcmp(choices[j][0], t) == 0 )
	  {
	    *(int*)ptr = j ;
	    break ;
	  }
      if ( choices[j][0][0] == '\0' )
	{
	  g_Printf("NNFS: Choice `%s' not possible for `%s'\n", t, name) ;
	  g_Printf("NNFS: Accepted values are:\n") ;
	  for(j=0;j<choices[j][0][0];j++)
	    g_Printf("NNFS: %s (%s)\n", choices[j][0], choices[j][1]) ;
	}
      break ;
    case g_Var_Integer:
    case g_Var_Boolean:
      *(int*)ptr = v[i].default_choice ;
      if ( t[0] )
	{
	  if ( strcmp(t,"yes")==0 || strcmp(t,"y")==0
	       || strcmp(t,"1")==0 || strcmp(t,"true")==0 )
	    *(int*)ptr = 1 ;
	  else
	    *(int*)ptr = atoi(t) ;
	}
      break ;
    }
}

G_RETURN(;) ;

     }

/*
 *
 */

void G_FUNCTION(g_Read_Config, (g_Config *config, const char *hostname))

g_Read_Config_Char(config->nnfsrc, g_Nnfsrc_Var_Table(), hostname) ;

g_Compile_Config(g_Nnfsrc_Var_Table(), config) ;
G_RETURN(;) ;
}

/*
 *
 */

char * G_FUNCTION(g_Create_Nnfsrc,(char *home, g_Boolean *created))

char *nnfsrc ;
FILE *f ;

nnfsrc = g_Nnfsrc(home) ;
f = fopen(nnfsrc,"r") ;

if ( f )
	{
	fclose(f) ;
	*created = g_False ;
	G_RETURN(nnfsrc) ;
	}

if ( errno != ENOENT )
	{
	PERROR(nnfsrc) ;
	G_EXIT(1) ;
	}

f = g_fopen(nnfsrc,"w") ;

#include "nnfsrc"

g_fclose(f) ;
*created = g_True ;
G_RETURN(nnfsrc) ;
}

