import sys
import os
import collections

class Function(object):
    def __init__(self, function, functionality, before, after, always):
        self.function = function
        self.functionality = functionality
        self.always = always
        self.enabled = True
        self.name = self.function.__name__
        if self.name == '<lambda>':
            self.name = 'lambda%d' % id(self.function)
        if isinstance(before, str):
            before = (before,)
        self.before = list(before)
        if isinstance(after, str):
            after = (after,)
        self.after = list(after)
    def __str__(self):
        s = "%s %s" % (self.functionality, self.name)
        if self.before:
            s += " before=%s" % (self.before, )
        if self.after:
            s += " after=%s" % (self.after, )
        if self.always:
            s += " always"
        if self.priority:
            s += " priority=%s" % str(self.priority)
        return s
    def html(self):
        return "<td>%s<td>%s<td>%s<td>%s<td>%s" % (
            self.name.strip('<>'),
            ' '.join("'%s'" % i for i in self.before) + "&nbsp;",
            ' '.join("'%s'" % i for i in self.after) + "&nbsp;",
            self.priority,
            self.always or "&nbsp;")

def default_evaluator(hook, *args, **kargs):
    run = True
    value = None
    for function in hook.functions:
        if function.enabled and (run or function.always):
            v = function.function(*args, **kargs)
            if v:
                run = False
                value = v
    return value

cwd = os.getcwd()
    
def get_fy(fy=None, level=2):
    return fy or sys._getframe(level).f_code.co_filename.replace(cwd, ''
    ).replace(os.path.sep, ".")[1:-3]

class Hook(object):
    functions = ()
    need_sort = False
    def __init__(self, name, documentation, functionality, evaluator, args):
        self.__name__ = self.name = name
        self.documentation = documentation
        self.functionality = functionality
        self.fct_by_fy = collections.defaultdict(list)
        self.evaluator = evaluator
        self.args = args
    def add(self, function=None, before=(), after=(), always=False, fy=None):
        fy = get_fy(fy)
        def decorator(to_decorate):
            fct = Function(to_decorate, fy, before, after, always)
            self.fct_by_fy[fct.functionality].append(fct)
            self.need_sort = True
            return to_decorate
        if function is None:
            return decorator
        else:
            return decorator(function)
    def remove(self, functionality):
        if functionality in self.fct_by_fy:
            del self.fct_by_fy[functionality]
            self.need_sort = True
    def all_functions(self):
        for fcts in self.fct_by_fy.values():
            for f in fcts:
                yield f
    def get_functions(self, list_of_functionality_and_fct):
        for functionality_and_fct in list_of_functionality_and_fct:
            if ':' in functionality_and_fct:
                fy, fct_name = functionality_and_fct.split(':')
                for f in self.fct_by_fy[fy]:
                    if f.name == fct_name:
                        yield f
                        break
                else:
                    raise ValueError("%s Can't find %s" % (
                        self.name, functionality_and_fct))
            else:
                if functionality_and_fct in self.fct_by_fy:
                    for f in self.fct_by_fy[functionality_and_fct]:
                        yield f
    def get_sorted_functions_by_priority_classes(self):
        fcts = list(self.all_functions())
        for f in fcts:
            f.priority = (''.join(f.after).count('*')
                          - ''.join(f.before).count('*'))
        # Select the good top priority.
        while True:
            change = False
            for f in fcts:
                for ff in self.get_functions(f.after):
                    if f.priority < ff.priority:
                        if f.priority >= 0:
                            f.priority = ff.priority
                        else:
                            ff.priority = f.priority
                        change = True
                for ff in self.get_functions(f.before):
                    if f.priority > ff.priority:
                        if f.priority <= 0:
                            f.priority = ff.priority
                        else:
                            ff.priority = f.priority
                        change = True
            if not change:
                break
        fcts.sort(key=lambda f: (f.priority, f.functionality, f.name))
        return fcts
    def init_after_before_list(self, fcts):
        for f in fcts:
            f.after_list = list(self.get_functions(f.after))
            f.before_list = list(self.get_functions(f.before))
        for f in fcts:
            for ff in self.get_functions(f.before):
                ff.after_list.append(f)
            for ff in self.get_functions(f.after):
                ff.before_list.append(f)
        for f in fcts:
            f.before_list.sort(key=lambda x:(x.functionality, x.name))
    def walk_tree(self, fcts):
        done = []
        def walk(f):
            if len(f.after_list) == 0:
                try:
                    fcts.remove(f)
                except ValueError:
                    return
                done.append(f)
                for ff in f.before_list:
                    ff.after_list.remove(f)
                for ff in f.before_list:
                    if ff.priority == f.priority:
                        walk(ff)
        while fcts:
            p = fcts[0].priority
            for f in tuple(fcts):
                if f.priority != p:
                    break
                walk(f)
        return done
    def sort(self):
        self.need_sort = False # Not thread safe
        fcts = self.get_sorted_functions_by_priority_classes()
        self.init_after_before_list(fcts)
        self.functions = self.walk_tree(fcts)
    def change_order(self, functionality, function, direction):
        "XXX Change the priority of a function"
        for i, f in enumerate(self.functions):
            if f.functionality == functionality and f.name == function:
                if direction == '-' and i > 0:
                    name = (self.functions[i-1].functionality
                            + ':' + self.functions[i-1].name)
                    f.before.append(name)
                    if name in f.after:
                        f.after.remove(name)
                    if self.functions[i-1].functionality in f.after:
                        f.after.remove(self.functions[i-1].functionality)
                elif direction == '+' and i < len(self.functions) - 1:
                    name = (self.functions[i+1].functionality
                            + ':' + self.functions[i+1].name)
                    f.after.append(name)
                    if name in f.before:
                        f.before.remove(name)
                    if self.functions[i+1].functionality in f.before:
                        f.before.remove(self.functions[i+1].functionality)
                self.need_sort = True
                return
    def __call__(self, *args, **keys):
        if self.need_sort:
            self.sort()
        return self.evaluator(self, *args, **keys)
    def __str__(self):
        self.sort()
        return "name=%s fy=%s args=%s doc=%s\n%s" % (
            self.name, self.functionality, self.args, self.documentation,
            "\n".join("\t" + str(f)
                      for f in self.functions)
        )

class Hooks(object):
    def __init__(self):
        self.hooks = {}
        self.disabled_functionalities = set()
    def new(self, name, documentation, evaluator=default_evaluator, args=None):
        fy = get_fy()
        if name in self.hooks:
            self.hooks[name].documentation = documentation
            self.hooks[name].functionality = fy
            self.hooks[name].evaluator = evaluator
            self.hooks[name].args = args
            return self.hooks[name]
        h = Hook(name, documentation, fy, evaluator, args)
        self.hooks[name] = h
        return h
    def remove(self, functionality):
        for hook in self.hooks.values():
            hook.remove(functionality)
    def reload(self, functionality):
        import imp
        self.remove(functionality)
        imp.reload(sys.modules[functionality])
    def functions(self, functionality):
        for hook in self.hooks.values():
            for fct in hook.fct_by_fy.get(functionality, ()):
                yield fct
    def is_enabled(self, functionality):
        return functionality not in self.disabled_functionalities
    def disable(self, functionality):
        self.disabled_functionalities.add(functionality)
        for fct in self.functions(functionality):
            fct.enabled = False
    def enable(self, functionality):
        self.disabled_functionalities.discard(functionality)
        for fct in self.functions(functionality):
            fct.enabled = True
    def get(self, name):
        if name in self.hooks:
            return self.hooks[name]
        else:
            return self.new(name, '???')
    def functionalities(self):
        fys = collections.defaultdict(list)
        for hook in self.hooks.values():
            for fy, fcts in hook.fct_by_fy.items():
                fys[fy].append((hook, fcts))
        return fys
    def add_to(self, hook_name, function=None, before=(), after=(),
               always=False, fy=None):
        return self.get(hook_name).add(function, before, after,
                                       always, get_fy(fy))
    def __str__(self):
        return '\n'.join('*'*79 + '\n' + name + '\n' + '.'*79 + '\n'
                         + str(hook)
                         for name, hook in self.hooks.items()
                         )

def load_functionalities(pattern):
    import glob
    for filename in glob.glob(pattern):
        __import__(filename[:-3].replace('/', '.'))

hooks = Hooks()
new = hooks.new
get = hooks.get
remove = hooks.remove
reload = hooks.reload
functionalities = hooks.functionalities
enable = hooks.enable
disable = hooks.disable
is_enabled = hooks.is_enabled
add_to = hooks.add_to
