"""
Translate the text content.
Each functionality can contain a 'translations' variable as:
translations = (
    ('Welcome to FyHooks!', {
        'fr': 'Bienvenue sur FyHooks !',
        'it': 'Benvenuti in FyHooks !',
    }),
)
"""
import collections
import importlib
import hooks

translator_text = hooks.new("translator_text",
                            "Update the content of translator.text",
                            args=('context', 'translator'))

@translator_text.add
@hooks.add_to("translator_html", before="HTTP.fy_tr_link")
def tr_language(context, translator):
    translator.text = context.languages[context.language].get(translator.text,
                                                              translator.text)

@hooks.add_to("reload")
def initialize_languages(context, dummy_fy):
    languages = collections.defaultdict(dict)
    for fy in hooks.functionalities():
        m = importlib.import_module(fy)
        translations = m.__dict__.get('translations', ())
        for message, outputs in translations:
            for lang, translation in outputs.items():
                languages[lang][message] = translation
    context.languages = languages

@hooks.add_to("args", before="fy_help")
def args(context):
    context.language = context.extract_arg("--lang=") or 'en'
    initialize_languages(context, "init_languages")
    class Translator(object):
        def __init__(self, translator, **kargs):
            self.content = []
            self.translator = translator
            self.kargs = kargs
        def append(self, string, *args):
            self.text = string
            self.translator(context, self)
            if args:
                self.content.append(self.text.format(*args))
            else:
                self.content.append(self.text)
        def translate(self, string):
            self.text = string
            self.translator(context, self)
            return self.text
        def append_raw(self, string):
            self.content.append(string)
        def __iter__(self):
            for s in self.content:
                yield s
        def __len__(self):
            return len(self.content)
    context.Translator = Translator
    context.translator_text = translator_text

@hooks.add_to("help", after="HTTP.fy_http")
def help(dummy_context, messages):
    messages.append("--lang=XX : set the language")

