"""
Web server functionality.
Add the hooks 'url' and 'translator_html' to do the url dispatch.
"""
import hooks

url = hooks.new(
    "url",
    "Parse server.path, write the page content with server.write()\n"
    "Return True to stop URL dispatch",
    args=("context", "server")
)
translator_html = hooks.new(
    "translator_html",
    "Update the content of translator.text\n"
    "It is possible to use translator.server",
    args=("context", "translator")
)

@translator_html.add(before="*")
def escape(context, translator):
    if translator.kargs.get("escape", False):
        translator.text = translator.server.escape(translator.text)

@url.add(after="*")
def bad_url(dummy_context, server):
    server.write("Bad url: %s" % server.path)

def run_server(context):
    try:
        import http.server as BaseHTTPServer
        my_bytes = bytes
    except ImportError:
        import BaseHTTPServer
        def my_bytes(string, dummy_encoding):
            return string
    class RequestBroker(BaseHTTPServer.BaseHTTPRequestHandler):
        server_running = True
        def do_GET(self):
            url(context, self)
        def get_parameter(self, start):
            if not self.path.startswith(start):
                return
            return self.path[len(start):]
        def get_action(self, action):
            x = self.path.split("?" + action + "=")
            if len(x) == 1:
                return
            self.path = x[0]
            return x[1]
        def translate(self, context, translator):
            translator.server = self
            translator_html(context, translator)
        def write(self, string):
            self.wfile.write(my_bytes(string, 'utf-8'))
        def escape(self, txt):
            return (txt.replace("&", "&amps;")
                    .replace("<","&lt;")
                    .replace(">","&gt;")
                    .replace("\n","<br>")
                    .replace(" ","&nbsp;"))

    server = BaseHTTPServer.HTTPServer(("0.0.0.0", context.port),RequestBroker)
    print("Run web server on http://127.0.0.1:%d" % context.port)
    context.real_server = server
    server.hooks_running = True
    while server.hooks_running:
        server.handle_request()

@hooks.add_to("args")
def args(context):
    if context.extract_arg(('-r', '--run')):
        hooks.add_to("thread", run_server)

@hooks.add_to("help")
def help(dummy_context, messages):
    messages.append(
        "\n"
        "HTTP services and configuration:\n"
        "-r, --run: run HTTP server")
