


#include "App.h"
#include "Widgets/nvSDLContext.h"

#include "Geometry.h"
#include "Transform.h"

#include "Mesh.h"
#include "MeshIO.h"

#include "GL/GLTexture.h"
#include "GL/GLQuery.h"
#include "GL/GLBasicMesh.h"
#include "ProgramManager.h"


class Batch : public gk::App
{
    nv::SdlContext m_widgets;

    gk::GLProgram *m_program[2];
    gk::GLBasicMesh *m_mesh[4];
    gk::GLVertexArray *m_vao;
    
    gk::GLQuery *m_time;
    
    int m_draw_count;
    bool m_draw_degenerate;
    
    bool m_static_vao;
    bool m_change_mesh;
    bool m_change_shader;
    bool m_change_texture;
    
public:
    // creation du contexte openGL et d'une fenetre
    Batch( )
        :
        gk::App()
    {
        // specifie le type de contexte openGL a creer :
        gk::AppSettings settings;
        settings.setGLVersion(3,3);     // version 3.3
        settings.setGLCoreProfile();      // core profile
        settings.setGLDebugContext();     // version debug pour obtenir les messages d'erreur en cas de probleme
        
        // cree le contexte et une fenetre
        if(createWindow(1024, 1024, settings) < 0)
            return;
    
        m_widgets.init();
        m_widgets.reshape(windowWidth(), windowHeight());
    }
    
    ~Batch( ) {}

    // a redefinir pour utiliser les widgets.
    void processWindowResize( SDL_WindowEvent& event )
    {
        m_widgets.reshape(event.data1, event.data2);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseButtonEvent( SDL_MouseButtonEvent& event )
    {
        m_widgets.processMouseButtonEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseMotionEvent( SDL_MouseMotionEvent& event )
    {
        m_widgets.processMouseMotionEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processKeyboardEvent( SDL_KeyboardEvent& event )
    {
        m_widgets.processKeyboardEvent(event);
    }
    
    int init( )
    {
        // compile 2 shaders
        gk::programPath("shaders");
        m_program[0]= gk::createProgram("batch.glsl");
        if(m_program[0] == gk::GLProgram::null())
            return -1;
    
        m_program[1]= gk::createProgram("batch.glsl");
        if(m_program[1] == gk::GLProgram::null())
            return -1;
        
        // charge un mesh
        gk::Mesh *mesh= gk::MeshIO::readOBJ("bigguy.obj");
        if(mesh == NULL)
            return -1;
    
        // cree 2 objets
        m_mesh[0]= new gk::GLBasicMesh(GL_TRIANGLES, mesh->indices.size());
        m_mesh[0]->createBuffer(0, mesh->positions);
        m_mesh[0]->createBuffer(1, mesh->texcoords);
        m_mesh[0]->createBuffer(2, mesh->normals);
        m_mesh[0]->createIndexBuffer(mesh->indices);
        
        m_mesh[1]= new gk::GLBasicMesh(GL_TRIANGLES, mesh->indices.size());
        m_mesh[1]->createBuffer(0, mesh->positions);
        m_mesh[1]->createBuffer(1, mesh->texcoords);
        m_mesh[1]->createBuffer(2, mesh->normals);
        m_mesh[1]->createIndexBuffer(mesh->indices);
        
        // cree 2 objets degeneres, meme nombre de triangles
        for(unsigned int i= 1; i < mesh->indices.size(); i++)
            mesh->indices[i]= mesh->indices[0]; // tous les triangles referencent le meme sommet
        
        m_mesh[2]= new gk::GLBasicMesh(GL_TRIANGLES, mesh->indices.size());
        m_mesh[2]->createBuffer(0, mesh->positions);
        m_mesh[2]->createBuffer(1, mesh->texcoords);
        m_mesh[2]->createBuffer(2, mesh->normals);
        m_mesh[2]->createIndexBuffer(mesh->indices);
        
        m_mesh[3]= new gk::GLBasicMesh(GL_TRIANGLES, mesh->indices.size());
        m_mesh[3]->createBuffer(0, mesh->positions);
        m_mesh[3]->createBuffer(1, mesh->texcoords);
        m_mesh[3]->createBuffer(2, mesh->normals);
        m_mesh[3]->createIndexBuffer(mesh->indices);
        
        // vao
        m_vao= gk::createVertexArray();
        
        glBindVertexArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
        
        m_time= gk::createTimeQuery();
        
        //
        m_draw_count= 1;
        m_draw_degenerate= false;
        
        m_change_shader= false;
        m_change_mesh= false;
        m_change_texture= false;
        
        m_static_vao= true;
        return 0;
    }
    
    int quit( )
    {
        return 0;
    }
    
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            // fermer l'application si l'utilisateur appuie sur ESCAPE
            closeWindow();
        
        if(key('r'))
        {
            key('r')= 0;
            // recharge et recompile les shaders
            gk::reloadPrograms();
        }
        
        if(key('s'))
        {
            key('s')= 0;
            // enregistre l'image opengl
            gk::writeFramebuffer("screenshot.png");
        }
        
        glViewport(0, 0, windowWidth(), windowHeight());
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        gk::Transform model= gk::Translate( gk::Vector(0.f, 0.f, -50.f) );
        gk::Transform view;
        gk::Transform perspective= gk::Perspective(50.f, 1.f, 1.f, 1000.f);
        gk::Transform mv= view * model;
        gk::Transform mvp= perspective * mv;
        
        int program_id= 0;
        int mesh_id= 0;
        int texture_id= 0;
        
        int current_program= -1;
        int current_mesh= -1;
        int current_texture= -1;
        
        GLint64 start; glGetInteger64v(GL_TIMESTAMP, &start);
        m_time->begin();
        
        for(int draw= 0; draw < m_draw_count; draw++)
        {
            // modifie l'etat, si necessaire
            if(m_change_shader)
                program_id= draw % 2;
            if(m_change_mesh)
                mesh_id= draw % 2;
            if(m_change_texture)
                texture_id= draw % 2;
            
            if(current_program != program_id)
            {
                current_program= program_id;
                glUseProgram(m_program[program_id]->name);
            
                m_program[program_id]->uniform("mvpMatrix")= mvp.matrix();
                m_program[program_id]->uniform("normalMatrix")= mv.normalMatrix();
                m_program[program_id]->uniform("color")= gk::Vec4(1.f, 1.f, 1.f);
            }
            
            if(current_mesh != mesh_id)
            {
                current_mesh= mesh_id;
                if(m_draw_degenerate)
                    current_mesh= mesh_id +2;
                
                if(m_static_vao)
                    glBindVertexArray(m_mesh[current_mesh]->vao->name);
                
                else
                {
                    glBindVertexArray(m_vao->name);
                    
                    // recycle le meme vao pour chaque draw
                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh[current_mesh]->buffers[0]->name);
                    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
                    glEnableVertexAttribArray(0);
                    
                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh[current_mesh]->buffers[1]->name);
                    glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, 0);
                    glEnableVertexAttribArray(1);
                    
                    glBindBuffer(GL_ARRAY_BUFFER, m_mesh[current_mesh]->buffers[2]->name);
                    glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, 0, 0);
                    glEnableVertexAttribArray(2);
                    
                    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_mesh[current_mesh]->index_buffer->name);
                }
            }
            
            if(current_texture != texture_id)
            {
                current_texture= texture_id;
                // todo
            }
            
            // draw
            if(m_mesh[current_mesh]->index_type == 0u)
                glDrawArrays(m_mesh[current_mesh]->primitive, 0, m_mesh[current_mesh]->count);
            else
                glDrawElements(m_mesh[current_mesh]->primitive, m_mesh[current_mesh]->count, m_mesh[current_mesh]->index_type, 0);			
        }
        
        glUseProgram(0);
        glBindVertexArray(0);
        
        m_time->end();
        GLint64 stop; glGetInteger64v(GL_TIMESTAMP, &stop);
        GLuint64 gpu_time= m_time->result64();
        
        // 
        glFinish(); 
        
        char tmp[1024];
        m_widgets.begin();
        m_widgets.beginGroup(nv::GroupFlags_GrowDownFromLeft);
            
            m_widgets.doButton(nv::Rect(), "change shader", &m_change_shader);
            m_widgets.doButton(nv::Rect(), "change buffer", &m_change_mesh);
            m_widgets.doButton(nv::Rect(), "change texture", &m_change_texture);
            m_widgets.doButton(nv::Rect(), "draw degenerate triangles", &m_draw_degenerate);
            m_widgets.doButton(nv::Rect(), "use static vao", &m_static_vao);
            m_widgets.doLabel(nv::Rect(), "use bindless buffers");
        
            sprintf(tmp, "draw count %d", m_draw_count);
            m_widgets.doLabel(nv::Rect(), tmp);
            
            float count= m_draw_count;
            if(m_widgets.doHorizontalSlider(nv::Rect(0, 0, 1000, 0), 1.f, 5000.f, &count))
                m_draw_count= count;
            
            sprintf(tmp, "cpu time % 6ldus", (stop - start) / 1000);
            m_widgets.doLabel(nv::Rect(), tmp);
            
            sprintf(tmp, "gpu time % 3ldms", gpu_time / 1000000);
            m_widgets.doLabel(nv::Rect(), tmp);
            
        m_widgets.endGroup();
        m_widgets.end();
        glFinish(); 
    
        present();
        return 1;
    }
};


int main( int argc, char **argv )
{
    Batch app;
    app.run();
    
    return 0;
}

