
/*
    pbrt source code Copyright(c) 1998-2010 Matt Pharr and Greg Humphreys.

    This file is part of pbrt.

    pbrt is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.  Note that the text contents of
    the book "Physically Based Rendering" are *not* licensed under the
    GNU GPL.

    pbrt is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

 */

#ifndef _BVH_PRIVATE_H
#define _BVH_PRIVATE_H

#include "Geometry.h"

//! utilisation interne. construction d'un bvh.
struct BVHPrimitiveInfo
{
    BBox bounds;
    Point centroid;
    int primitiveNumber;
    
    BVHPrimitiveInfo() { }
    
    BVHPrimitiveInfo( int pn, const BBox &b )
        :
        bounds(b),
        centroid(bounds.getCenter()),
        primitiveNumber(pn)
    {}
};

//! utilisation interne. construction d'un bvh.
struct BVHBuildNode
{
    BBox bounds;
    BVHBuildNode *children[2];
    uint32_t splitAxis, firstPrimOffset, nPrimitives;

    // BVHBuildNode Public Methods
    BVHBuildNode() 
    { 
        children[0] = children[1] = NULL; 
    }
    
    void InitLeaf( uint32_t first, uint32_t n, const BBox &b )
    {
        firstPrimOffset = first;
        nPrimitives = n;
        bounds = b;
    }
    
    void InitInterior( uint32_t axis, BVHBuildNode *c0, BVHBuildNode *c1 )
    {
        children[0] = c0;
        children[1] = c1;
        bounds = Union( c0->bounds, c1->bounds );
        splitAxis = axis;
        nPrimitives = 0;
    }
};

//! utilisation interne. construction d'un bvh.
struct CompareToMid
{
    int dim;
    float mid;
    
    CompareToMid( int d, float m )
        :
        dim(d),
        mid(m)
    {}
    
    bool operator( )( const BVHPrimitiveInfo &a ) const
    {
        return a.centroid[dim] < mid;
    }
};

//! utilisation interne. construction d'un bvh.
struct ComparePoints
{
    int dim;

    ComparePoints( int d )
        :
        dim(d)
    {}
    
    bool operator( )( const BVHPrimitiveInfo &a, const BVHPrimitiveInfo &b ) const
    {
        return a.centroid[dim] < b.centroid[dim];
    }
};

//! utilisation interne. construction d'un bvh.
struct CompareToBucket
{
    int splitBucket, nBuckets, dim;
    const BBox &centroidBounds;

    CompareToBucket( int split, int num, int d, const BBox &b )
        : 
        centroidBounds( b )
    { 
        splitBucket = split; 
        nBuckets = num; 
        dim = d; 
    }
    
    bool operator( )( const BVHPrimitiveInfo &p ) const
    {
        int b = nBuckets * (( p.centroid[dim] - centroidBounds.pMin[dim] ) /
            ( centroidBounds.pMax[dim] - centroidBounds.pMin[dim] ) );
        if ( b == nBuckets ) 
            b = nBuckets - 1;
        assert( b >= 0 && b < nBuckets );
        return b <= splitBucket;
    }
    
};

//! utilisation interne. representation d'un noeud d'un bvh.
struct LinearBVHNode
{
    BBox bounds;
    
    union
    {
        uint32_t primitivesOffset;    // leaf
        uint32_t secondChildOffset;   // interior
    };
    
    uint8_t nPrimitives;  // 0 -> interior node
    uint8_t axis;         // interior node: xyz
    uint8_t pad[2];       // ensure 32 byte total size
};

#endif
