
#include <cmath>

#define NV_REPORT_COMPILE_ERRORS
#include <nvglutils/nvShaderUtils.h>

#include "nvGLWidgets.h"
#include "nvGLCoreUIPainter.h"


namespace nv {

#define norm255( i ) ( (float) ( i ) / 255.0f )

enum Color {
    cBase = 0,
    cBool = 4,
    cOutline = 8,
    cFont = 12,
    cFontBack = 16,
    cTranslucent = 20,
    cNbColors = 24,
};

const static nv::GLCore::vec4 s_colors[cNbColors] = {
    // cBase
    nv::GLCore::vec4( norm255(89), norm255(89), norm255(89), 0.7f ),
    nv::GLCore::vec4( norm255(166), norm255(166), norm255(166), 0.8f ),
    nv::GLCore::vec4( norm255(212), norm255(228), norm255(60), 0.5f ),
    nv::GLCore::vec4( norm255(227), norm255(237), norm255(127), 0.5f ),

    // cBool
    nv::GLCore::vec4( norm255(99), norm255(37), norm255(35), 1.0f ),
    nv::GLCore::vec4( norm255(149), norm255(55), norm255(53), 1.0f ),
    nv::GLCore::vec4( norm255(212), norm255(228), norm255(60), 1.0f ),
    nv::GLCore::vec4( norm255(227), norm255(237), norm255(127), 1.0f ),

    // cOutline
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),

    // cFont
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),

    // cFontBack
    nv::GLCore::vec4( norm255(79), norm255(129), norm255(189), 1.0 ),
    nv::GLCore::vec4( norm255(79), norm255(129), norm255(189), 1.0 ),
    nv::GLCore::vec4( norm255(128), norm255(100), norm255(162), 1.0 ),
    nv::GLCore::vec4( norm255(128), norm255(100), norm255(162), 1.0 ),

    // cTranslucent
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
};
    
    
GLCoreUIPainter::GLCoreUIPainter( TextPainter& textPainter )
    :
    GLUIPainter(textPainter)
{}
    
GLCoreUIPainter::~GLCoreUIPainter( )
{
    glDeleteProgram(m_widget_program.program);
}

static
const char *widget_vertex= {
"   #version 330    // core vertex shader\n\
    uniform mat4 projection;\n\
    layout(location= 0) in vec2 position;\n\
    layout(location= 1) in vec3 texcoord;\n\
    out vec3 vertex_texcoord;\n\
    void main( )\n\
    {\n\
        vertex_texcoord= texcoord;\n\
        gl_Position= projection * vec4(position, 0.0, 1.0);\n\
    }\n"
};

static 
const char *widget_fragment= {
"    #version 330    // core fragment shader\n\
    uniform vec4 fillColor;\n\
    uniform vec4 borderColor;\n\
    uniform vec2 zones;\n\
    in vec3 vertex_texcoord;\n\
    out vec4 fragment_color;\n\
    void main()\n\
    {\n\
        float doTurn = float(vertex_texcoord.y > 0);\n\
        float radiusOffset = doTurn * abs( vertex_texcoord.z );\n\
        float turnDir = sign( vertex_texcoord.z );\n\
        vec2 uv = vec2(vertex_texcoord.x + turnDir*radiusOffset, vertex_texcoord.y);\n\
        float l = abs( length(uv) - radiusOffset );\n\
        float a = clamp( l - zones.x, 0.0, 2.0);\n\
        float b = clamp( l - zones.y, 0.0, 2.0);\n\
        b = exp2(-2.0*b*b);\n\
        fragment_color= ( fillColor * b + (1.0-b)*borderColor );\n\
        fragment_color.a *= exp2(-2.0*a*a);\n\
    }\n\
"
};

void GLCoreUIPainter::init()
{
    GLuint vertex= nv::CompileGLSLShader(GL_VERTEX_SHADER, widget_vertex);
    GLuint fragment= nv::CompileGLSLShader(GL_FRAGMENT_SHADER, widget_fragment);
    GLuint program= nv::LinkGLSLProgram(vertex, fragment);
    if(program == 0)
        return;
    assert(glGetError() == GL_NO_ERROR);
    
    m_widget_program.program= program;
    m_widget_program.projection= glGetUniformLocation(program, "projection");
    m_widget_program.fillColor= glGetUniformLocation(program, "fillColor");
    m_widget_program.borderColor= glGetUniformLocation(program, "borderColor");
    m_widget_program.zones= glGetUniformLocation(program, "zones");
    assert(glGetError() == GL_NO_ERROR);

    m_widgets.init();
    m_textPainter.init();
}

void GLCoreUIPainter::begin( const Rect& window ) {
    m_widgets.reshape(window);
    m_textPainter.begin(window);
}

void GLCoreUIPainter::end() {
    GLboolean depth= glIsEnabled(GL_DEPTH_TEST);
    if(depth)
        glDisable(GL_DEPTH_TEST);

    GLint fill[4];
    glGetIntegerv(GL_POLYGON_MODE, fill);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    
    GLboolean blend= glIsEnabled(GL_BLEND);
    if(blend == GL_FALSE)
        glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    m_widgets.draw();
    m_widgets.clear();
    
    m_textPainter.end();
    
    if(depth)
        glEnable(GL_DEPTH_TEST);
    if(blend == GL_FALSE)
        glDisable(GL_BLEND);
    
    glPolygonMode(GL_FRONT_AND_BACK, fill[0]);
}


// Draw Primitive shapes
#if 0
void GLCoreUIPainter::drawTextureView(const Rect & rect, const void* texID, const Rect& rt, const Rect & rz, int mipLevel,
        float texelScale, float texelOffset, int r, int g, int b, int a,
        int /*style*/) {
    drawFrame( rect, Point(rt.x, rt.y), false, false, false );

    GLuint lTexID = *static_cast<const GLuint *> ( texID );

    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, lTexID);

    glUseProgram(m_textureViewProgram);
    glUniform1f( m_texMipLevelUniform, (float) mipLevel);
    glUniform1f( m_texelScaleUniform, texelScale);
    glUniform1f( m_texelOffsetUniform, texelOffset);
    glUniform4i( m_texelSwizzlingUniform, r, g, b, a);

    glBegin(GL_QUADS);
    m_widgets.push_texcoord( (float) rz.x / (float) rt.w , (float) rz.y / (float) rt.h);
    m_widgets.push_vertex(rect.x + rt.x, rect.y + rt.y);
    m_widgets.push_texcoord((float) rz.x / (float) rt.w , (float) (rz.y + rz.h) / (float) rt.h);
    m_widgets.push_vertex(rect.x + rt.x, rect.y + rt.y + rt.h);
    m_widgets.push_texcoord((float) (rz.x + rz.w) / (float) rt.w , (float) (rz.y + rz.h) / (float) rt.h);
    m_widgets.push_vertex(rect.x + rt.x + rt.w, rect.y + rt.y + rt.h);
    m_widgets.push_texcoord((float) (rz.x + rz.w) / (float) rt.w , (float) (rz.y) / (float) rt.h);
    m_widgets.push_vertex(rect.x + rt.x + rt.w, rect.y + rt.y);
    glEnd();

    glUseProgram(0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisable(GL_TEXTURE_2D);
}
#endif

void GLCoreUIPainter::drawString( int x, int y, const char * text, int nbLines, int colorId, const float *color ) {
    m_textPainter.drawString(x, y, text, nbLines, colorId, color);
}

void GLCoreUIPainter::drawRect( const Rect & rect, int fillColorId, int borderColorId ) {
    float x0 = rect.x;
    float x1 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + rect.h;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(0, 0);

    m_widgets.begin(params);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex(x0, y0);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex(x1, y0);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex(x0, y1);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex(x1, y1);
    m_widgets.end();
}

void GLCoreUIPainter::drawRoundedRect( const Rect& rect, const Point& corner, int fillColorId, int borderColorId ) {
    float xb = corner.x;
    float yb = corner.y;

    float x0 = rect.x;
    float x1 = rect.x + corner.x;
    float x2 = rect.x + rect.w - corner.x;
    float x3 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + corner.y;
    float y2 = rect.y + rect.h - corner.y;
    float y3 = rect.y + rect.h;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(corner.x - 1, corner.x - 2);

    //~ glUseProgram(m_widgetProgram);
    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, corner.x - 1, corner.x - 2);

    m_widgets.begin(params);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x0, y0);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y0);

        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x0, y1);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y1);

        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x0, y2);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y2);

        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x0, y3);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y3);
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
        
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y0);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x3, y0);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y1);
        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x3, y1);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y2);
        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x3, y2);

        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y3);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x3, y3);
        
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
        
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y0);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y0);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y1);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y1);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y2);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y2);

        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y3);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y3);
    //~ glEnd();
    m_widgets.end();
}

void GLCoreUIPainter::drawRoundedRectOutline( const Rect& rect, const Point& corner, int borderColorId ) {
    float xb = corner.x;
    float yb = corner.y;

    float x0 = rect.x;
    float x1 = rect.x + corner.x;
    float x2 = rect.x + rect.w - corner.x;
    float x3 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + corner.y;
    float y2 = rect.y + rect.h - corner.y;
    float y3 = rect.y + rect.h;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= cTranslucent;
    params.fill= s_colors[cTranslucent];
    params.zones= nv::GLCore::vec2(corner.x - 1, corner.x - 2);

    //~ glUseProgram(m_widgetProgram);
    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[cTranslucent]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, corner.x - 1, corner.x - 2);
    
    m_widgets.begin(params);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x0, y0);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y0);

        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x0, y1);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y1);

        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x0, y2);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y2);

        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x0, y3);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y3);
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
    
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y0);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x3, y0);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y1);
        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x3, y1);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y2);
        m_widgets.push_texcoord(xb, 0);
        m_widgets.push_vertex( x3, y2);

        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y3);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x3, y3);
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
    
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y0);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y0);

        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y1);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y1);
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
    
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x1, y2);
        m_widgets.push_texcoord(0, 0);
        m_widgets.push_vertex( x2, y2);

        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x1, y3);
        m_widgets.push_texcoord(0, yb);
        m_widgets.push_vertex( x2, y3);
    //~ glEnd();
    m_widgets.end();
}

void GLCoreUIPainter::drawCircle( const Rect& rect, int fillColorId, int borderColorId ) {
    float xb = rect.w / 2;
    float yb = rect.w / 2;

    float x0 = rect.x;
    float x1 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + rect.h;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2((rect.w / 2) - 1, (rect.w / 2) - 2);

    //~ glUseProgram(m_widgetProgram);

    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, (rect.w / 2) - 1, (rect.w / 2) - 2);

    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.begin(params);
        m_widgets.push_texcoord(-xb, -yb);
        m_widgets.push_vertex( x0, y0);
        m_widgets.push_texcoord(xb, -yb);
        m_widgets.push_vertex( x1, y0);
        m_widgets.push_texcoord(-xb, yb);
        m_widgets.push_vertex( x0, y1);
        m_widgets.push_texcoord(xb, yb);
        m_widgets.push_vertex( x1, y1);
    //~ glEnd();
    m_widgets.end();
}

void GLCoreUIPainter::drawMinus( const Rect& rect, int width, int fillColorId, int borderColorId ) {
    float xb = width;
    float yb = width;

    float xoff = xb ;
    float yoff = yb ;

    float x0 = rect.x + rect.w * 0.1 ;
    float x1 = rect.x + rect.w * 0.9;

    float y1 = rect.y + rect.h * 0.5;
    
    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(width - 1, width - 2);

    //~ glUseProgram(m_widgetProgram);

    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);
    
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.begin(params);
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x0, y1 + yoff);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x0, y1 - yoff);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x0 + xoff , y1 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x0 + xoff, y1 - yoff);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x1 - xoff , y1 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x1 - xoff, y1 - yoff);

        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x1, y1 + yoff);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x1, y1 - yoff);
    //~ glEnd();
    m_widgets.end();
}

void GLCoreUIPainter::drawPlus( const Rect& rect, int width, int fillColorId, int borderColorId ) {
    float xb = width;
    float yb = width;

    float xoff = xb ;
    float yoff = yb ;

    float x0 = rect.x + rect.w * 0.1 ;
    float x1 = rect.x + rect.w * 0.5;
    float x2 = rect.x + rect.w * 0.9;

    float y0 = rect.y + rect.h * 0.1;
    float y1 = rect.y + rect.h * 0.5;
    float y2 = rect.y + rect.h * 0.9;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(xb - 1, xb - 2);
    
    //~ glUseProgram(m_widgetProgram);

    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.begin(params);
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x0, y1 + yoff);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x0, y1 - yoff);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x0 + xoff , y1 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x0 + xoff, y1 - yoff);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x2 - xoff , y1 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x2 - xoff, y1 - yoff);

        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x2, y1 + yoff);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x2, y1 - yoff);
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
        
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x1 + yoff, y0);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x1 - yoff, y0);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x1 + yoff, y0 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x1 - yoff, y0 + yoff);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x1 + yoff, y2 - yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x1 - yoff, y2 - yoff);

        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x1 + yoff, y2);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x1 - yoff, y2);
    //~ glEnd();
    m_widgets.end();
}

void GLCoreUIPainter::drawDownArrow( const Rect& rect, int width, int fillColorId, int borderColorId ) {
    float offset = sqrt(2.0) / 2.0 ;

    float xb = width;
    float yb = width;

    float xoff = offset * xb ;
    float yoff = offset * yb ;
    float xoff2 = offset * xb * 2.0 ;
    float yoff2 = offset * yb * 2.0;

    float x0 = rect.x + xoff2;
    float x1 = rect.x + rect.w * 0.5;
    float x2 = rect.x + rect.w - xoff2;

    float y0 = rect.y + rect.h * 0.1 + yoff2;
    float y1 = rect.y + rect.h * 0.6;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(xb - 1, xb - 2);

    //~ glUseProgram(m_widgetProgram);

    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.begin(params);
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x0, y1 + yoff2);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x0 - xoff2, y1);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x0 + xoff, y1 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x0 - xoff, y1 - yoff);

        m_widgets.push_texcoord(-xb, 0, xb);
        m_widgets.push_vertex( x1, y0 + yoff2);
        m_widgets.push_texcoord(xb, 0, xb);
        m_widgets.push_vertex( x1 - xoff2, y0);

        m_widgets.push_texcoord(xb, 2*yb, xb);
        m_widgets.push_vertex( x1, y0 - yoff2);

    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
        
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x2 + xoff2, y1);
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x2, y1 + yoff2);

        m_widgets.push_texcoord(xb, 0, xb);
        m_widgets.push_vertex( x2 + xoff, y1 - yoff);
        m_widgets.push_texcoord(-xb, 0, xb);
        m_widgets.push_vertex( x2 - xoff, y1 + yoff);

        m_widgets.push_texcoord(xb, 0, xb);
        m_widgets.push_vertex( x1 + xoff2, y0);
        m_widgets.push_texcoord(-xb, 0, xb);
        m_widgets.push_vertex( x1, y0 + yoff2);

        m_widgets.push_texcoord(xb, 2*yb, xb);
        m_widgets.push_vertex( x1, y0 - yoff2);
    //~ glEnd();
    m_widgets.end();
}

void GLCoreUIPainter::drawUpArrow( const Rect& rect, int width, int fillColorId, int borderColorId ) {
    float offset = sqrt(2.0) / 2.0 ;

    float xb = width;
    float yb = width;

    float xoff = offset * xb ;
    float yoff = - offset * yb ;
    float xoff2 = offset * xb * 2.0 ;
    float yoff2 = - offset * yb * 2.0;

    float x0 = rect.x + xoff2;
    float x1 = rect.x + rect.w * 0.5;
    float x2 = rect.x + rect.w - xoff2;

    float y0 = rect.y + rect.h * 0.9 + yoff2;
    float y1 = rect.y + rect.h * 0.4;

    nv::GLCore::widget_params params(m_widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(xb - 1, xb - 2);
    //~ glUseProgram(m_widgetProgram);

    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.begin(params);
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x0, y1 + yoff2);
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x0 - xoff2, y1);

        m_widgets.push_texcoord(-xb, 0, 0);
        m_widgets.push_vertex( x0 + xoff, y1 + yoff);
        m_widgets.push_texcoord(xb, 0, 0);
        m_widgets.push_vertex( x0 - xoff, y1 - yoff);

        m_widgets.push_texcoord(-xb, 0, xb);
        m_widgets.push_vertex( x1, y0 + yoff2);
        m_widgets.push_texcoord(xb, 0, xb);
        m_widgets.push_vertex( x1 - xoff2, y0);

        m_widgets.push_texcoord(xb, 2*yb, xb);
        m_widgets.push_vertex( x1, y0 - yoff2);

    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    m_widgets.restart();
        m_widgets.push_texcoord(xb, -yb, 0);
        m_widgets.push_vertex( x2 + xoff2, y1);
        m_widgets.push_texcoord(-xb, -yb, 0);
        m_widgets.push_vertex( x2, y1 + yoff2);

        m_widgets.push_texcoord(xb, 0, xb);
        m_widgets.push_vertex( x2 + xoff, y1 - yoff);
        m_widgets.push_texcoord(-xb, 0, xb);
        m_widgets.push_vertex( x2 - xoff, y1 + yoff);

        m_widgets.push_texcoord(xb, 0, xb);
        m_widgets.push_vertex( x1 + xoff2, y0);
        m_widgets.push_texcoord(-xb, 0, xb);
        m_widgets.push_vertex( x1, y0 + yoff2);

        m_widgets.push_texcoord(xb, 2*yb, xb);
        m_widgets.push_vertex( x1, y0 - yoff2);
    //~ glEnd();
    m_widgets.end();
}

//~ void GLCoreUIPainter::drawDebugRect(const Rect & rect) {
    //~ glBegin(GL_LINE_STRIP);
    //~ glVertex2i( rect.x + 1, rect.y + 1);
    //~ glVertex2i( rect.x + rect.w, rect.y + 1);
    //~ glVertex2i( rect.x + rect.w, rect.y + rect.h);
    //~ glVertex2i( rect.x + 1, rect.y + rect.h);
    //~ glVertex2i( rect.x + 1, rect.y);
    //~ glEnd();
//~ }

}       // namespace nv
