
#ifndef _GK_TRIANGLEBOX_H
#define _GK_TRIANGLEBOX_H

#include "Geometry.h"
#include "RTTriangle.h"


namespace gk {
    
//! representation d'un triangle pour la construction efficace d'une partition sah
//! l'operation la plus frequente est l'obtention de la boite englobante du triangle et de son centre,
//! pMin, pMax representent les points extremes de la bbox, et p stocke les autres coordonnees, 
//! shuffle indexe pour chaque sommet dans quel point (pMin, pMax, p) se trouve chaque composante
struct TriangleBox
{
    gk::Point pMin;
    gk::Point pMax;
    gk::Point p;
    unsigned char shuffle[4];   // [abc0]= xindex | yindex | zindex
    //~ unsigned int id;

    enum
    {
        X_MASK= 2+1,
        Y_MASK= 8+4,
        Z_MASK= 32+16
    };
    
    enum
    {
        X_SHIFT= 0,
        Y_SHIFT= 2,
        Z_SHIFT= 4,
    };
    
    enum
    {
        PMIN_ID= 0,
        PMAX_ID= 1,
        P_ID= 2
    };
    
    //! retrouve les composantes d'un sommet du triangle.
    gk::Point vertex( const unsigned int v ) const
    {
        unsigned int idx= (shuffle[v] & X_MASK)>>X_SHIFT;
        float x= (&pMin)[idx].x;
        
        unsigned int idy= (shuffle[v] & Y_MASK)>>Y_SHIFT;
        float y= (&pMin)[idy].y;
        
        unsigned int idz= (shuffle[v] & Z_MASK)>>Z_SHIFT;
        float z= (&pMin)[idz].z;
        
        return gk::Point(x, y, z);
    }
    
    //! renvoie le sommet a du triangle abc.
    gk::Point a( ) const
    {
        return vertex(0);
    }
    
    //! renvoie le sommet b du triangle abc.
    gk::Point b( ) const
    {
        return vertex(1);
    }
    
    //! renvoie le sommet c du triangle abc.
    gk::Point c( ) const
    {
        return vertex(2);
    }
    
    unsigned int get_shuffle( const unsigned int shift, const float v, const float min, const float max, float& p )
    {
        if(v == min)
            return (PMIN_ID<<shift);
        
        else if(v == max)
            return (PMAX_ID<<shift);
        
        p= v;
        return (P_ID<<shift);
    }
    
    void init( const gk::Point& a, const gk::Point& b, const gk::Point& c )
    {
        BBox bounds(a, b);
        bounds.Union(c);
        pMin= bounds.pMin;
        pMax= bounds.pMax;
        
        shuffle[0]= get_shuffle(X_SHIFT, a.x, pMin.x, pMax.x, p.x);
        shuffle[0]|= get_shuffle(Y_SHIFT, a.y, pMin.y, pMax.y, p.y);
        shuffle[0]|= get_shuffle(Z_SHIFT, a.z, pMin.z, pMax.z, p.z);

        shuffle[1]= get_shuffle(X_SHIFT, b.x, pMin.x, pMax.x, p.x);
        shuffle[1]|= get_shuffle(Y_SHIFT, b.y, pMin.y, pMax.y, p.y);
        shuffle[1]|= get_shuffle(Z_SHIFT, b.z, pMin.z, pMax.z, p.z);
        
        shuffle[2]= get_shuffle(X_SHIFT, c.x, pMin.x, pMax.x, p.x);
        shuffle[2]|= get_shuffle(Y_SHIFT, c.y, pMin.y, pMax.y, p.y);
        shuffle[2]|= get_shuffle(Z_SHIFT, c.z, pMin.z, pMax.z, p.z);
        
        shuffle[3]= 0;
        
    #if 0
        printf("pMin "); pMin.print();
        printf("pMax "); pMax.print();
        printf("p "); p.print();
        
        printf("a "); a.print();
        printf("  shuffle %d %d %d, values ", 
            (shuffle[0] & X_MASK)>>X_SHIFT, (shuffle[0] & Y_MASK)>>Y_SHIFT, (shuffle[0] & Z_MASK)>>Z_SHIFT);
        vertex(0).print();
        
        printf("b "); b.print();
        printf("  shuffle %d %d %d, values ", 
            (shuffle[1] & X_MASK)>>X_SHIFT, (shuffle[1] & Y_MASK)>>Y_SHIFT, (shuffle[2] & Z_MASK)>>Z_SHIFT);
        vertex(1).print();
        printf("c "); c.print();
        printf("  shuffle %d %d %d, values ", 
            (shuffle[2] & X_MASK)>>X_SHIFT, (shuffle[2] & Y_MASK)>>Y_SHIFT, (shuffle[2] & Z_MASK)>>Z_SHIFT);
        vertex(2).print();
    #endif
    }
    
    //! constructeur par defaut.
    TriangleBox( ) 
        //~ :
        //~ id(-1)
    {}
    
    //! construit un triangle connaissant ses 3 sommets.
    TriangleBox( const gk::Point& _a, const gk::Point& _b, const gk::Point& _c, const unsigned int _id= -1 )
        //~ :
        //~ id(_id)
    {
        init(_a, _b, _c);
        assert(a() == _a);
        assert(b() == _b);
        assert(c() == _c);
    }
    
    //! construit un triangle a partir d'un triangle 'geometrique'.
    TriangleBox( const gk::Triangle& triangle, const unsigned int _id= -1 )
        //~ :
        //~ id(_id)
    {
        init(triangle.a, triangle.b, triangle.c);
        assert(a() == triangle.a);
        assert(b() == triangle.b);
        assert(c() == triangle.c);
    }
    
    //! destructeur.
    ~TriangleBox( ) {}
 
    //! renvoie la bbox du triangle.
    gk::BBox getBBox( ) const
    {
        gk::BBox bbox;
        bbox.pMin= pMin;
        bbox.pMax= pMax;
        
        return bbox;
    }
    
    //! renvoie une composante du centre de la bbox du triangle.
    float getBBoxCenter( const int axis ) const
    {
        return .5f * (pMin[axis] + pMax[axis]);
    }

    //! intersection avec un rayon.
    //! renvoie faux s'il n'y a pas d'intersection, une intersection peut exister mais peut ne pas se trouver dans [tmin tmax] du rayon.
    //! renvoie vrai + les coordonnees barycentriques du point d'intersection + sa position le long du rayon.
    //! convention barycentrique : t(u, v)= (1 - u - v) * a + u * b + v * c
    /*! "Fast, Minimum Storage Ray/Triangle Intersection"
        cf http://www.acm.org/jgt/papers/MollerTrumbore97/
    */
    bool Intersect( const gk::BasicRay &ray, const float htmax, 
        float &rt, float &ru, float&rv ) const
    {
        // solution naive : extrait les composantes des sommets abc et renvoie un objet triangle classique.
        return gk::RTTriangle(a(), b(), c()).Intersect(ray, htmax, rt, ru, rv);
    }
};

}

#endif
