
#ifndef _IOTEXTFILE_H
#define _IOTEXTFILE_H

#include "IOManager.h"
#include "IOFileManager.h"
#include "TextFile.h"


namespace gk {

//! manager pour importer des ficheirs textes, sources, shaders, etc.
class TextFileIO : public IOManager<TextFile>
{
    // non copyable
    TextFileIO( const TextFileIO& );
    TextFileIO& operator=( const TextFileIO& );
    
    // private default constructor, singleton
    TextFileIO( )
        :
        IOManager<TextFile>()
    {}
    
    ~TextFileIO( ) {}
    
public:
    //! importe l'objet 'name' du fichier 'filename'.
    static
    TextFile *read( const std::string& filename, const std::string& name= "" ) 
    {
        // importer le fichier, si necessaire
        if(filename.empty() == true)
            return NULL;
        
        IOFileInfo handle= IOFileManager::manager().file(filename);
        if(handle.isValid() == false)
            return NULL;
        TextFile *file= manager().find(handle, name);
        if(file != NULL)
            return file;
        
        // importer le fichier
    #ifdef VERBOSE      //_DEBUG
        printf("loading '%s'...\n", filename.c_str());
    #endif
        file= new TextFile(filename);
        if(file->read(filename) < 0)
        {
        #ifdef VERBOSE      //_DEBUG
            printf("loading '%s'... failed.\n", filename.c_str());
        #endif
            delete file;
            return NULL;
        }
        
        // reference l'objet avec le manager
        return manager().insert(file, handle, name);
    }

    //! renvoie -1 en cas d'erreur, 1 lorsque le fichier a ete recharge, et 0 sinon (le fichier n'a pas change).
    static
    int reload( TextFile *object )
    {
        if(object == NULL)
            return -1;
        
        IOName *ioname= manager().find(object);
        if(ioname == NULL || ioname->handle().isValid() == false)
        {
            // le fichier n'est pas reference ... pas lu en passant par le manager.
            printf("reloading textfile %p... failed.\n", object);
            return 0;
        }
        
        // verifie la date et la taille du fichier
        if(ioname->handle().modified() <= 0)
            // le fichier n'est pas modifie
            return 0;
        
        // recharger le fichier
        object->clear();
        if(object->read(ioname->filename()) < 0)
        {
        #ifdef VERBOSE      //_DEBUG
            printf("reloading '%s'... failed.\n", ioname->filename().c_str());
        #endif
            return -1;
        }
        
        // mettre a jour les infos sur le fichier
        ioname->handle().update();
    #ifdef VERBOSE      //_DEBUG
        printf("reloading '%s'... done.\n", ioname->filename().c_str());
    #endif
        return 1;
    }
    
    static
    TextFile *reload( const std::string& filename, const std::string& name= "" ) 
    {
        if(filename.empty() == true)
            return NULL;
        TextFile *object= manager().find(filename, name);
        int code= reload(object);
        if(code < 0)
            return NULL;
        return object;
    }
    
    static
    TextFileIO& manager( )  //!< singleton
    {
        static TextFileIO manager;
        return manager;
    }
};

} // namespace

#endif
