
#ifndef _RT_TRIANGLE_H
#define _RT_TRIANGLE_H

#include "Triangle.h"
#include "Geometry.h"

namespace gk {

//! representation d'un triangle pour l'intersecteur rayon / triangle.
struct RTTriangle
{
    Point pa;
    Vector ab;
    Vector ac;
    unsigned int id;
    
    //! constructeur par defaut.
    RTTriangle( ) {}
    
    //! construit un triangle connaissant ses 3 sommets.
    RTTriangle( const Point& _a, const Point& _b, const Point& _c )
        :
        pa(_a),  ab(_a, _b), ac(_a, _c)
    {}
    
    //! construit un triangle a partir d'un triangle 'geometrique'.
    RTTriangle( const Triangle& triangle )
        :
        pa(triangle.a),  ab(triangle.a, triangle.b), ac(triangle.a, triangle.c)
    {}
    
    //! destructeur.
    ~RTTriangle( ) {}
    
    gk::Point a( ) const
    {
        return pa;
    }
    
    gk::Point b( ) const
    {
        return pa + ab;
    }
    
    gk::Point c( ) const
    {
        return pa + ac;
    }
    
    float getArea( ) const
    {
        return .5f * Cross(ab, ac).Length();
    }
    
    BBox getBBox( ) const
    {
        BBox bbox(pa, pa+ab);
        bbox.Union(pa + ac);
        return bbox;
    }
    
    float getBBoxCenter( const int axis ) const
    {
        float pMin ; 
        float pMax;
        pMin= std::min(pa[axis], pa[axis] + ab[axis]);
        pMin= std::min(pMin, pa[axis] + ac[axis]);
        pMax= std::max(pa[axis], pa[axis] + ab[axis]);
        pMax= std::max(pMax, pa[axis] + ac[axis]);
        return .5f * (pMin + pMax);
    }
    
    Point getCenter( ) const
    {
        return pa + (ab + ac) / 3.f;
    }
    
    float getCenter( const int axis ) const
    {
        return pa[axis] + (ab[axis] + ac[axis]) / 3.f;
    }
    
    //! calcule la normale geometrique du triangle.
    void getTriangleNormal( Normal &n ) const
    {
        n= Normal( Normalize(Cross(ab, ac)) ) ;
    }

    Normal getTriangleNormal( ) const
    {
        return Normal( Normalize( Cross(ab, ac) ));
    }
    
    //! intersection avec un rayon.
    //! renvoie faux s'il n'y a pas d'intersection, une intersection peut exister mais peut ne pas se trouver dans [tmin tmax] du rayon.
    //! renvoie vrai + les coordonnees barycentriques du point d'intersection + sa position le long du rayon.
    //! convention barycentrique : t(u, v)= (1 - u - v) * a + u * b + v * c
    /*! "Fast, Minimum Storage Ray/Triangle Intersection"
        cf http://www.acm.org/jgt/papers/MollerTrumbore97/
    */
    bool Intersect( const Ray &ray, const float htmax, 
        float &rt, float &ru, float&rv ) const
    {
        /* begin calculating determinant - also used to calculate U parameter */
        const Vector pvec= Cross(ray.d, ac);
        
        /* if determinant is near zero, ray lies in plane of triangle */
        const float det= Dot(ab, pvec);
        if (det > -EPSILON && det < EPSILON)
            return false;
        
        const float inv_det= 1.0f / det;
        
        /* calculate distance from vert0 to ray origin */
        const Vector tvec(pa, ray.o);
        
        /* calculate U parameter and test bounds */
        const float u= Dot(tvec, pvec) * inv_det;
        if(u < 0.0f || u > 1.0f)
            return false;
        
        /* prepare to test V parameter */
        const Vector qvec= Cross(tvec, ab);
        
        /* calculate V parameter and test bounds */
        const float v= Dot(ray.d, qvec) * inv_det;
        if(v < 0.0f || u + v > 1.0f)
            return false;
        
        /* calculate t, ray intersects triangle */
        rt= Dot(ac, qvec) * inv_det;
        ru= u;
        rv= v;
        
        // ne renvoie vrai que si l'intersection est valide (comprise entre tmin et tmax du rayon)
        //~ return (rt < htmax && rt > ray.tmin);
        return (rt < htmax && rt > RAY_EPSILON);
    }
    
    bool Intersect( const BasicRay &ray, const float htmax, 
        float &rt, float &ru, float&rv ) const
    {
        /* begin calculating determinant - also used to calculate U parameter */
        const Vector pvec= Cross(ray.d, ac);
        
        /* if determinant is near zero, ray lies in plane of triangle */
        const float det= Dot(ab, pvec);
        if (det > -EPSILON && det < EPSILON)
            return false;
        
        const float inv_det= 1.0f / det;
        
        /* calculate distance from vert0 to ray origin */
        const Vector tvec(pa, ray.o);
        
        /* calculate U parameter and test bounds */
        const float u= Dot(tvec, pvec) * inv_det;
        if(u < 0.0f || u > 1.0f)
            return false;
        
        /* prepare to test V parameter */
        const Vector qvec= Cross(tvec, ab);
        
        /* calculate V parameter and test bounds */
        const float v= Dot(ray.d, qvec) * inv_det;
        if(v < 0.0f || u + v > 1.0f)
            return false;
        
        /* calculate t, ray intersects triangle */
        rt= Dot(ac, qvec) * inv_det;
        ru= u;
        rv= v;
        
        // ne renvoie vrai que si l'intersection est valide (comprise entre tmin et tmax du rayon)
        //~ return (rt < htmax && rt > ray.tmin);
        return (rt < htmax && rt > RAY_EPSILON);
    }
};

}

#endif
