
#ifndef _MESHOBJ_H
#define _MESHOBJ_H

#include <string>

namespace gk {

class Mesh;
//! renvoie vrai si 'filename' se termine par '.obj'.
bool isMeshOBJ( const std::string& filename );

int MeshLoadFromOBJ( const std::string& filename, Mesh *mesh );
int MaterialLoadFromMTL( const std::string& filename );

int MeshWriteToOBJ( Mesh *mesh, const std::string& filename );
int MeshMaterialsMTLLIB( const Mesh *mesh, std::string& mtllib );
int MeshMaterialsWriteToMTL( const Mesh *mesh, const std::string& filename );

#if 0
//! namespace prive.
/*! namespace prive, pour isoler les classes utilisees pour charger les objets maya .obj et leurs matieres .mtl.
    
    chargement d'un objet maya obj + construction lineaire d'un maillage triangule.

    cf http://local.wasp.uwa.edu.au/~pbourke/dataformats/obj/
    et  http://local.wasp.uwa.edu.au/~pbourke/dataformats/mtl/
 */    
namespace OBJ
{

//! parser pour analyser les fichiers obj et mtl.
class Parser
{
    FILE *m_in;
    std::string m_token;
    
    bool is_token( const char c )
    {
        return ( isalnum( c ) || c == '_' || c == '-' || c == '.' );
    }
    
public:
    Parser( const std::string& filename )
            :
            m_in( NULL ),
            m_token()
    {
        m_in = fopen( filename.c_str(), "rt" );
    }
    
    ~Parser( )
    {
        if ( m_in != NULL )
            fclose( m_in );
    }
    
    bool isValid( ) const
    {
        return ( m_in != NULL );
    }
    
    //! lecture sequentielle d'un 'mot' dans le fichier.
    //! un mot est un ensemble de caracteres quelconques delimite par des separateurs.
    //! renvoie le separateur, si la lecture d'un mot est impossible.
    int readToken( )
    {
        if ( !isValid() )
            return EOF;
            
        // sauter les blancs
        char c = fgetc( m_in );
        while ( c != EOF && ( c == ' ' || c == '\t' ) )
            c = fgetc( m_in );
            
        if ( c == '\r' )
            c = '\n'; // meme gestion des fins de lignes pour linux, mac et windows
            
        // lire tous les caracteres alphanumeriques
        m_token.resize( 0 );
        while ( c != EOF && is_token( c ) )
        {
            m_token.push_back( c );
            
            c = fgetc( m_in );
            if ( c == '\r' )
                c = '\n'; // meme gestion des fins de lignes pour linux, mac et windows
        }
        
        // separateurs
        if ( c == '#' )
        {
            if ( m_token.empty() )
                m_token.push_back( c );
            else
                ungetc( c, m_in );
        }
        else if ( c == '/' ) // ambiguite sur les noms de fichiers ...
            m_token.push_back( c );
            
        // indiquer la fin de la ligne ou du fichier
        if ( m_token.empty() )
            return c;
            
        // forcer une fin de ligne avant la fin du fichier
        if ( c == EOF )
            ungetc( '\n', m_in );
            
        else if ( c == '\n' )
            ungetc( c, m_in );
            
        return 0;
    }
    
    //! lecture de la fin de la ligne, renvoie la chaine de caracteres.
    int readString( )
    {
        std::string string;
        
        // concatene tous les tokens jusqu'a la fin de la ligne
#if 0
        while ( readToken() != '\n' )
            string += m_token;
#else
        // ajoute aussi les separateurs dans la chaine ...
        int code = readToken();
        while ( code != '\n' )
        {
            string.append( m_token );
            if ( code != 0 )
                string.push_back( code );
            code = readToken();
        }
#endif
            
        assert( string.length() == strlen( string.c_str() ) );
        
        m_token.swap( string );
        if ( m_token.empty() )
            return -1;
        else
            return 0;
    }
    
    //! renvoie le dernier 'mot' lu par readToken().
    const std::string& getToken( ) const
    {
        return m_token;
    }
    
    //! renvoie le dernier caractere du dernier 'mot' lu.
    char getLastChar( ) const
    {
        const int length = ( int ) m_token.length();
        if ( length == 0 )
            return 0;
            
        return m_token[length -1];
    }
    
    //! renvoie un caractere du dernier 'mot' lu.
    char getChar( const int i = 0 ) const
    {
        assert( i >= 0 && i < ( int ) m_token.length() );
        return m_token[i];
    }
    
    //! converti le dernier 'mot' lu en reel.
    int getFloat( float& v ) const
    {
        v = 0.f;
        if ( sscanf( m_token.c_str(), "%f", &v ) != 1 )
            return -1;
        else
            return 0;
    }
    
    //! converti le dernier 'mot' lu en entier.
    int getInt( int& v ) const
    {
        v = 0;
        if ( sscanf( m_token.c_str(), "%d", &v ) != 1 )
            return -1;
        else
            return 0;
    }
    
    //! lit un vecteur 3.
    int readVector3( Vector& v )
    {
        int i;
        for ( i = 0; readToken() != '\n'; i++ )
            if ( i < 3 && getFloat( v[i] ) < 0 )
                return -1;
                
        if ( i < 3 )
            return -1;
        else
            return 0;
    }
    
    //! lit un vecteur 2.
    int readVector2( Point2& v )
    {
        int i;
        for ( i = 0; readToken() != '\n'; i++ )
            if ( i < 2 && getFloat( v[i] ) < 0 )
                return -1;
                
        if ( i < 2 )
            return -1;
        else
            return 0;
    }
    
    //! saute la ligne.
    //! lecture de tous les 'mots' jusqu'a la fin de la ligne.
    int skipLine( )
    {
        while ( readToken() != '\n' )
        {
        #ifdef VERBOSE_DEBUG
            printf("skip '%s'\n", getToken().c_str());
        #endif
        }
            
        return 0;
    }
};

//! determine l'indice d'un attribut de sommet dans le maillage.
//! renvoie 0 si l'attribut n'existe pas, 1 si l'attribut existe et -1 en cas d'erreur.
inline
int getAttribute( const Parser& parser, int& attr, const int attributes_n )
{
    if ( parser.getChar( 0 ) == '/' )
        return 0;
        
    if ( parser.getInt( attr ) < 0 )
        return -1;
        
    if ( attr < 0 )
        attr += attributes_n;
    else
        attr -= 1;
        
    if ( attr < 0 || attr >= attributes_n )
        return -1;
    else
        return 1;
}


//! representation d'un sommet, indice de matiere + triplet d'indices position, normale et texcoord.
struct Vertex
{
    int m_indices[4];
    
    Vertex( ) {}
    
    Vertex( const int m, const int p, const int n, const int t )
    {
        m_indices[0] = m;
        m_indices[1] = p;
        m_indices[2] = n;
        m_indices[3] = t;
    }
    
    ~Vertex( ) {}
    
    //! comparaison de 2 sommets pour l'insertion dans une std::map.
    static
    bool less( const Vertex& a, const Vertex& b )
    {
        // definit un ordre lexicographique sur la matiere + les 3 indices : position, normale, texcoord
        for ( int i = 0; i < 4; i++ )
        {
            if ( a.m_indices[i] < b.m_indices[i] )
                // renvoie vrai lorsque a < b
                return true;
            else if ( a.m_indices[i] > b.m_indices[i] )
                return false;
        }
        
        return false;
    }
    
    //! renvoie l'indice de la matiere du sommet.
    int material( ) const
    {
        return m_indices[0];
    }
    
    //! renvoie l'indice de la position du sommet.
    int position( ) const
    {
        return m_indices[1];
    }
    
    //! renvoie l'indice de la normale du sommet.
    int normal( ) const
    {
        return m_indices[2];
    }
    
    //! renvoie l'indice de la coordonnee de texture du sommet.
    int texcoord( ) const
    {
        return m_indices[3];
    }
};

//! operateur de comparaison pour l'insertion d'un Vertex dans une std::map.
inline
bool operator<( const Vertex& a, const Vertex& b )
{
    return Vertex::less( a, b );
}

//! representation d'un triangle <abc> pour le maillage.
struct Triangle
{
    int m_indices[3];
    int m_material_id;  // a stocker a part ... cf. construction de l'index buffer
    int m_smooth_id;
    
    //! contructeur par defaut.
    Triangle( )
            :
            m_material_id( -1 ),
            m_smooth_id( -1 )
    {}
    
    //! destructeur.
    ~Triangle( ) {}
    
    //! construit un triangle.
    Triangle( const int a, const int b, const int c )
    {
        m_indices[0] = a;
        m_indices[1] = b;
        m_indices[2] = c;
    }
    
    //! fixe l'identifiant de la matiere du triangle.
    void setMaterial( const int id )
    {
        m_material_id = id;
    }
    
    //! fixe le smooth group du triangle
    void setSmoothGroup( const int group_id )
    {
        m_smooth_id = group_id;
    }
    
    //! renvoie l'indice du sommet a du triangle.
    int a( ) const
    {
        return m_indices[0];
    }
    
    //! renvoie l'indice du sommet b du triangle.
    int b( ) const
    {
        return m_indices[1];
    }
    
    //! renvoie l'indice du sommet c du triangle.
    int c( ) const
    {
        return m_indices[2];
    }
    
    //! renvoie l'identifiant de la matiere du triangle.
    int material( ) const
    {
        return m_material_id;
    }
    
    //! renvoie l'identifiant du smooth group du triangle.
    int smoothGroup( ) const
    {
        return m_smooth_id;
    }
    
    //! comparaison de 2 sommets pour l'insertion dans une std::map
    static
    bool material_less( const Triangle& a, const Triangle& b )
    {
        return ( a.material() < b.material() );
    }
};

} // namespace OBJ
#endif

}

#endif
