
#ifndef _IOFILESYSTEM_H
#define _IOFILESYSTEM_H

#ifdef APPLE_OSX
    #include <sys/stat.h>
    
#elif defined WIN32
    #include <sys/types.h>
    #include <sys/stat.h>
    
#else
    #include <sys/types.h>
    #include <sys/stat.h>
    #include <unistd.h>
#endif

#include <cstdio>
#include <string>


namespace gk {

//! representation des informations sur un fichier.
struct IOInfo
{
    size_t size;        //!< longueur en octets.
    size_t time;        //!< date du fichier.
    
    IOInfo( )
        :
        size(0),
        time(0)
    {}
    
    bool operator == ( const IOInfo& b ) const
    {
        return (size == b.size && time == b.time);
    }
    
    bool operator != ( const IOInfo& b ) const
    {
        return !(*this == b);
    }
};

struct IOFileSystem
{
    //! renvoie le chemin d'acces a un fichier.

    /*! le chemin est toujours termine par /
     pathname("path/to/file") == "path/to/"
     pathname("file") == "./"
     */
    static
    std::string pathname( const std::string& filename )
    {
        size_t slash = filename.find_last_of( '/' );
        size_t bslash = filename.find_last_of( '\\' );
        
        if ( slash == std::string::npos && bslash != std::string::npos )
            slash = bslash;
        else if ( slash != std::string::npos && bslash != std::string::npos && bslash > slash )
            slash = bslash;
            
        if ( slash != std::string::npos )
            return filename.substr( 0, slash +1 ); // inclus le slash
        else
            return "./";
    }
    
    //! renvoie le nom du fichier sans son extension (chemin inclus).
    static
    std::string basename( const std::string& filename )
    {
        size_t ext= filename.find_last_of('.');
        if(ext== std::string::npos)
            return filename;    // renvoie le nom tel quel
        else
            return filename.substr(0, ext);  // n'inclus pas le point
    }
    
    //! verifie que le fichier est bien du type represente par 'suffix'.
    static
    bool isType( const std::string& filename, const std::string& suffix )
    {
        size_t ext= filename.find_last_of('.');
        if(ext != std::string::npos)
            return (filename.substr(ext, std::string::npos).rfind(suffix) != std::string::npos);
        else
            return (filename.rfind(suffix) != std::string::npos);
    }
    
    //! change l'extension du fichier.
    static
    std::string changeType( const std::string& filename, const std::string& suffix )
    {
        size_t ext= filename.find_last_of('.');
        if(ext == std::string::npos)
            return filename + suffix;
        else
            return filename.substr(0, ext) + suffix;
    }
    
    //! verifie l'existance d'un fichier.
    static
    int exists( const std::string& filename )
    {
    #ifndef WIN32
        struct stat info;
        if(stat(filename.c_str(), &info) < 0)
            return -1;
        
        // verifie aussi que c'est bien un fichier standard
        return (S_ISREG(info.st_mode)) ? 0 : -1;
        
    #else
        //! \todo a modifier pour compiler avec codeblock/mingw sous windows, utiliser la version linux.
        struct _stat64 info;
        if(_stat64(filename.c_str(), &info) < 0)
            return -1;
        
        // verifie aussi que c'est bien un fichier standard
        return ((info.st_mode & _S_IFREG) == _S_IFREG);
    #endif
    }
    
    //! renvoie les informations taille et date sur un fichier. 
    //! \param size peut etre NULL si l'information n'est pas souhaitee. 
    //! \param time peut etre NULL si l'information n'est pas souhaitee.
    static
    int infos( const std::string& filename, size_t *size, size_t *time )
    {
    #ifndef WIN32
        struct stat info;
        if(stat(filename.c_str(), &info) < 0)
            return -1;
        
        if(size != NULL)
            *size= info.st_size;
        if(time != NULL)
            *time= info.st_mtime;
        return 0;
        
    #else
        //! \todo a modifier pour compiler avec codeblock/mingw sous windows, utiliser la version linux.
        struct _stat64 info;
        if(_stat64(filename.c_str(), &info) < 0)
            return -1;
        
        if(size != NULL)
            *size= info.st_size;
        if(time != NULL)
            *time= info.st_mtime;
        return 0;
    #endif
    }
    
    //! renvoie les informations sur un fichier.
    //! renvoie -1 en cas d'erreur (le fichier n'existe pas, par exemple).
    static
    int infos( const std::string& filename, IOInfo& info )
    {
        return infos(filename, &info.size, &info.time);
    }

    //! renvoie 1 si le fichier a ete modifie depuis qu'il a ete lu, 0 sinon, et -1 en cas d'erreur.
    static
    int modified( const std::string& filename, const IOInfo& last_info )
    {
        IOInfo info;
        if(infos(filename, info) < 0)
            return -1;
        
        return (last_info != info) ? 1 : 0;
    }
};

}

#endif
