
#ifndef _IOFILE_MANAGER_H
#define _IOFILE_MANAGER_H

#include <string>
#include <vector>
#include <map>

#include "IOFile.h"


namespace gk {
    
//! representation de l'ensemble de fichiers ouverts par l'application.
class IOFileManager
{
    // non copyable
    IOFileManager( const IOFileManager& );
    IOFileManager& operator=( const IOFileManager& );
    
protected:
    typedef std::map<std::string, unsigned int> files_map_type;
    files_map_type m_files_map;
    std::vector<IOFile *> m_files;

    IOFileManager( )
        :
        m_files()
    {}

    virtual ~IOFileManager( ) 
    {
        const int n= m_files.size();
        for(int i= 0; i < n; i++)
            delete m_files[i];
    }

public:
    //! renvoie un description du fichier 'filename'.
    IOFileInfo file( const std::string& filename )
    {
        std::pair<files_map_type::iterator, bool>
            found= m_files_map.insert( std::make_pair(filename, m_files.size()) );
        if(found.second == false)
            return IOFileInfo(m_files[found.first->second]);
        
        // insertion du nouveau fichier
        IOInfo info;
        if(IOFileSystem::infos(filename, info) < 0)
            info= IOInfo();
        
    #ifdef VERBOSE_DEBUG
        printf("IOFileManager::insert('%s') %d: length %lu, time %lu\n",
            filename.c_str(), (int) m_files.size(), info.size, info.time);
    #endif
        
        m_files.push_back( new IOFile(filename, info) );
        return IOFileInfo(m_files.back());
    }
    
    //! renvoie 1 si le fichier a ete modifie.
    int modified( const IOFileInfo& b )
    {
        if(b.file == NULL)
            return -1;
        // compare la date des infos a celles du fichier
        return (b.info != b.file->info) ? 1 : 0;
    }
    
    //! met a jour le descripteur.
    int update( IOFileInfo& b )
    {
        if(b.file == NULL)
            return -1;
        // renvoie les nouvelles infos
        b= IOFileInfo(b.file);
        return 0;
    }

    //! met a jour les descripteurs de l'ensemble de fichiers ouverts par l'application.
    int reload( )
    {
    #ifdef VERBOSE
        printf("scanning files...\n");
    #endif
        const int n= m_files.size();
        for(int i= 0; i < n; i++)
        {
            if(m_files[i] == NULL || m_files[i]->filename.empty())
                continue;
            
            IOInfo info;
            if(IOFileSystem::infos(m_files[i]->filename, info) < 0)
                // erreur ?
                continue;
            
            // ne pas modifier les infos en cas d'erreur
        #ifdef VERBOSE  //_DEBUG
            printf("  %s: ", m_files[i]->filename.c_str());
            if(info != m_files[i]->info)
                printf("modified.\n");
        #endif
            
        #if VERBOSE_DEBUG
            else
                printf("no modifications.\n");
        #endif
            
            m_files[i]->info= info;
        }
        
    #ifdef VERBOSE
        printf("done.\n");
    #endif
        return 0;
    }
    
    static 
    IOFileManager& manager( )   //!< singleton
    {
        static IOFileManager manager;
        return manager;
    }
};

}       // namespace gk

#endif
