
#include <cstdio>

#include "GL/TPTexture.h"


namespace gk {

TextureFormat TextureRGBA= TextureFormat( GL_RGBA, GL_RGBA, GL_UNSIGNED_BYTE );
TextureFormat TextureRGBA16F= TextureFormat( GL_RGBA16F, GL_RGBA, GL_FLOAT );
TextureFormat TextureRGBA32F= TextureFormat( GL_RGBA32F, GL_RGBA, GL_FLOAT );

TextureFormat TextureDepth= TextureFormat( GL_DEPTH_COMPONENT, GL_DEPTH_COMPONENT, GL_FLOAT );
TextureFormat TextureDepth24= TextureFormat( GL_DEPTH_COMPONENT24, GL_DEPTH_COMPONENT, GL_FLOAT );
TextureFormat TextureDepth32= TextureFormat( GL_DEPTH_COMPONENT32, GL_DEPTH_COMPONENT, GL_FLOAT );

TextureFormat TextureR32F= TextureFormat( GL_R32F, GL_RED, GL_FLOAT );
TextureFormat TextureRG32F= TextureFormat( GL_RG32F, GL_RG, GL_FLOAT );
TextureFormat TextureR16F= TextureFormat( GL_R16F, GL_RED, GL_FLOAT );
TextureFormat TextureRG16F= TextureFormat( GL_RG16F, GL_RG, GL_FLOAT );
    
TextureFormat TextureR32UI= TextureFormat( GL_R32UI, GL_RED_INTEGER, GL_UNSIGNED_INT );
TextureFormat TextureR16UI= TextureFormat( GL_R16UI, GL_RED_INTEGER, GL_UNSIGNED_SHORT );
TextureFormat TextureRG16UI= TextureFormat( GL_RG16UI, GL_RG_INTEGER, GL_UNSIGNED_SHORT );

TextureFormat TextureRGBA_MS4= TextureFormat( GL_RGBA, GL_RGBA, GL_UNSIGNED_BYTE, 4 );
TextureFormat TextureDepth_MS4= TextureFormat( GL_DEPTH_COMPONENT, GL_DEPTH_COMPONENT, GL_FLOAT, 4 );


Image *GLTexture::getImage( const int unit, const int level ) const
{
    // verifie le type des pixels de la texture
    if(m_format.data_type != GL_UNSIGNED_BYTE)
    {
        printf("GLTexture::getImage( ): not a rgba8 texture. failed.\n");
        return NULL;
    }
    if(m_format.samples > 0)
    {
        printf("GLTexture::getImage( ): multisample texture. failed.\n");
        return NULL;
    }
    
    if(count() > 1)
    {
        printf("GLTexture::getImage( ): texture arrays not supported. failed.\n");
        return NULL;
    }
    
    // recupere les dimensions de l'image 
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    int w, h, d;
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_WIDTH, &w);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_HEIGHT, &h);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_DEPTH, &d);
    if(d > 1)
    {
        printf("GLTexture::getImage( ): texture array not implemented.\n");
        return NULL;
    }
    if(w < 1 && h < 1)
        return NULL;
    
    Image *image= new Image(w, h);
    // glPixelStore(GL_PACK_ALIGNEMENT ...) ?
    glGetTexImage(m_target, level, GL_RGBA, GL_UNSIGNED_BYTE, image->data());
    return image;
}

ImageArray *GLTexture::getImageArray( const int unit, const int level ) const
{
    // verifie le type des pixels de la texture
    if(m_format.data_type != GL_UNSIGNED_BYTE)
    {
        printf("GLTexture::getImageArray( ): not a rgba8 texture. failed.\n");
        return NULL;
    }
    if(m_format.samples > 0)
    {
        printf("GLTexture::getImage( ): multisample texture. failed.\n");
        return NULL;
    }

    if(m_target == GL_TEXTURE_CUBE_MAP)
    {
        printf("GLTexture::getImageCube( ): not implemented.\n");
        return NULL;
    }
    
    // recupere les dimensions de l'image 
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    int w, h, d;
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_WIDTH, &w);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_HEIGHT, &h);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_DEPTH, &d);
    
    // recupere les pixels
    // glPixelStore(GL_PACK_ALIGNEMENT ...) ?
    ImageArray *image= new ImageArray(d);
    unsigned char *data= new unsigned char [4 * w * h * d];
    glGetTexImage(m_target, level, GL_RGBA, GL_UNSIGNED_BYTE, data);
    
    // re-organise les pixels
    for(int i= 0; i < d; i++)
        image->push_back( new Image(w, h, data + (4 * w * h) * i) );
    
    delete [] data;
    return image;
}

HDRImage *GLTexture::getHDRImage( const int unit, const int level ) const
{
    // verifie le type des pixels de la texture
    if(m_format.data_type != GL_FLOAT || m_format.internal == GL_DEPTH_COMPONENT)
    {
        printf("GLTexture::getImage( ): not a rgba32f texture. failed.\n");
        return NULL;
    }
    if(m_format.samples > 0)
    {
        printf("GLTexture::getImage( ): multisample texture. failed.\n");
        return NULL;
    }
        
    if(count() > 1)
    {
        printf("GLTexture::getHDRImage( ): texture arrays not supported. failed.\n");
        return NULL;
    }
        
    // recupere les dimensions de l'image 
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    int w, h, d;
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_WIDTH, &w);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_HEIGHT, &h);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_DEPTH, &d);
    if(d > 1)
    {
        printf("GLTexture::getImage( ): texture array not implemented.\n");
        return NULL;
    }
    
    HDRImage *image= new HDRImage(w, h);
    // glPixelStore(GL_PACK_ALIGNEMENT ...) ?
    glGetTexImage(m_target, level, GL_RGBA, GL_FLOAT, image->data());
    return image;
}

HDRImageArray *GLTexture::getHDRImageArray( const int unit, const int level ) const
{
    // verifie le type des pixels de la texture
    if(m_format.data_type != GL_FLOAT || m_format.internal == GL_DEPTH_COMPONENT)
    {
        printf("GLTexture::getImageArray( ): not a rgba32f texture. failed.\n");
        return NULL;
    }
    if(m_format.samples > 0)
    {
        printf("GLTexture::getImage( ): multisample texture. failed.\n");
        return NULL;
    }
    
    if(m_target == GL_TEXTURE_CUBE_MAP)
    {
        printf("GLTexture::getHDRImageCube( ): not implemented.\n");
        return NULL;
    }
    
    // recupere les dimensions de l'image 
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    int w, h, d;
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_WIDTH, &w);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_HEIGHT, &h);
    glGetTexLevelParameteriv(m_target, level, GL_TEXTURE_DEPTH, &d);
    
    // recupere les pixels
    // glPixelStore(GL_PACK_ALIGNEMENT ...) ?
    HDRImageArray *image= new HDRImageArray(d);
    float *data= new float [4 * w * h * d];
    glGetTexImage(m_target, level, GL_RGBA, GL_FLOAT, data);
    
    // re-organise les pixels
    for(int i= 0; i < d; i++)
        image->push_back( new HDRImage(w, h, data + (4 * w * h) * i) );
    
    delete [] data;
    return image;
}

GLTexture2D::GLTexture2D( const int unit, const int w, const int h, const TextureFormat& format )
    :
    GLTexture(GL_TEXTURE_2D)
{
    assert(glGetError() == GL_NO_ERROR);
    
    m_width= w;
    m_height= h;
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    if(format.samples == 0)
    {
        glBindTexture(m_target, m_name);
        // initialise la texture a 0
        // alloue un buffer assez gros pour tous les formats
        std::vector<unsigned int> zeros(w * h * 4, 0);
        
        glTexImage2D(m_target, 0, 
            m_format.internal, m_width, m_height, 0,
            m_format.data_format, m_format.data_type, &zeros.front());
        
        // construit la pyramide de mipmap
        glGenerateMipmap(m_target);
        assert(glGetError() == GL_NO_ERROR);
    }
    else
    {
        m_target= GL_TEXTURE_2D_MULTISAMPLE;
        glBindTexture(m_target, m_name);
        glTexImage2DMultisample(m_target, format.samples, 
            m_format.internal, w, h, m_format.fixed_samples ? GL_TRUE : GL_FALSE);
        assert(glGetError() == GL_NO_ERROR);
    }
    
    // definir les parametres de filtrages de base
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    assert(glGetError() == GL_NO_ERROR);
}

GLTexture2D::GLTexture2D( const int unit, const HDRImage *image, const TextureFormat& format )
    :
    GLTexture(GL_TEXTURE_2D)
{
    if(image == NULL)
        return; // utiliser une texture debug ?? damier rouge / orange ?
    
    m_width= image->width();
    m_height= image->height();
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    glTexImage2D(m_target, 0, 
        m_format.internal, m_width, m_height, 0,
        m_format.data_format, m_format.data_type, image->data());
    
    // definir les parametres de filtrages de base
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    glGenerateMipmap(m_target);
    assert(glGetError() == GL_NO_ERROR);    
}

GLTexture2D::GLTexture2D( const int unit, const Image *image, const TextureFormat& format )
    :
    GLTexture(GL_TEXTURE_2D)
{
    if(image == NULL)
        return; // utiliser une texture debug ?? damier rouge / orange ?
    
    m_width= image->width();
    m_height= image->height();
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    glTexImage2D(m_target, 0, 
        m_format.internal, m_width, m_height, 0,
        m_format.data_format, m_format.data_type, image->data());
    
    // definir les parametres de filtrages de base
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    
    glGenerateMipmap(m_target);
    assert(glGetError() == GL_NO_ERROR);
}

GLDepthTexture::GLDepthTexture( const int unit, const int w, const int h, const TextureFormat &format )
    :
    GLTexture2D(GL_TEXTURE_2D)
{
    assert(glGetError() == GL_NO_ERROR);
    
    m_width= w;
    m_height= h;
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    if(format.samples == 0)
    {
        glBindTexture(m_target, m_name);
        // initialise la texture a 0
        std::vector<unsigned int> zeros(w * h, 0);
        
        glTexImage2D(m_target, 0, 
            m_format.internal, m_width, m_height, 0,
            m_format.data_format, m_format.data_type, &zeros.front());
        
        // construit la pyramide de mipmap
        glGenerateMipmap(m_target);
        assert(glGetError() == GL_NO_ERROR);
    }
    else
    {
        m_target= GL_TEXTURE_2D_MULTISAMPLE;
        glBindTexture(m_target, m_name);
        glTexImage2DMultisample(m_target, format.samples, 
            m_format.internal, w, h, m_format.fixed_samples ? GL_TRUE : GL_FALSE);
        assert(glGetError() == GL_NO_ERROR);
    }
    
    // definir les parametres de filtrages de base ...
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    //~ glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    // ... ils seront remplaces par ceux d'un sampler, normalement.
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, GL_NONE);    
    
    assert(glGetError() == GL_NO_ERROR);
}

HDRImage *GLDepthTexture::getHDRImage( const int unit ) const
{
    // verifie le type des pixels de la texture
    if(m_format.data_type != GL_FLOAT)
    {
        printf("GLDepthTexture::getImage( ): not a depth32f texture. failed.\n");
        return NULL;
    }
    if(m_format.samples > 0)
    {
        printf("GLDepthTexture::getImage( ): multisample texture. failed.\n");
        return NULL;
    }
    
    // glPixelStore(GL_PACK_ALIGNEMENT ...) ?
    // recupere les donnees sur 1 canal
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    float *data= new float[m_width * m_height];
    glGetTexImage(m_target, 0, GL_DEPTH_COMPONENT, GL_FLOAT, data);
    
    // conversion en niveau de gris
    HDRImage *image= new HDRImage(m_width, m_height);
    for(int y= 0; y < m_height; y++)
        for(int x= 0; x < m_width; x++)
        {
            const float depth= data[y * m_width + x];
            image->setPixel(x,y, HDRPixel(depth));
        }

    delete [] data;
    return image;
}

GLDepthTextureCube::GLDepthTextureCube( const int unit, const int w, const int h, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_CUBE_MAP)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLDepthTextureCube: multisample texture format not implemented.\n");
    
    m_width= w;
    m_height= h;
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    for(int face= 0; face < 6; face++)
        glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + face, 
            0, m_format.internal, m_width, m_height, 0,
            m_format.data_format, m_format.data_type, NULL);
    
    glGenerateMipmap(m_target);    
}

GLTexture2DArray::GLTexture2DArray( const int unit, const int w, const int h, const int count, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_2D_ARRAY)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLTexture2DArray: multisample texture format not implemented.\n");
    
    m_width= w;
    m_height= h;
    m_depth= count;
    m_format= format;

    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, 
        m_format.internal, m_width, m_height, m_depth, 0,
        m_format.data_format, m_format.data_type, NULL);
}

GLTexture2DArray::GLTexture2DArray( const int unit, const HDRImageArray *images, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_2D_ARRAY)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLTexture2DArray: multisample texture format not implemented.\n");
    
    if(images == NULL)
        return;
    
    m_width= images->width();
    m_height= images->height();
    m_depth= images->size();
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, 
        m_format.internal, m_width, m_height, m_depth, 0,
        m_format.data_format, m_format.data_type, NULL);

    for(int i= 0; i < m_depth; i++)
        glTexSubImage3D(GL_TEXTURE_2D_ARRAY, 0, 
            0, 0, i,  
            m_width, m_height, 1, 
            m_format.data_format, m_format.data_type, (*images)[i]->data());
    
    glGenerateMipmap(m_target);
}

GLTexture2DArray::GLTexture2DArray( const int unit, const ImageArray *images, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_2D_ARRAY)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLTexture2DArray: multisample texture format not implemented.\n");
    
    if(images == NULL)
        return;
    
    m_width= images->width();
    m_height= images->height();
    m_depth= images->size();
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    glTexImage3D(GL_TEXTURE_2D_ARRAY, 0, 
        m_format.internal, m_width, m_height, m_depth, 0,
        m_format.data_format, m_format.data_type, NULL);

    for(int i= 0; i < m_depth; i++)
        glTexSubImage3D(GL_TEXTURE_2D_ARRAY, 0, 
            0, 0, i,  
            m_width, m_height, 1, 
            m_format.data_format, m_format.data_type, (*images)[i]->data());
    
    glGenerateMipmap(m_target);
}

GLTextureCube::GLTextureCube( const int unit, const int w, const int h, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_CUBE_MAP)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLTextureCube: multisample texture format not implemented.\n");
    
    m_width= w;
    m_height= h;
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    for(int face= 0; face < 6; face++)
        glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + face, 
            0, m_format.internal, m_width, m_height, 0,
            m_format.data_format, m_format.data_type, NULL);
    
    glGenerateMipmap(m_target);
}

GLTextureCube::GLTextureCube( const int unit, const HDRImageCube *faces, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_CUBE_MAP)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLTextureCube: multisample texture format not implemented.\n");
    if(faces == NULL)
        return;
    
    m_width= faces->width();
    m_height= faces->height();
    m_depth= 1;
    m_format= format;

    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    for(int face= 0; face < 6; face++)
        glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + face, 
            0, m_format.internal, m_width, m_height, 0,
            m_format.data_format, m_format.data_type, (*faces)[face]->data());
    
    glGenerateMipmap(m_target);
}

GLTextureCube::GLTextureCube( const int unit, const ImageCube *faces, const TextureFormat& format )
    :
    GLTexture2D(GL_TEXTURE_CUBE_MAP)
{
    if(m_name == 0)
        return;
    if(format.samples > 0)
        printf("GLTextureCube: multisample texture format not implemented.\n");
    if(faces == NULL)
        return;
    
    m_width= faces->width();
    m_height= faces->height();
    m_depth= 1;
    m_format= format;
    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(m_target, m_name);
    for(int face= 0; face < 6; face++)
        glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + face, 
            0, m_format.internal, m_width, m_height, 0,
            m_format.data_format, m_format.data_type, (*faces)[face]->data());
    
    glGenerateMipmap(m_target);
}

}
