
#include <cassert>
#include <cstring>

#include "Logger.h"
#include "GL/TPShaderProgram.h"
#include "GL/TPAttributes.h"


namespace gk {

bool GLShaderProgram::is_sampler( const GLenum type )
{
    switch(type)
    {
        case  GL_SAMPLER_1D:
        case  GL_SAMPLER_2D:
        case  GL_SAMPLER_3D:
        case  GL_SAMPLER_CUBE:
        case  GL_SAMPLER_1D_SHADOW:
        case  GL_SAMPLER_2D_SHADOW:
        
    #ifdef GL_VERSION_3_0
        case  GL_SAMPLER_1D_ARRAY:
        case  GL_SAMPLER_2D_ARRAY:
        case  GL_SAMPLER_1D_ARRAY_SHADOW:
        case  GL_SAMPLER_2D_ARRAY_SHADOW:
        case  GL_SAMPLER_CUBE_SHADOW:
        case  GL_UNSIGNED_INT_VEC2:
        case  GL_UNSIGNED_INT_VEC3:
        case  GL_UNSIGNED_INT_VEC4:
        case  GL_INT_SAMPLER_1D:
        case  GL_INT_SAMPLER_2D:
        case  GL_INT_SAMPLER_3D:
        case  GL_INT_SAMPLER_CUBE:
        case  GL_INT_SAMPLER_1D_ARRAY:
        case  GL_INT_SAMPLER_2D_ARRAY:
        case  GL_UNSIGNED_INT_SAMPLER_1D:
        case  GL_UNSIGNED_INT_SAMPLER_2D:
        case  GL_UNSIGNED_INT_SAMPLER_3D:
        case  GL_UNSIGNED_INT_SAMPLER_CUBE:
        case  GL_UNSIGNED_INT_SAMPLER_1D_ARRAY:
        case  GL_UNSIGNED_INT_SAMPLER_2D_ARRAY:
    #endif
    
    #ifdef GL_VERSION_3_1
        case  GL_SAMPLER_2D_RECT:
        case  GL_SAMPLER_2D_RECT_SHADOW:
        case GL_SAMPLER_BUFFER:
        case GL_INT_SAMPLER_2D_RECT:
        case GL_INT_SAMPLER_BUFFER:
        case GL_UNSIGNED_INT_SAMPLER_2D_RECT:
        case GL_UNSIGNED_INT_SAMPLER_BUFFER:
    #endif
    
    #ifdef GL_VERSION_4_0
    case GL_SAMPLER_CUBE_MAP_ARRAY:
    case GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW:
    case GL_INT_SAMPLER_CUBE_MAP_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY:
    #endif
    
    #ifdef GL_ARB_texture_multisample
    case GL_SAMPLER_2D_MULTISAMPLE:
    case GL_INT_SAMPLER_2D_MULTISAMPLE:
    case GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE:
    case GL_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY:
    #endif
            return true;

        default:
            return false;
    }
}


bool GLShaderProgram::is_integer( const GLenum type )
{
    switch(type)
    {
        case GL_INT:
        case GL_INT_VEC2:
        case GL_INT_VEC3:
        case GL_INT_VEC4:
        case GL_UNSIGNED_INT:
        case GL_UNSIGNED_INT_VEC2:
        case GL_UNSIGNED_INT_VEC3: 
        case GL_UNSIGNED_INT_VEC4:
            return true;
        
        default:
            return false;
    }
}


GLShaderProgram::GLShaderProgram( )
    :
    GLResource(),
    m_shaders(),
    m_feedbacks(),
    m_is_linked(false),
    m_is_validated(false)
{
    for(int i= 0; i < GLShaderObject::SHADERTYPE_LAST; i++)
        m_shaders[i]= NULL;
    
    m_name= glCreateProgram();
}

//! ajoute un shader object au shader program.
int GLShaderProgram::attachShader( GLShaderObject *shader )
{
    if(shader == NULL)
    {
        WARNING("GLShaderProgram( ): warning, no shader object.\n");
        return 0;
    }

    GLenum type= shader->type();
    if(type >= GLShaderObject::SHADERTYPE_LAST || m_shaders[type] != NULL)
    {
        ERROR("GLShaderProgram( ): error duplicate shader object (type 0x%x).\n", type);
        return -1;
    }
    
    m_shaders[type]= shader;
    return 0;
}


int GLShaderProgram::createGLResource( )
{
    if(m_name == 0)
        return -1;

    return make();
}

int GLShaderProgram::releaseGLResource( )
{
    if(m_name != 0)
        glDeleteProgram(m_name);

    m_name= 0;
    m_is_linked= false;
    m_is_validated= false;
    return 0;
}

//! construit le shader program.
int GLShaderProgram::make( )
{
    if(m_name == 0)
        return -1;

    while(glGetError() != GL_NO_ERROR)
        {;}
    assert(glGetError() == GL_NO_ERROR);

    for(int i= 0; i < GLShaderObject::SHADERTYPE_LAST; i++)
    {
        if(m_shaders[i] == NULL)
            continue;
        
        const unsigned int type= m_shaders[i]->type();
        if(m_shaders[i]->createGLResource() < 0)
        {
            ERROR("GLShaderProgram( ): error compiling %s shader object.\n", GLShaderObject::ShaderTypeString[type]);
            return -1;
        }

    #ifdef VERBOSE_DEBUG
        {
            MESSAGE("GLShaderProgram( ): error compiling %s shader object...\n", GLShaderObject::ShaderTypeString[type]);
        
            GLint length;
            glGetShaderiv(m_shaders[i]->name(), GL_SHADER_SOURCE_LENGTH, &length);
            GLchar *source= new GLchar[length];
            
            glGetShaderSource(m_shaders[i]->name(), length, NULL, source);
            
            MESSAGE("%s\n", source);
            delete [] source;
        }
    #endif
        
        glAttachShader(m_name, m_shaders[i]->name());
        assert(glGetError() == GL_NO_ERROR);
    }

    return link();
}

GLShaderObject *GLShaderProgram::shader( const unsigned int type )
{
    if(type >= GLShaderObject::SHADERTYPE_LAST)
        return NULL;
    return m_shaders[type];
}

int GLShaderProgram::clear( )
{
    // detache les shaders
    for(int i= 0; i < GLShaderObject::SHADERTYPE_LAST; i++)
        if(m_shaders[i] != NULL)
        {
            glDetachShader(m_name, m_shaders[i]->name());
            m_shaders[i]= NULL;
        }
    
    // reinitialise l'etat 
    m_uniforms.clear();
    m_samplers.clear();
    m_attributes.clear();
    m_feedbacks.clear();
    
    m_is_linked= false;
    m_is_validated= false;
    
    // recupere toutes les erreurs opengl creees par les erreurs de compilation / link, etc.
    while(glGetError() != GL_NO_ERROR)
        {;}
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

ProgramUniform GLShaderProgram::uniform( const char *name ) const
{
    if(m_name == 0)
        return ProgramUniform();

    const int n= (int) m_uniforms.size();
    for(int i= 0; i < n; i++)
        if(m_uniforms[i].name == name)
            return ProgramUniform( this, m_uniforms[i].location, m_uniforms[i].index, 
                m_uniforms[i].size, m_uniforms[i].type, m_uniforms[i].is_integer );

    WARNING("ProgramUniform('%s'): not found.\n", name);
    return ProgramUniform();
}

ProgramSampler GLShaderProgram::sampler( const char *name ) const
{
    if(m_name == 0)
        return ProgramSampler();

    const int n= (int) m_samplers.size();
    for(int i= 0; i < n; i++)
        if(m_samplers[i].name == name)
            return ProgramSampler( this, m_samplers[i].location, m_samplers[i].index,
                m_samplers[i].size, m_samplers[i].type );
    
    WARNING("ProgramSampler('%s'): not found.\n", name);
    return ProgramSampler();
}

ProgramInterface GLShaderProgram::interface( const char *name ) const
{
    if(m_name == 0)
        return ProgramInterface();
    
    GLint length= 0;
    GLuint index= glGetUniformBlockIndex(m_name, name);
    if(index != GL_INVALID_INDEX)
    {
        glGetActiveUniformBlockiv(m_name, index, GL_UNIFORM_BLOCK_DATA_SIZE, &length);
        return ProgramInterface( this, index, length );
    }
    
    WARNING("ProgramInterface('%s'): not found.\n", name);
    return ProgramInterface();
}

ProgramAttribute GLShaderProgram::attribute( const char *name ) const
{
    if(m_name == 0)
        return ProgramAttribute();
    
    const int n= (int) m_attributes.size();
    for(int i= 0; i < n ; i++)
        if(m_attributes[i].name == name)
            return ProgramAttribute( this, m_attributes[i].location, m_attributes[i].index, 
                m_attributes[i].size, m_attributes[i].type, m_attributes[i].is_integer );
    
    WARNING("ProgramAttribute('%s'): not found.\n", name);
    return ProgramAttribute();
}

ProgramDrawbuffer GLShaderProgram::drawbuffer( const char *name ) const
{
    if(m_name == 0)
        return ProgramDrawbuffer();
    
    return ProgramDrawbuffer( this, glGetFragDataLocation(m_name, name), glGetFragDataIndex(m_name, name), 0, 0 );
    //! \todo size et type d'un varying ...
}

ProgramFeedback GLShaderProgram::feedback( const char *name ) const
{
    if(m_name == 0)
        return ProgramFeedback();

    const int n= (int) m_feedbacks.size();
    for(int i= 0; i < n; i++)
        if(m_feedbacks[i].name == name)
            return ProgramFeedback( this, -1, m_feedbacks[i].index, 
                m_feedbacks[i].size, m_feedbacks[i].type, m_feedbacks[i].is_integer );
    
    WARNING("ProgramFeedback('%s'): not found.\n", name);
    return ProgramFeedback();
}


//! (re-)linke le shader program.
int GLShaderProgram::link( )
{
    if(m_name == 0)
        return -1;

    m_uniforms.clear();
    m_samplers.clear();
    m_attributes.clear();
    m_feedbacks.clear();
    
    while(glGetError() != GL_NO_ERROR)
        {;}
    assert(glGetError() == GL_NO_ERROR);
    
    GLint code;
    glLinkProgram(m_name);
    glGetProgramiv(m_name, GL_LINK_STATUS, &code);
    if(code == GL_FALSE)
    {
        GLint length= 0;
        glGetShaderiv(m_name, GL_INFO_LOG_LENGTH, &length);
        if(length == 0)
        {
            ERROR("GLShaderProgram( ): error linking shader program (no info log).\nfailed.\n");
            clear();
            return -1;
        }
        
        // afficher les erreurs de link
        GLchar *log= new GLchar[length];
        glGetProgramInfoLog(m_name, (GLsizei) length, NULL, log);
        
        ERROR("GLShaderProgram( ): error linking shader program:\n%s\nfailed.\n", log);
        delete [] log;
        clear();
        return -1;
    }

    // recupere les samplers et leur affecte une unite de texture.
    GLint uniform_count= 0;
    glGetProgramiv(m_name, GL_ACTIVE_UNIFORMS, &uniform_count);
    if(uniform_count > 0)
    {
        GLint max_length= 0;
        glGetProgramiv(m_name, GL_ACTIVE_UNIFORM_MAX_LENGTH, &max_length);
        char *name= new char [max_length];
        
        int unit_id= 0;
        for(int i= 0; i < uniform_count; i++)
        {
            GLint size;
            GLenum type;
            name[0]= 0;
            glGetActiveUniform(m_name, i, max_length, NULL, &size, &type, name);
            GLint location= glGetUniformLocation(m_name, name);
            
            if(is_sampler(type) == false)
            {
                MESSAGE("  uniform '%s' location %d, index %d, size %d, type 0x%x\n", 
                    name, location, i, size, type);
                
                m_uniforms.push_back( parameter(name, location, m_uniforms.size(), size, type, is_integer(type)) );
            }
            else
            {
                MESSAGE("  sampler '%s' location %d, index %d, size %d, type 0x%x\n", 
                    name, location, unit_id, size, type);
                
                m_samplers.push_back( parameter(name, location, unit_id, size, type, false) );
                unit_id++;
            }
        }
        
        delete [] name;
    }
    
    m_sampler_count= m_samplers.size();
    m_uniform_count= m_uniforms.size();
    
    m_attribute_count= 0;
    glGetProgramiv(m_name, GL_ACTIVE_ATTRIBUTES, &m_attribute_count);
    if(m_attribute_count > 0)
    {
        GLint max_length= 0;
        glGetProgramiv(m_name, GL_ACTIVE_ATTRIBUTE_MAX_LENGTH, &max_length);
        char *name= new char [max_length];
        
        for(int i= 0; i < m_attribute_count; i++)
        {
            GLint size;
            GLenum type;
            name[0]= 0;
            glGetActiveAttrib(m_name, i, max_length, NULL, &size, &type, name);
            GLint location= glGetAttribLocation(m_name, name);
            
            MESSAGE("  attribute '%s' location %d, index %d, size %d, type 0x%x\n", 
                name, location, i, size, type);
            
            m_attributes.push_back( parameter(name, location, i, size, type, is_integer(type)) );
        }
        
        delete [] name;
    }

    // recupere les feedbacks
    GLint feedback_count= 0;
    glGetProgramiv(m_name, GL_TRANSFORM_FEEDBACK_VARYINGS, &feedback_count);
    if(feedback_count > 0)
    {
        GLint max_length= 0;
        glGetProgramiv(m_name, GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH, &max_length);
        char *name= new char [max_length];
        
        GLint buffer_mode= 0;
        glGetProgramiv(m_name, GL_TRANSFORM_FEEDBACK_BUFFER_MODE, &buffer_mode);
        MESSAGE("  feedback mode '%s'\n", 
            (buffer_mode == GL_SEPARATE_ATTRIBS) ? "separate" :
            (buffer_mode == GL_INTERLEAVED_ATTRIBS) ? "interleaved" : "??");
        
        int buffer_id= 0;
        for(int i= 0; i < feedback_count; i++)
        {
            GLint size;
            GLenum type;
            name[0]= 0;            
            glGetTransformFeedbackVarying(m_name, i, max_length, NULL, &size, &type, name);
            MESSAGE("  feedback '%s' index %d, size %d, type 0x%x, buffer %d\n", name, i, size, type, buffer_id);
            
            m_feedbacks.push_back( parameter(name, -1, buffer_id, size, type, is_integer(type)) );
            
            // determiner dans quel buffer le feedback sera stocke
            if(buffer_mode == GL_SEPARATE_ATTRIBS)
                buffer_id++;
            else if(buffer_mode == GL_INTERLEAVED_ATTRIBS && type == GL_NONE 
            && strcmp(name, "gl_NextBuffer") == 0)      // extension arb_feedback_transform3
                buffer_id++;
        }       //! \todo calculer le buffer layout des feedbacks
        
        delete [] name;
    }
    
    m_feedback_count= m_feedbacks.size();
    
    MESSAGE("done.\n");
    
    m_is_linked= true;
    return 0;
}

//! valide la configuration du shader program.
int GLShaderProgram::validate( )
{
    if(m_name == 0)
        return -1;
    if(m_is_linked == false)
        return -1;

    GLint code;
    glValidateProgram(m_name);
    glGetProgramiv(m_name, GL_VALIDATE_STATUS, &code);
    if(code == GL_FALSE)
    {
        GLint length= 0;
        glGetShaderiv(m_name, GL_INFO_LOG_LENGTH, &length);
        if(length == 0)
        {
            ERROR("GLShaderProgram( ): error validating shader (no info log).\n");
            return -1;
        }

        GLchar *log= new GLchar[length];
        glGetProgramInfoLog(m_name, (GLsizei) length, NULL, log);

        ERROR("GLShaderProgram( ): error validating shader:\n%s\n", log);
        delete [] log;
        return -1;
    }

    m_is_validated= true;
    return 0;
}

}
