
#ifndef _TP_PROGRAM_NAME_H
#define _TP_PROGRAM_NAME_H

#include "GL/GLPlatform.h"

namespace gk {
    
class GLShaderProgram;

//! utilisation interne. identifiant generique.
class ProgramName
{
    ProgramName& operator= ( const ProgramName& );
    
    const GLShaderProgram *m_program;
    GLint m_location;
    GLint m_index;
    bool m_is_integer;
    
public:
    int size;
    GLenum type;

    ProgramName( )
        :
        m_program(NULL),
        m_location(-1),
        m_index(-1),
        m_is_integer(false),
        size(0),
        type(0)
    {}

    ProgramName( const ProgramName& name ) 
        :
        m_program(name.m_program),
        m_location(name.m_location),
        m_index(name.m_index),
        m_is_integer(name.m_is_integer),
        size(name.size),
        type(name.type)
    {}
        
    explicit ProgramName( const GLShaderProgram *program, const GLint location, const GLint index, 
        const int _size, const GLenum _type, const bool is_integer= false ) 
        :
        m_program(program),
        m_location(location),
        m_index(index),
        m_is_integer(is_integer),
        size(_size),
        type(_type)
    {}
    
    ~ProgramName( ) {}
    
    const GLShaderProgram *program( ) const
    {
        return m_program;
    }
    
    const GLShaderProgram *program( )
    {
        return m_program;
    }
    
    GLint location( ) const
    {
        return m_location;
    }
    
    GLint index( ) const
    {
        return m_index;
    }
    
    bool isValid( ) const
    {
        return (m_program != NULL && m_location >= 0);
    }
    
    operator GLint ( ) const
    {
        return m_location;
    }
    
    bool isIntegerType( ) const
    {
        return m_is_integer;
    }
};


//! identifiant d'un uniform.
typedef ProgramName ProgramUniform;
//! identifiant d'un attribut.
typedef ProgramName ProgramAttribute;
//! identifiant d'un draw buffer / fragment varying.
typedef ProgramName ProgramDrawbuffer;
//! identifiant d'un transform feedback varying.
typedef ProgramName ProgramFeedback;
//! identifiant d'un sampler.
typedef ProgramName ProgramSampler;


//! modifier la valeur d'un sampler du shader program actif : index de l'unite de texture.
int setSamplerUniform( const ProgramSampler& sampler, const int unit );

class GLTexture;
class GLSampler;
//! activer une texture et l'associer a un sampler du shader program actif.
int setTexture( const ProgramSampler& sampler_uniform, GLTexture *texture, GLSampler *texture_sampler= NULL );
//! desactiver une texture activee par setTexture().
int resetTexture(const ProgramSampler& sampler_uniform, GLTexture *texture );

//! modifier la valeur d'un uniform du shader program actif, unsigned int.
int setUniform( const ProgramUniform& uniform, const unsigned int x );
//! modifier la valeur d'un uniform du shader program actif, uvec2.
int setUniform( const ProgramUniform& uniform, const unsigned int x, const unsigned int y );
//! modifier la valeur d'un uniform du shader program actif, uvec3.
int setUniform( const ProgramUniform& uniform, const unsigned int x, const unsigned int y, const unsigned int z );
//! modifier la valeur d'un uniform du shader program actif, uvec4.
int setUniform( const ProgramUniform& uniform, const unsigned int x, const unsigned int y, const unsigned int z, const unsigned int w );

//! modifier la valeur d'un uniform du shader program actif, int.
int setUniform( const ProgramUniform& uniform, const int x );
//! modifier la valeur d'un uniform du shader program actif, ivec2.
int setUniform( const ProgramUniform& uniform, const int x, const int y );
//! modifier la valeur d'un uniform du shader program actif, ivec3.
int setUniform( const ProgramUniform& uniform, const int x, const int y, const int z );
//! modifier la valeur d'un uniform du shader program actif, ivec4.
int setUniform( const ProgramUniform& uniform, const int x, const int y, const int z, const int w );

//! modifier la valeur d'un uniform du shader program actif, float.
int setUniform( const ProgramUniform& uniform, const float x );
//! modifier la valeur d'un uniform du shader program actif, vec2.
int setUniform( const ProgramUniform& uniform, const float x, const float y );
//! modifier la valeur d'un uniform du shader program actif, vec3.
int setUniform( const ProgramUniform& uniform, const float x, const float y, const float z );
//! modifier la valeur d'un uniform du shader program actif, vec4.
int setUniform( const ProgramUniform& uniform, const float x, const float y, const float z, const float w );

//! modifier la valeur d'un uniform du shader program actif, mat4.
int setUniform( const ProgramUniform& uniform, const float *m, const GLboolean transpose );

//! identifiant d'un uniform block / interface.
class ProgramInterface
{
    const GLShaderProgram *m_program;
    int m_index;
    
public:
    unsigned int length;

    ProgramInterface( )
        :
        m_program(NULL),
        m_index(-1),
        length(0)
    {}

    ProgramInterface( const GLShaderProgram *program, const GLint index, const unsigned int _length )
        :
        m_program(program),
        m_index(index),
        length(_length)
    {}

    const GLShaderProgram *program( ) const
    {
        return m_program;
    }
    
    const GLShaderProgram *program( )
    {
        return m_program;
    }
    
    GLint index( ) const
    {
        return m_index;
    }
    
    bool isValid( ) const
    {
        return (m_program != NULL && m_index >= 0);
    }
};

int setUniformInterface( const ProgramInterface& interface, const GLuint interface_buffer );

}

#endif
