
#include "Logger.h"
#include "GL/TPTexture.h"
#include "GL/TPSampler.h"
#include "SamplerManager.h"
#include "GL/TPShaderProgram.h"
#include "GL/TPProgramName.h"


namespace gk {  

int setSamplerUniform( const ProgramSampler& sampler, const int unit )
{
    if(!sampler.isValid())
        return -1;

#ifndef NDEBUG
    if(sampler.size != 1)
    {
        ERROR("setSamplerUniform('%s'): size != 1\n", sampler.program()->samplerName(sampler));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != sampler.program()->name())
        {
            ERROR("setSamplerUniform('%s'): program is not in use.\n", sampler.program()->samplerName(sampler));
            return -1;
        }
    }
#endif
    glUniform1i(sampler.location(), unit);
#else
    glProgramUniform1i(sampler.program()->name(), sampler.location(), unit);
#endif
    return 0;
}


static GLSampler *default_sampler= NULL;

int setTexture( const ProgramSampler& uniform, GLTexture *texture, GLSampler *sampler )
{
    if(uniform.isValid() == false)
        return -1;
    if(texture == NULL)
    {
        ERROR("setTexture('%s'): null texture\n", uniform.program()->samplerName(uniform));
        return -1;
    }
    
    if(sampler == NULL)
    {
        if(default_sampler == NULL)
            default_sampler= gk::createLinearSampler();
        sampler= default_sampler;
    }
    
    int unit= uniform.index();
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(texture->target(), texture->name());
    glBindSampler(unit, (sampler != NULL) ? sampler->name() : 0);
    
    return setSamplerUniform(uniform, uniform.index());
}

int resetTexture( const ProgramSampler& uniform, GLTexture *texture )
{
    if(uniform.isValid() == false)
        return -1;
    if(texture == NULL)
    {
        ERROR("setTexture('%s'): null texture\n", uniform.program()->samplerName(uniform));
        return -1;
    }
    
    int unit= uniform.index();    
    glActiveTexture(GL_TEXTURE0 + unit);
    glBindTexture(texture->target(), 0);
    return 0;
}

int setUniform( const ProgramUniform& uniform, const unsigned int x )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_UNSIGNED_INT)
    {
        ERROR("setUniform('%s'): type != uint\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform1ui(uniform.location(), x);
#else
    glProgramUniform1ui(uniform.program()->name(), uniform.location(), x);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const unsigned int x, const unsigned int y )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_UNSIGNED_INT_VEC2)
    {
        ERROR("setUniform('%s'): type != uvec2\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform2ui(uniform.location(), x, y);
#else
    glProgramUniform2ui(uniform.program()->name(), uniform.location(), x, y);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const unsigned int x, const unsigned int y, const unsigned int z )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_UNSIGNED_INT_VEC3)
    {
        ERROR("setUniform('%s'): type != uvec3\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform3ui(uniform.location(), x, y, z);
#else
    glProgramUniform3ui(uniform.program()->name(), uniform.location(), x, y, z);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const unsigned int x, const unsigned int y, const unsigned int z, const unsigned int w )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_UNSIGNED_INT_VEC4)
    {
        ERROR("setUniform('%s'): type != uvec4\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform4ui(uniform.location(), x, y, z, w);
#else
    glProgramUniform4ui(uniform.program()->name(), uniform.location(), x, y, z, w);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const int x )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_INT)
    {
        ERROR("setUniform('%s'): type != int\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform1i(uniform.location(), x);
#else
    glProgramUniform1i(uniform.program()->name(), uniform.location(), x);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const int x, const int y )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_INT_VEC2)
    {
        ERROR("setUniform('%s'): type != ivec2\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform2i(uniform.location(), x, y);
#else
    glProgramUniform2i(uniform.program()->name(), uniform.location(), x, y);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const int x, const int y, const int z )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_INT_VEC3)
    {
        ERROR("setUniform('%s'): type != ivec3\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform3i(uniform.location(), x, y, z);
#else
    glProgramUniform3i(uniform.program()->name(), uniform.location(), x, y, z);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const int x, const int y, const int z, const int w )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_INT_VEC4)
    {
        ERROR("setUniform('%s'): type != ivec4\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform4i(uniform.location(), x, y, z, w);
#else
    glProgramUniform4i(uniform.program()->name(), uniform.location(), x, y, z, w);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const float x )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_FLOAT)
    {
        ERROR("setUniform('%s'): type != float\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform1f(uniform.location(), x);
#else
    glProgramUniform1f(uniform.program()->name(), uniform.location(), x);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const float x, const float y )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_FLOAT_VEC2)
    {
        ERROR("setUniform('%s'): type != vec2\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform2f(uniform.location(), x, y);
#else
    glProgramUniform2f(uniform.program()->name(), uniform.location(), x, y);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const float x, const float y, const float z )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_FLOAT_VEC3)
    {
        ERROR("setUniform('%s'): type != vec3\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif
    
#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform3f(uniform.location(), x, y, z);
#else
    glProgramUniform3f(uniform.program()->name(), uniform.location(), x, y, z);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const float x, const float y, const float z, const float w )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_FLOAT_VEC4)
    {
        ERROR("setUniform('%s'): type != vec4\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniform4f(uniform.location(), x, y, z, w);
#else
    glProgramUniform4f(uniform.program()->name(), uniform.location(), x, y, z, w);
#endif
    return 0;
}

int setUniform( const ProgramUniform& uniform, const float *m, const GLboolean transpose )
{
    if(!uniform.isValid())
        return -1;

#ifndef NDEBUG
    if(uniform.size != 1)
    {
        ERROR("setUniform('%s'): size != 1\n", uniform.program()->uniformName(uniform));
        return -1;
    }
    if(uniform.type != GL_FLOAT_MAT4)
    {
        ERROR("setUniform('%s'): type != mat4\n", uniform.program()->uniformName(uniform));
        return -1;
    }
#endif

#ifndef GK_OPENGL4
#ifndef NDEBUG
    {
        GLint program= 0;
        glGetIntegerv(GL_CURRENT_PROGRAM, &program);
        if(program == 0 || (GLuint) program != uniform.program()->name())
        {
            ERROR("setUniform('%s'): program is not in use.\n", uniform.program()->uniformName(uniform));
            return -1;
        }
    }
#endif
    glUniformMatrix4fv(uniform.location(), 1, transpose, m);
#else
    glProgramUniformMatrix4fv(uniform.program()->name(), uniform.location(), 1, transpose, m);
#endif
    return 0;
}

int setUniformInterface( const ProgramInterface& interface, const GLuint interface_buffer )
{
    if(!interface.isValid())
        return -1;
    
    glUniformBlockBinding(interface.program()->name(), interface.index(), interface_buffer);
    return 0;
}

}
