
#include <cstdio>

#include "GLManager.h"
#include "GL/TPFramebuffer.h"


namespace gk {

GLRenderbuffer2D::GLRenderbuffer2D( const int w, const int h, const RenderbufferFormat& format )
    :
    GLRenderbuffer(GL_RENDERBUFFER)
{
    assert(glGetError() == GL_NO_ERROR);
    
    m_width= w;
    m_height= h;
    m_format= format;
    
    glBindRenderbuffer(m_target, m_name);
    if(format.samples == 0)
        glRenderbufferStorage(m_target, format.internal, w, h);
        
    else if(format.color_samples == 0)
        glRenderbufferStorageMultisample(m_target, format.samples, format.internal, w, h);

    else if(GLEW_NV_framebuffer_multisample_coverage)
    {
        glRenderbufferStorageMultisampleCoverageNV(m_target, format.samples, format.color_samples, format.internal, w, h);
        
        #ifdef VERBOSE_DEBUG
        {
            int count;
            glGetIntegerv(GL_MAX_MULTISAMPLE_COVERAGE_MODES_NV, &count);
            
            printf("renderbuffer nv supported samples:\n");
            int *samples= new int[count*2];
            glGetIntegerv(GL_MULTISAMPLE_COVERAGE_MODES_NV, samples);
            for(int i= 0; i < count; i++)
                printf("  %d coverage +%d color\n", samples[2*i], samples[2*i+1]);
            printf("\n");
            delete [] samples;
        }
        #endif
        
        {
            int coverage= 0;
            glGetRenderbufferParameteriv(m_target, GL_RENDERBUFFER_COVERAGE_SAMPLES_NV, &coverage);
            int color= 0;
            glGetRenderbufferParameteriv(m_target, GL_RENDERBUFFER_COLOR_SAMPLES_NV, &color);
            
            printf("renderbuffer_nv coverage %d, color %d\n", coverage, color);
        }
    }
    
    else
        glRenderbufferStorageMultisample(m_target, format.samples, format.internal, w, h);
    
    {
        int samples= 0;
        glGetRenderbufferParameteriv(m_target, GL_RENDERBUFFER_SAMPLES, &samples);
        
        printf("renderbuffer samples %d\n", samples);
    }
    
    assert(glGetError() == GL_NO_ERROR);
}

GLRenderbuffer2D::~GLRenderbuffer2D( ) {}


GLRendertarget::GLRendertarget( const GLenum target )
    :
    GLResource(),
    m_textures(FRAMEBUFFER_LAST, NULL),
    m_renderbuffers(FRAMEBUFFER_LAST, NULL),
    m_draw_buffers(),
    m_width(0),
    m_height(0),
    m_color_mask(0),
    m_depth_mask(0)
{
    glGenFramebuffers(1, &m_name);
    glBindFramebuffer(target, m_name);
    assert(glGetError() == GL_NO_ERROR);
}

GLRendertarget::GLRendertarget( const GLenum target, const int w, const int h, const unsigned int buffer_bits,
    const TextureFormat& color_format, const TextureFormat& depth_format )
    :
    GLResource(),
    m_textures(FRAMEBUFFER_LAST, NULL),
    m_renderbuffers(FRAMEBUFFER_LAST, NULL),
    m_draw_buffers(),
    m_width(w),
    m_height(h),
    m_color_mask(0),
    m_depth_mask(0)
{
    glGenFramebuffers(1, &m_name);
    glBindFramebuffer(target, m_name);
    assert(glGetError() == GL_NO_ERROR);
    
    // parcourir tous les bits : enumerer les textures couleurs a creer / initialiser.
    unsigned int count= 0;
    for(unsigned int bit= 0; bit < FRAMEBUFFER_LAST_COLOR; bit++)
    {
        if((buffer_bits & (1<<bit)) == 0)
            continue;
        
        GLTexture2D *color= new GLTexture2D(gk::UNIT0, w, h, color_format);   // creation arbitraire sur texture unit 0
        if(color == NULL || color->createGLResource() < 0)
            return;
        
        glFramebufferTexture2D(target, 
            GL_COLOR_ATTACHMENT0 + count, color->target(), color->name(), 0);
        assert(glGetError() == GL_NO_ERROR);
        
        m_draw_buffers.push_back(GL_COLOR_ATTACHMENT0 + count);
        m_textures[bit]= GLManager<GLTexture2D>::manager().insert(color);
        m_color_mask= 1;
        count++;
    }
    
    // cree un zbuffer, si necessaire.
    if(buffer_bits & DEPTH_BIT)
    {
        GLDepthTexture *depth= new GLDepthTexture(gk::UNIT0, w, h, depth_format);     // creation arbitraire sur texture unit 0
        if(depth == NULL || depth->createGLResource() < 0)
            return;
        
        glFramebufferTexture2D(target, 
            GL_DEPTH_ATTACHMENT, depth->target(), depth->name(), 0);
        assert(glGetError() == GL_NO_ERROR);

        m_textures[DEPTH]= GLManager<GLDepthTexture>::manager().insert(depth);
        m_depth_mask= 1;
    }
    
    // verifie la configuration du frmaebuffer.
    if(validate(target) < 0)
    {
        glDeleteFramebuffers(1, &m_name);
        m_name= 0;
    }
    
    //~ // desactive le nouveau framebuffer.
    //~ glBindFramebuffer(target, 0);
}

GLRendertarget::~GLRendertarget(  )
{
    if(m_name != 0)
        glDeleteFramebuffers(1, &m_name);
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLTexture *texture, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;
    
    glFramebufferTexture2D(target, 
        draw_buffer, texture->target(), texture->name(), level);
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLDepthTexture *texture, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;

    glFramebufferTexture2D(target, 
        draw_buffer, texture->target(), texture->name(), level);
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLTexture2DArray *texture, const int layer, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;
    
    glFramebufferTextureLayer(target, 
        draw_buffer, texture->name(), level, layer);
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}

int GLRendertarget::attachTexture( const GLenum target, const unsigned int buffer, GLTextureCube *texture, const GLenum face, const int level )
{
    int draw_buffer= attach_buffer(buffer, texture);
    if(draw_buffer < 0)
        return -1;
    
    glFramebufferTexture2D(target, 
        draw_buffer, face, texture->name(), level);
    assert(glGetError() == GL_NO_ERROR);
    return 0;    
}

int GLRendertarget::attachRenderbuffer( const GLenum target, const unsigned int buffer, GLRenderbuffer *renderbuffer )
{
    if(m_name == 0)
        return -1;
    if(buffer >= FRAMEBUFFER_LAST)
        return -1;
    if(renderbuffer == NULL)
        return -1;

#ifdef VERBOSE  //_DEBUG
    if(buffer < FRAMEBUFFER_LAST_COLOR)
        printf("GLFramebuffer::attachRenderbuffer( ): buffer 0x%x, renderbuffer %d.\n", 
            GL_COLOR_ATTACHMENT0 + buffer, renderbuffer->name());
    else if(buffer == DEPTH)
        printf("GLFramebuffer::attachRenderbuffer( ): buffer 0x%x, renderbuffer %d.\n", 
            GL_DEPTH_ATTACHMENT, renderbuffer->name());
#endif
    
    // verifie les dimensions de la texture
    if(m_width > 0 && renderbuffer->width() != m_width)
        return -1;
    m_width= renderbuffer->width();
    if(m_height > 0 && renderbuffer->height() != m_height)
        return -1;
    m_height= renderbuffer->height();
    
    // attache la texture
    if(m_textures[buffer] != NULL)
        // une texture et un renderbuffer sont attaches au meme draw buffer.
        return -1;
    
    int draw_buffer= -1;
    m_renderbuffers[buffer]= renderbuffer;
    if(buffer == DEPTH)
    {
        m_depth_mask= 1;
        draw_buffer= GL_DEPTH_ATTACHMENT;
    }
    else
    {
        // recompte le nombre de draw buffers couleur
        m_color_mask= 0;
        m_draw_buffers.clear();
        for(unsigned int i= 0; i < FRAMEBUFFER_LAST_COLOR; i++)
        {
            if(m_textures[i]  == NULL
            && m_renderbuffers[i] == NULL)
                continue;

            if(i == buffer)
                draw_buffer= GL_COLOR_ATTACHMENT0 + (unsigned int) m_draw_buffers.size();
            
            m_draw_buffers.push_back(GL_COLOR_ATTACHMENT0 + (unsigned int) m_draw_buffers.size());
            m_color_mask= 1;
        }
    }
    
    glFramebufferRenderbuffer(target, draw_buffer, renderbuffer->target(), renderbuffer->name());
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}


GLTexture *GLRendertarget::texture( const unsigned int buffer )
{
    if(m_name == 0)
        return NULL;
    if(buffer >= FRAMEBUFFER_LAST)
        return NULL;
    
    return m_textures[buffer];
}

GLDepthTexture *GLRendertarget::depthtexture( )
{
    if(m_name == 0)
        return NULL;
    if(m_textures[DEPTH] == NULL)
        return NULL;
    
    if(m_textures[DEPTH]->format().internal != GL_DEPTH_COMPONENT)
        //! \todo inclure les autres formats depth component
        return NULL;
    
    return static_cast<GLDepthTexture *>(m_textures[DEPTH]);
}

GLRenderbuffer *GLRendertarget::renderbuffer( const unsigned int buffer )
{
    if(m_name == 0)
        return NULL;
    if(buffer >= FRAMEBUFFER_LAST)
        return NULL;

    return m_renderbuffers[buffer];
}

int GLRendertarget::validate( const GLenum target )
{
    assert(glGetError() == GL_NO_ERROR);

    //~ glBindFramebuffer(target, m_name);
    GLenum status= (GLenum) glCheckFramebufferStatus(target);
    if(status != GL_FRAMEBUFFER_COMPLETE)
    {
        printf("GLFramebuffer::validate( ) error:\n");
        switch(status)
        {
            case GL_FRAMEBUFFER_UNDEFINED:
                printf("  framebuffer undefined");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT:
                printf("  framebuffer incomplete attachment");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT:
                printf("  framebuffer incomplete missing attachment");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER:
                printf("  framebuffer incomplete draw buffer");
                break;
            case GL_FRAMEBUFFER_UNSUPPORTED:
                printf("  framebuffer unsupported");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS:
                printf("  framebuffer incomplete layer targets");
                break;
            case GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE:
                printf("  framebuffer incomplete multisample");
                break;
            
            default:
                printf("  ??");
                break;
        }
        
        printf("\n");
        return -1;
    }
    
    return 0;
}

}
