
#include "Logger.h"

#include "EffectIO.h"
#include "EffectShaderManager.h"


namespace gk {

EffectShaderManager::EffectShaderManager( )
    :
    m_effect(NULL),
    m_programs_map(),
    m_programs(),
    m_program_names()
{}

int EffectShaderManager::read( const std::string& effect )
{
    m_effect= EffectIO::read(effect);
    return (m_effect != NULL) ? 0 : -1;
}

EffectShaderManager::EffectShaderManager( const std::string& effect )
    :
    m_effect(NULL),
    m_programs_map(),
    m_programs(),
    m_program_names()
{
    m_effect= EffectIO::read(effect);
}

EffectShaderManager::EffectShaderManager( Effect *effect )
    :
    m_effect(effect),
    m_programs_map(),
    m_programs(),
    m_program_names()
{}

EffectShaderManager::~EffectShaderManager( ) {}

int EffectShaderManager::reload( )
{
    if(m_effect == NULL)
        return -1;
    
    int code= EffectIO::reload(m_effect);
    if(code < 1)
        return code;
    
    const char *shader_keys[]= { 
        "vertex", "fragment", 
        "geometry", 
        "control", "evaluation" 
    };

    const GLenum shader_types[]= { 
        GLShaderObject::VERTEX, GLShaderObject::FRAGMENT, 
        GLShaderObject::GEOMETRY, 
        GLShaderObject::CONTROL, GLShaderObject::EVALUATION
    };

    int n= (int) m_programs.size();
    for(int i= 0; i < n; i++)
    {
        const std::string& program_key= m_program_names[i];
    
        MESSAGE("EffectShaderManager::reloadShaderProgram('%s')...\n",
            program_key.c_str());

        TextFile *program_section= m_effect->find(program_key);
        if(program_section == TextFile::notFound())
        {
            // le program n'existe plus...
            MESSAGE("  program key '%s' not found.\n", program_key.c_str());
            // le supprimer du manager ?
            continue;
        }

        // conserve les shaders attaches au program
        GLShaderProgram *program= m_programs[i];
        GLShaderObject *shaders[GLShaderObject::SHADERTYPE_LAST];
        for(unsigned int k= 0; k < GLShaderObject::SHADERTYPE_LAST; k++)
            shaders[k]= program->shader(k);

        // reinitialise le program
        program->clear();
        
        for(unsigned int k= 0; k < GLShaderObject::SHADERTYPE_LAST; k++)
        {
            const TextValue& shader_section= program_section->find(shader_keys[k]);
            if(shader_section == TextValue::notFound())
                continue;
            
            TextFile *source= m_effect->find(shader_section.asName());
            if(source == NULL || source == TextFile::notFound())
            {
                ERROR("  section '%s' not found...\n", shader_keys[k]);
                continue;
            }
            
            MESSAGE("  %s shader '%s'...\n", shader_keys[k], shader_section.asName().c_str());
            
            GLShaderObject *shader= shaders[k];
            if(shader != NULL)
                shader->clear();
            else
                shader= GLManager<GLShaderObject>::manager().insert( 
                    new GLShaderObject(shader_types[k]) );
            if(shader == NULL)
                continue;
            
            shader->pushSource(source);
            program->attachShader(shader);
        }
        
        if(program->createGLResource() < 0)
            ERROR("failed.\n");
    }
    
    return 1;
}
    
GLShaderProgram *EffectShaderManager::createShaderProgram( const std::string& program_key )
{
    if(m_effect == NULL)
        return NULL;
    
    MESSAGE("shader program '%s'...\n",
        program_key.c_str());
    
    TextFile *program_section= m_effect->find(program_key);
    if(program_section == TextFile::notFound())
    {
        ERROR("  program key '%s' not found.\n", program_key.c_str());
        return NULL;
    }
    
    GLShaderProgram *program= new GLShaderProgram();
    if(program == NULL)
        return NULL;
    
    const char *keys[]= { 
        "vertex", "fragment", 
        "geometry", 
        "control", "evaluation" 
    };
    GLenum types[]= { 
        GLShaderObject::VERTEX, GLShaderObject::FRAGMENT, 
        GLShaderObject::GEOMETRY, 
        GLShaderObject::CONTROL, GLShaderObject::EVALUATION
    };
    
    for(int i= 0; i < GLShaderObject::SHADERTYPE_LAST; i++)
    {
        const TextValue& shader_section= program_section->find(keys[i]);
        if(shader_section == TextValue::notFound())
            continue;
        
        TextFile *source= m_effect->find(shader_section.asName());
        if(source == NULL || source == TextFile::notFound())
        {
            ERROR("  section '%s' not found...\n", keys[i]);
            delete program;
            return NULL;
        }
        
        MESSAGE("  %s shader '%s'...\n", keys[i], shader_section.asName().c_str());
        
        GLShaderObject *shader= GLManager<GLShaderObject>::manager().insert( 
            new GLShaderObject(types[i]) );
        if(shader == NULL)
        {
            delete program;
            return NULL;
        }
        
        shader->pushSource(source);
        program->attachShader(shader);
    }
    
    return GLManager<GLShaderProgram>::manager().insert(
        insert( program, program_key ));      // conserve la liste de programmes crees a partir du fichier d'effet.
}

}       // namespace gk

