
#include <cstdio>

#include "Effect.h"


namespace gk {

Effect::Effect( )
    :
    IOResource(),
    m_text(),
    m_last_line(),
    m_last_section(),
    m_line_number(1),
    m_last_char(0)
{}
    

Effect::Effect( const std::string& filename )
    :
    IOResource(),
    m_text(),
    m_last_line(),
    m_last_section(),
    m_line_number(1),
    m_last_char(0)
{
    read(filename);
}

Effect::~Effect( )
{
    for(int i= 0; i < (int) m_text.size(); i++)
        delete m_text[i];
} 

int Effect::read_line( FILE *in )
{
    m_last_line.clear();
    for(;;)
    {
        int c= fgetc(in);
        if(c == EOF)
            return -1;
        
        if((c == '\r')    // windows + mac
        || (c == '\n' && m_last_char != '\r'))        // linux
        {
            m_line_number++;
            m_last_line.push_back('\n');        // fin de ligne normalisee
            m_last_char= c;
            return 0;
        }
        
        if(c != '\n')
        {
            m_last_line.push_back(c);
            m_last_char= c;
        }
    }
}

int Effect::read_section( FILE *in )
{
    char tmp[1024];
    m_last_section.clear();

    int code= 0;
    while(code == 0)
    {
        code= read_line(in);
        
        // detecter la fin de section
        if(sscanf(m_last_line.c_str(), " -- %1023[^ \n] \n ", tmp) == 1)
            return 0;
        m_last_section.append(m_last_line);
    }
    
    return code;
}

int Effect::read( const std::string& filename )
{
#ifdef VERBOSE
    printf("loading effect file '%s'...\n", filename.c_str());
#endif

    FILE *in= fopen(filename.c_str(), "rb");
    if(in == NULL)
    {
    #ifdef VERBOSE
        printf("failed.\n");
    #endif
        return -1;
    }
    
    // lire les sections
    char key[1024];
    int section_begin= 0;
    for(;;)
    {
        section_begin= m_line_number;
        if(read_section(in) < 0)
            break;
        
        if(sscanf(m_last_line.c_str(), " -- %1023[^ \n] \n ", key) != 1)
            break;
        
        if(strcmp(key, "include") == 0)
        {
            if(m_text.empty() == true)
            {
                // erreur, include en dehors d'une section
            #ifdef VERBOSE
                printf("    invalid include.\n");
            #endif
                break;
            }
            
            if(sscanf(m_last_line.c_str(), " -- include %1023[^ \n] \n ", key) != 1)
                break;
            
        #ifdef VERBOSE
            printf("    include section '%s'\n", key);
        #endif
            
            // retrouver le contenu de la section
            TextFile *include= find(key);
            if(include  == NULL)
            {
            #ifdef VERBOSE
                printf("      not found.\n");
            #endif
                break;
            }
            
            // inserer la section
            m_text.back()->include(m_last_section, filename, section_begin);            
            m_text.back()->include(include);
        }
        else
        {
        #ifdef VERBOSE
            printf("  key '%s', line %d\n", key, m_line_number);
        #endif
        
            // inserer les dernieres lignes lues dans la section
            if(m_text.empty() == false)
                m_text.back()->include(m_last_section, filename, section_begin);

            // creer la nouvelle section
            m_text.push_back( new TextFile(key) );
        }
    }

    // inserer le contenu de la derniere section...
    if(m_text.empty() == false)
        m_text.back()->include(m_last_section, filename, section_begin);
    
#ifdef VERBOSE
    printf("done.\n");
#endif
    fclose(in);
    return 0;
}

int Effect::clear( )
{
    for(int i= 0; i < (int) m_text.size(); i++)
        delete m_text[i];
    m_text.clear();

    m_last_line.clear();
    m_last_section.clear();
    m_line_number= 1;
    m_last_char= 0;
    return 0;
}

TextFile *Effect::find( const std::string& key ) const
{
    // debile et sans surpise ...
    const int n= (int) m_text.size();
    for(int i= 0; i < n; i++)
        if(m_text[i]->name() == key)
            return m_text[i];
        
#ifdef VERBOSE
    printf("Effect::find('%s'): not found, using default value.\n", key.c_str());
#endif
    
    return TextFile::notFound();
}


}
